import _ from 'lodash';
import utils from '@strapi/utils';
import { getService } from '../utils/index.mjs';
import { FILE_MODEL_UID } from '../constants.mjs';
import { validateUploadBody } from './validation/content-api/upload.mjs';

const { ValidationError } = utils.errors;
var contentApi = (({ strapi })=>{
    const sanitizeOutput = async (data, ctx)=>{
        const schema = strapi.getModel(FILE_MODEL_UID);
        const { auth } = ctx.state;
        return strapi.contentAPI.sanitize.output(data, schema, {
            auth
        });
    };
    const validateQuery = async (data, ctx)=>{
        const schema = strapi.getModel(FILE_MODEL_UID);
        const { auth } = ctx.state;
        return strapi.contentAPI.validate.query(data, schema, {
            auth
        });
    };
    const sanitizeQuery = async (data, ctx)=>{
        const schema = strapi.getModel(FILE_MODEL_UID);
        const { auth } = ctx.state;
        return strapi.contentAPI.sanitize.query(data, schema, {
            auth
        });
    };
    return {
        async find (ctx) {
            await validateQuery(ctx.query, ctx);
            const sanitizedQuery = await sanitizeQuery(ctx.query, ctx);
            const files = await getService('upload').findMany(sanitizedQuery);
            ctx.body = await sanitizeOutput(files, ctx);
        },
        async findOne (ctx) {
            const { params: { id } } = ctx;
            await validateQuery(ctx.query, ctx);
            const sanitizedQuery = await sanitizeQuery(ctx.query, ctx);
            const file = await getService('upload').findOne(id, sanitizedQuery.populate);
            if (!file) {
                return ctx.notFound('file.notFound');
            }
            ctx.body = await sanitizeOutput(file, ctx);
        },
        async destroy (ctx) {
            const { params: { id } } = ctx;
            const file = await getService('upload').findOne(id);
            if (!file) {
                return ctx.notFound('file.notFound');
            }
            await getService('upload').remove(file);
            ctx.body = await sanitizeOutput(file, ctx);
        },
        async updateFileInfo (ctx) {
            const { query: { id }, request: { body } } = ctx;
            const data = await validateUploadBody(body);
            if (!id || typeof id !== 'string' && typeof id !== 'number') {
                throw new ValidationError('File id is required and must be a single value');
            }
            const result = await getService('upload').updateFileInfo(id, data.fileInfo);
            ctx.body = await sanitizeOutput(result, ctx);
        },
        async replaceFile (ctx) {
            const { query: { id }, request: { body, files: { files } = {} } } = ctx;
            // cannot replace with more than one file
            if (Array.isArray(files)) {
                throw new ValidationError('Cannot replace a file with multiple ones');
            }
            if (!id || typeof id !== 'string' && typeof id !== 'number') {
                throw new ValidationError('File id is required and must be a single value');
            }
            const data = await validateUploadBody(body);
            const replacedFiles = await getService('upload').replace(id, {
                data,
                file: files
            });
            ctx.body = await sanitizeOutput(replacedFiles, ctx);
        },
        async uploadFiles (ctx) {
            const { request: { body, files: { files } = {} } } = ctx;
            const data = await validateUploadBody(body, Array.isArray(files));
            const apiUploadFolderService = getService('api-upload-folder');
            const apiUploadFolder = await apiUploadFolderService.getAPIUploadFolder();
            if (Array.isArray(files)) {
                data.fileInfo = data.fileInfo || [];
                data.fileInfo = files.map((_f, i)=>({
                        ...data.fileInfo[i],
                        folder: apiUploadFolder.id
                    }));
            } else {
                data.fileInfo = {
                    ...data.fileInfo,
                    folder: apiUploadFolder.id
                };
            }
            const uploadedFiles = await getService('upload').upload({
                data,
                files: Array.isArray(files) ? files : [
                    files
                ]
            });
            ctx.body = await sanitizeOutput(uploadedFiles, ctx);
            ctx.status = 201;
        },
        // TODO: split into multiple endpoints
        async upload (ctx) {
            const { query: { id }, request: { files: { files } = {} } } = ctx;
            if (_.isEmpty(files) || !Array.isArray(files) && files.size === 0) {
                if (id) {
                    return this.updateFileInfo(ctx);
                }
                throw new ValidationError('Files are empty');
            }
            await (id ? this.replaceFile : this.uploadFiles)(ctx);
        }
    };
});

export { contentApi as default };
//# sourceMappingURL=content-api.mjs.map
