'use strict';

var _ = require('lodash');
var utils = require('@strapi/utils');
var index = require('../utils/index.js');
var constants = require('../constants.js');
var upload = require('./validation/admin/upload.js');
var findEntityAndCheckPermissions = require('./utils/find-entity-and-check-permissions.js');
var mimeValidation = require('../utils/mime-validation.js');

var adminUpload = {
    async bulkUpdateFileInfo (ctx) {
        const { state: { userAbility, user }, request: { body } } = ctx;
        const { updates } = await upload.validateBulkUpdateBody(body);
        const uploadService = index.getService('upload');
        const results = await utils.async.map(updates, async ({ id, fileInfo })=>{
            const { pm } = await findEntityAndCheckPermissions.findEntityAndCheckPermissions(userAbility, constants.ACTIONS.update, constants.FILE_MODEL_UID, id);
            const updated = await uploadService.updateFileInfo(id, fileInfo, {
                user
            });
            return pm.sanitizeOutput(updated, {
                action: constants.ACTIONS.read
            });
        });
        ctx.body = results;
    },
    async updateFileInfo (ctx) {
        const { state: { userAbility, user }, query: { id }, request: { body } } = ctx;
        if (typeof id !== 'string') {
            throw new utils.errors.ValidationError('File id is required');
        }
        const uploadService = index.getService('upload');
        const { pm } = await findEntityAndCheckPermissions.findEntityAndCheckPermissions(userAbility, constants.ACTIONS.update, constants.FILE_MODEL_UID, id);
        const data = await upload.validateUploadBody(body);
        const file = await uploadService.updateFileInfo(id, data.fileInfo, {
            user
        });
        ctx.body = await pm.sanitizeOutput(file, {
            action: constants.ACTIONS.read
        });
    },
    async replaceFile (ctx) {
        const { state: { userAbility, user }, query: { id }, request: { body, files: { files } = {} } } = ctx;
        if (typeof id !== 'string') {
            throw new utils.errors.ValidationError('File id is required');
        }
        const uploadService = index.getService('upload');
        const { pm } = await findEntityAndCheckPermissions.findEntityAndCheckPermissions(userAbility, constants.ACTIONS.update, constants.FILE_MODEL_UID, id);
        if (Array.isArray(files)) {
            throw new utils.errors.ApplicationError('Cannot replace a file with multiple ones');
        }
        const securityResults = await mimeValidation.enforceUploadSecurity(files, strapi);
        if (securityResults.errors.length > 0) {
            const { error } = securityResults.errors[0];
            switch(error.code){
                case 'MIME_TYPE_NOT_ALLOWED':
                    throw new utils.errors.ValidationError(error.message, error.details);
                default:
                    throw new utils.errors.ApplicationError(error.message, error.details);
            }
        }
        const data = await upload.validateUploadBody(body);
        const replacedFile = await uploadService.replace(id, {
            data,
            file: securityResults.validFiles[0]
        }, {
            user
        });
        // Sign file urls for private providers
        const signedFile = await index.getService('file').signFileUrls(replacedFile);
        ctx.body = await pm.sanitizeOutput(signedFile, {
            action: constants.ACTIONS.read
        });
    },
    async uploadFiles (ctx) {
        const { state: { userAbility, user }, request: { body, files: { files } = {} } } = ctx;
        const uploadService = index.getService('upload');
        const pm = strapi.service('admin::permission').createPermissionsManager({
            ability: userAbility,
            action: constants.ACTIONS.create,
            model: constants.FILE_MODEL_UID
        });
        if (!pm.isAllowed) {
            return ctx.forbidden();
        }
        const securityResults = await mimeValidation.enforceUploadSecurity(files, strapi);
        if (securityResults.validFiles.length === 0) {
            throw new utils.errors.ValidationError(securityResults.errors[0].error.message, securityResults.errors[0].error.details);
        }
        let filteredBody = body;
        if (body?.fileInfo && Array.isArray(body.fileInfo)) {
            const filteredFileInfo = body.fileInfo.filter((fi)=>{
                const info = typeof fi === 'string' ? JSON.parse(fi) : fi;
                return securityResults.validFileNames.includes(info.name);
            });
            if (filteredFileInfo.length === 1) {
                filteredBody = {
                    ...body,
                    fileInfo: filteredFileInfo[0]
                };
            } else {
                filteredBody = {
                    ...body,
                    fileInfo: filteredFileInfo
                };
            }
        }
        const isMultipleFiles = Array.isArray(filteredBody.fileInfo) && filteredBody.fileInfo.length > 1;
        const data = await upload.validateUploadBody(filteredBody, isMultipleFiles);
        let filesArray = securityResults.validFiles;
        if (data.fileInfo && Array.isArray(data.fileInfo) && filesArray.length === data.fileInfo.length) {
            // Reorder filesArray to match data.fileInfo order
            const alignedFilesArray = data.fileInfo.map((info)=>{
                return filesArray.find((file)=>file.originalFilename === info.name);
            }).filter(Boolean);
            filesArray = alignedFilesArray;
        }
        // Upload files first to get thumbnails
        const uploadedFiles = await uploadService.upload({
            data,
            files: filesArray
        }, {
            user
        });
        if (uploadedFiles.some((file)=>file.mime?.startsWith('image/'))) {
            await index.getService('metrics').trackUsage('didUploadImage');
        }
        const aiMetadataService = index.getService('aiMetadata');
        // AFTER upload - use thumbnail versions for AI processing
        if (await aiMetadataService.isEnabled()) {
            try {
                // Use thumbnail URLs instead of original files
                const thumbnailFiles = uploadedFiles.map((file)=>({
                        filepath: file.formats?.thumbnail?.url || file.url,
                        mimetype: file.mime,
                        originalFilename: file.name,
                        size: file.formats?.thumbnail?.size || file.size,
                        provider: file.provider
                    }));
                const metadataResults = await aiMetadataService.processFiles(thumbnailFiles);
                // Update the uploaded files with AI metadata
                await Promise.all(uploadedFiles.map(async (uploadedFile, index)=>{
                    const aiMetadata = metadataResults[index];
                    if (aiMetadata) {
                        await uploadService.updateFileInfo(uploadedFile.id, {
                            alternativeText: aiMetadata.altText,
                            caption: aiMetadata.caption
                        }, {
                            user
                        });
                        uploadedFiles[index].alternativeText = aiMetadata.altText;
                        uploadedFiles[index].caption = aiMetadata.caption;
                    }
                }));
            } catch (error) {
                strapi.log.warn('AI metadata generation failed, proceeding without AI enhancements', {
                    error: error instanceof Error ? error.message : String(error)
                });
            }
        }
        // Sign file urls for private providers
        const signedFiles = await utils.async.map(uploadedFiles, index.getService('file').signFileUrls);
        ctx.body = await pm.sanitizeOutput(signedFiles, {
            action: constants.ACTIONS.read
        });
        ctx.status = 201;
    },
    // TODO: split into multiple endpoints
    async upload (ctx) {
        const { query: { id }, request: { files: { files } = {} } } = ctx;
        if (_.isEmpty(files) || !Array.isArray(files) && files.size === 0) {
            if (id) {
                return this.updateFileInfo(ctx);
            }
            throw new utils.errors.ApplicationError('Files are empty');
        }
        await (id ? this.replaceFile : this.uploadFiles)(ctx);
    }
};

module.exports = adminUpload;
//# sourceMappingURL=admin-upload.js.map
