import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { Page, useNotification, useFetchClient, Layouts } from '@strapi/admin/strapi-admin';
import { useAIAvailability } from '@strapi/admin/strapi-admin/ee';
import { Button, Flex, Box, Grid, Typography, Field, Toggle } from '@strapi/design-system';
import { Check, Sparkle } from '@strapi/icons';
import isEqual from 'lodash/isEqual';
import { useIntl } from 'react-intl';
import { useMutation } from 'react-query';
import { PERMISSIONS } from '../../constants.mjs';
import { useSettings } from '../../hooks/useSettings.mjs';
import 'byte-size';
import 'date-fns';
import { getTrad } from '../../utils/getTrad.mjs';
import 'qs';
import '../../utils/typeFromMime.mjs';
import '../../utils/urlYupSchema.mjs';
import { init } from './init.mjs';
import { reducer, initialState } from './reducer.mjs';

// TODO: find a better naming convention for the file that was an index file before
const SettingsPage = ()=>{
    const { formatMessage } = useIntl();
    const { toggleNotification } = useNotification();
    const { put } = useFetchClient();
    const [{ initialData, modifiedData }, dispatch] = React.useReducer(reducer, initialState, init);
    const { data, isLoading, refetch } = useSettings();
    const isAIAvailable = useAIAvailability();
    React.useEffect(()=>{
        if (data) {
            dispatch({
                type: 'GET_DATA_SUCCEEDED',
                data
            });
        }
    }, [
        data
    ]);
    const isSaveButtonDisabled = isEqual(initialData, modifiedData);
    const { mutateAsync, isLoading: isSubmitting } = useMutation(async (body)=>{
        const { data } = await put('/upload/settings', body);
        return data;
    }, {
        onSuccess () {
            refetch();
            toggleNotification({
                type: 'success',
                message: formatMessage({
                    id: 'notification.form.success.fields'
                })
            });
        },
        onError (err) {
            toggleNotification({
                type: 'danger',
                message: err.message || formatMessage({
                    id: 'notification.error'
                })
            });
        }
    });
    const handleSubmit = async (e)=>{
        e.preventDefault();
        if (isSaveButtonDisabled) {
            return;
        }
        await mutateAsync(modifiedData);
    };
    const handleChange = ({ target: { name, value } })=>{
        dispatch({
            type: 'ON_CHANGE',
            keys: name,
            value
        });
    };
    if (isLoading) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    return /*#__PURE__*/ jsxs(Page.Main, {
        tabIndex: -1,
        children: [
            /*#__PURE__*/ jsx(Page.Title, {
                children: formatMessage({
                    id: getTrad('page.title'),
                    defaultMessage: 'Settings - Media Library'
                })
            }),
            /*#__PURE__*/ jsxs("form", {
                onSubmit: handleSubmit,
                children: [
                    /*#__PURE__*/ jsx(Layouts.Header, {
                        title: formatMessage({
                            id: getTrad('settings.header.label'),
                            defaultMessage: 'Media Library'
                        }),
                        primaryAction: /*#__PURE__*/ jsx(Button, {
                            disabled: isSaveButtonDisabled,
                            loading: isSubmitting,
                            type: "submit",
                            startIcon: /*#__PURE__*/ jsx(Check, {}),
                            size: "S",
                            children: formatMessage({
                                id: 'global.save',
                                defaultMessage: 'Save'
                            })
                        }),
                        subtitle: formatMessage({
                            id: getTrad('settings.sub-header.label'),
                            defaultMessage: 'Configure the settings for the Media Library'
                        })
                    }),
                    /*#__PURE__*/ jsx(Layouts.Content, {
                        children: /*#__PURE__*/ jsx(Layouts.Root, {
                            children: /*#__PURE__*/ jsxs(Flex, {
                                direction: "column",
                                alignItems: "stretch",
                                gap: 4,
                                children: [
                                    isAIAvailable && /*#__PURE__*/ jsx(Box, {
                                        background: "neutral0",
                                        padding: 6,
                                        shadow: "filterShadow",
                                        hasRadius: true,
                                        children: /*#__PURE__*/ jsx(Flex, {
                                            direction: "column",
                                            alignItems: "stretch",
                                            gap: 1,
                                            children: /*#__PURE__*/ jsxs(Grid.Root, {
                                                gap: 6,
                                                children: [
                                                    /*#__PURE__*/ jsxs(Grid.Item, {
                                                        col: 8,
                                                        xs: 12,
                                                        direction: "column",
                                                        alignItems: "stretch",
                                                        children: [
                                                            /*#__PURE__*/ jsxs(Flex, {
                                                                gap: 2,
                                                                children: [
                                                                    /*#__PURE__*/ jsx(Box, {
                                                                        color: "alternative700",
                                                                        children: /*#__PURE__*/ jsx(Sparkle, {})
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Typography, {
                                                                        variant: "delta",
                                                                        tag: "h2",
                                                                        children: formatMessage({
                                                                            id: getTrad('settings.form.aiMetadata.label'),
                                                                            defaultMessage: 'Generate AI captions and alt texts automatically on upload!'
                                                                        })
                                                                    })
                                                                ]
                                                            }),
                                                            /*#__PURE__*/ jsx(Flex, {
                                                                paddingTop: 1,
                                                                children: /*#__PURE__*/ jsx(Typography, {
                                                                    variant: "pi",
                                                                    textColor: "neutral600",
                                                                    children: formatMessage({
                                                                        id: getTrad('settings.form.aiMetadata.description'),
                                                                        defaultMessage: 'Enable this feature to save time, optimize your SEO and increase accessibility by letting our AI generate captions and alternative texts for you.'
                                                                    })
                                                                })
                                                            })
                                                        ]
                                                    }),
                                                    /*#__PURE__*/ jsx(Grid.Item, {
                                                        col: 4,
                                                        xs: 12,
                                                        direction: "column",
                                                        alignItems: "end",
                                                        justifyContent: 'center',
                                                        children: /*#__PURE__*/ jsx(Field.Root, {
                                                            name: "aiMetadata",
                                                            minWidth: "200px",
                                                            children: /*#__PURE__*/ jsx(Toggle, {
                                                                checked: modifiedData?.aiMetadata,
                                                                offLabel: formatMessage({
                                                                    id: 'app.components.ToggleCheckbox.disabled-label',
                                                                    defaultMessage: 'Disabled'
                                                                }),
                                                                onLabel: formatMessage({
                                                                    id: 'app.components.ToggleCheckbox.enabled-label',
                                                                    defaultMessage: 'Enabled'
                                                                }),
                                                                onChange: (e)=>{
                                                                    handleChange({
                                                                        target: {
                                                                            name: 'aiMetadata',
                                                                            value: e.target.checked
                                                                        }
                                                                    });
                                                                }
                                                            })
                                                        })
                                                    })
                                                ]
                                            })
                                        })
                                    }),
                                    /*#__PURE__*/ jsx(Box, {
                                        background: "neutral0",
                                        padding: 6,
                                        shadow: "filterShadow",
                                        hasRadius: true,
                                        children: /*#__PURE__*/ jsxs(Flex, {
                                            direction: "column",
                                            alignItems: "stretch",
                                            gap: 4,
                                            children: [
                                                /*#__PURE__*/ jsx(Flex, {
                                                    children: /*#__PURE__*/ jsx(Typography, {
                                                        variant: "delta",
                                                        tag: "h2",
                                                        children: formatMessage({
                                                            id: getTrad('settings.blockTitle'),
                                                            defaultMessage: 'Asset management'
                                                        })
                                                    })
                                                }),
                                                /*#__PURE__*/ jsxs(Grid.Root, {
                                                    gap: 6,
                                                    children: [
                                                        /*#__PURE__*/ jsx(Grid.Item, {
                                                            col: 6,
                                                            xs: 12,
                                                            direction: "column",
                                                            alignItems: "stretch",
                                                            children: /*#__PURE__*/ jsxs(Field.Root, {
                                                                hint: formatMessage({
                                                                    id: getTrad('settings.form.responsiveDimensions.description'),
                                                                    defaultMessage: 'Enabling this option will generate multiple formats (small, medium and large) of the uploaded asset.'
                                                                }),
                                                                name: "responsiveDimensions",
                                                                children: [
                                                                    /*#__PURE__*/ jsx(Field.Label, {
                                                                        children: formatMessage({
                                                                            id: getTrad('settings.form.responsiveDimensions.label'),
                                                                            defaultMessage: 'Responsive friendly upload'
                                                                        })
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Toggle, {
                                                                        checked: modifiedData?.responsiveDimensions,
                                                                        offLabel: formatMessage({
                                                                            id: 'app.components.ToggleCheckbox.off-label',
                                                                            defaultMessage: 'Off'
                                                                        }),
                                                                        onLabel: formatMessage({
                                                                            id: 'app.components.ToggleCheckbox.on-label',
                                                                            defaultMessage: 'On'
                                                                        }),
                                                                        onChange: (e)=>{
                                                                            handleChange({
                                                                                target: {
                                                                                    name: 'responsiveDimensions',
                                                                                    value: e.target.checked
                                                                                }
                                                                            });
                                                                        }
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Field.Hint, {})
                                                                ]
                                                            })
                                                        }),
                                                        /*#__PURE__*/ jsx(Grid.Item, {
                                                            col: 6,
                                                            xs: 12,
                                                            direction: "column",
                                                            alignItems: "stretch",
                                                            children: /*#__PURE__*/ jsxs(Field.Root, {
                                                                hint: formatMessage({
                                                                    id: getTrad('settings.form.sizeOptimization.description'),
                                                                    defaultMessage: 'Enabling this option will reduce the image size and slightly reduce its quality.'
                                                                }),
                                                                name: "sizeOptimization",
                                                                children: [
                                                                    /*#__PURE__*/ jsx(Field.Label, {
                                                                        children: formatMessage({
                                                                            id: getTrad('settings.form.sizeOptimization.label'),
                                                                            defaultMessage: 'Size optimization'
                                                                        })
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Toggle, {
                                                                        checked: modifiedData?.sizeOptimization,
                                                                        offLabel: formatMessage({
                                                                            id: 'app.components.ToggleCheckbox.off-label',
                                                                            defaultMessage: 'Off'
                                                                        }),
                                                                        onLabel: formatMessage({
                                                                            id: 'app.components.ToggleCheckbox.on-label',
                                                                            defaultMessage: 'On'
                                                                        }),
                                                                        onChange: (e)=>{
                                                                            handleChange({
                                                                                target: {
                                                                                    name: 'sizeOptimization',
                                                                                    value: e.target.checked
                                                                                }
                                                                            });
                                                                        }
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Field.Hint, {})
                                                                ]
                                                            })
                                                        }),
                                                        /*#__PURE__*/ jsx(Grid.Item, {
                                                            col: 6,
                                                            xs: 12,
                                                            direction: "column",
                                                            alignItems: "stretch",
                                                            children: /*#__PURE__*/ jsxs(Field.Root, {
                                                                hint: formatMessage({
                                                                    id: getTrad('settings.form.autoOrientation.description'),
                                                                    defaultMessage: 'Enabling this option will automatically rotate the image according to EXIF orientation tag.'
                                                                }),
                                                                name: "autoOrientation",
                                                                children: [
                                                                    /*#__PURE__*/ jsx(Field.Label, {
                                                                        children: formatMessage({
                                                                            id: getTrad('settings.form.autoOrientation.label'),
                                                                            defaultMessage: 'Auto orientation'
                                                                        })
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Toggle, {
                                                                        checked: modifiedData?.autoOrientation,
                                                                        offLabel: formatMessage({
                                                                            id: 'app.components.ToggleCheckbox.off-label',
                                                                            defaultMessage: 'Off'
                                                                        }),
                                                                        onLabel: formatMessage({
                                                                            id: 'app.components.ToggleCheckbox.on-label',
                                                                            defaultMessage: 'On'
                                                                        }),
                                                                        onChange: (e)=>{
                                                                            handleChange({
                                                                                target: {
                                                                                    name: 'autoOrientation',
                                                                                    value: e.target.checked
                                                                                }
                                                                            });
                                                                        }
                                                                    }),
                                                                    /*#__PURE__*/ jsx(Field.Hint, {})
                                                                ]
                                                            })
                                                        })
                                                    ]
                                                })
                                            ]
                                        })
                                    })
                                ]
                            })
                        })
                    })
                ]
            })
        ]
    });
};
const ProtectedSettingsPage = ()=>/*#__PURE__*/ jsx(Page.Protect, {
        permissions: PERMISSIONS.settings,
        children: /*#__PURE__*/ jsx(SettingsPage, {})
    });

export { ProtectedSettingsPage, SettingsPage };
//# sourceMappingURL=SettingsPage.mjs.map
