import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { useTracking, useQueryParams, Page, Layouts, SearchInput, Pagination } from '@strapi/admin/strapi-admin';
import { Box, Typography, Checkbox, IconButton, Grid, Flex, VisuallyHidden, Divider } from '@strapi/design-system';
import { Cog, List, GridFour, Pencil } from '@strapi/icons';
import { stringify } from 'qs';
import { useIntl } from 'react-intl';
import { useNavigate, useLocation, Link } from 'react-router-dom';
import { styled } from 'styled-components';
import { AIUploadModal } from '../../../ai/components/AIUploadModal.mjs';
import { AssetGridList } from '../../../components/AssetGridList/AssetGridList.mjs';
import { EditAssetDialog } from '../../../components/EditAssetDialog/EditAssetContent.mjs';
import { EditFolderDialog } from '../../../components/EditFolderDialog/EditFolderDialog.mjs';
import { FolderCard } from '../../../components/FolderCard/FolderCard/FolderCard.mjs';
import { FolderCardBody } from '../../../components/FolderCard/FolderCardBody/FolderCardBody.mjs';
import { FolderCardBodyAction } from '../../../components/FolderCard/FolderCardBodyAction/FolderCardBodyAction.mjs';
import { FolderCardCheckbox } from '../../../components/FolderCard/FolderCardCheckbox/FolderCardCheckbox.mjs';
import { FolderGridList } from '../../../components/FolderGridList/FolderGridList.mjs';
import { SortPicker } from '../../../components/SortPicker/SortPicker.mjs';
import { TableList } from '../../../components/TableList/TableList.mjs';
import { UploadAssetDialog } from '../../../components/UploadAssetDialog/UploadAssetDialog.mjs';
import { viewOptions, localStorageKeys } from '../../../constants.mjs';
import { useAIAvailability } from '../../../hooks/useAiAvailability.mjs';
import { useAssets } from '../../../hooks/useAssets.mjs';
import { useFolder } from '../../../hooks/useFolder.mjs';
import { useFolders } from '../../../hooks/useFolders.mjs';
import { useMediaLibraryPermissions } from '../../../hooks/useMediaLibraryPermissions.mjs';
import { usePersistentState } from '../../../hooks/usePersistentState.mjs';
import { useSelectionState } from '../../../hooks/useSelectionState.mjs';
import { containsAssetFilter } from '../../../utils/containsAssetFilter.mjs';
import 'byte-size';
import 'date-fns';
import { getTrad } from '../../../utils/getTrad.mjs';
import { getBreadcrumbDataML } from '../../../utils/getBreadcrumbDataML.mjs';
import { getFolderURL } from '../../../utils/getFolderURL.mjs';
import '../../../utils/typeFromMime.mjs';
import '../../../utils/urlYupSchema.mjs';
import { BulkActions } from './components/BulkActions.mjs';
import { EmptyOrNoPermissions } from './components/EmptyOrNoPermissions.mjs';
import { Filters } from './components/Filters.mjs';
import { Header } from './components/Header.mjs';

// TODO: find a better naming convention for the file that was an index file before
const BoxWithHeight = styled(Box)`
  height: 3.2rem;
  display: flex;
  align-items: center;
`;
const TypographyMaxWidth = styled(Typography)`
  max-width: 100%;
`;
const ActionContainer = styled(Box)`
  svg {
    path {
      fill: ${({ theme })=>theme.colors.neutral500};
    }
  }
`;
const MediaLibrary = ()=>{
    const navigate = useNavigate();
    const { canRead, canCreate, canUpdate, canCopyLink, canDownload, canConfigureView, isLoading: permissionsLoading } = useMediaLibraryPermissions();
    const { isEnabled: isAiEnabled, status: aiAvailabilityStatus } = useAIAvailability();
    const currentFolderToEditRef = React.useRef();
    const { formatMessage } = useIntl();
    const { pathname } = useLocation();
    const { trackUsage } = useTracking();
    const [{ query }, setQuery] = useQueryParams();
    const isFiltering = Boolean(query._q || query.filters);
    const [view, setView] = usePersistentState(localStorageKeys.view, viewOptions.GRID);
    const isGridView = view === viewOptions.GRID;
    const { data: assetsData, isLoading: assetsLoading, error: assetsError } = useAssets({
        skipWhen: !canRead,
        query
    });
    const { data: foldersData, isLoading: foldersLoading, error: foldersError } = useFolders({
        enabled: canRead && assetsData?.pagination?.page === 1 && !containsAssetFilter(query),
        query
    });
    const { data: currentFolder, isLoading: isCurrentFolderLoading, error: currentFolderError } = useFolder(query?.folder, {
        enabled: canRead && !!query?.folder
    });
    // Folder was not found: redirect to the media library root
    if (currentFolderError?.name === 'NotFoundError') {
        navigate(pathname);
    }
    const folders = foldersData?.map((folder)=>({
            ...folder,
            type: 'folder',
            folderURL: getFolderURL(pathname, query, {
                folder: folder.id.toString(),
                folderPath: folder.path
            }),
            isSelectable: canUpdate
        })) ?? [];
    const folderCount = folders?.length || 0;
    const assets = assetsData?.results?.map((asset)=>({
            ...asset,
            type: 'asset',
            isSelectable: canUpdate
        })) || [];
    const assetCount = assets?.length ?? 0;
    const totalAssetCount = assetsData?.pagination?.total;
    const isLoading = isCurrentFolderLoading || foldersLoading || permissionsLoading || assetsLoading || aiAvailabilityStatus === 'loading';
    const [showUploadAssetDialog, setShowUploadAssetDialog] = React.useState(false);
    const [showEditFolderDialog, setShowEditFolderDialog] = React.useState(false);
    const [assetToEdit, setAssetToEdit] = React.useState(undefined);
    const [folderToEdit, setFolderToEdit] = React.useState(undefined);
    const [selected, { selectOne, selectAll, setSelections }] = useSelectionState([
        'type',
        'id'
    ], []);
    // reset selection when folder changes to hide bulk actions
    React.useEffect(()=>{
        setSelections([]);
    }, [
        query.folder,
        setSelections
    ]);
    const indeterminateBulkSelect = selected?.length > 0 && selected?.length !== assetCount + folderCount;
    const toggleUploadAssetDialog = ()=>setShowUploadAssetDialog((prev)=>!prev);
    const toggleEditFolderDialog = ({ created = false } = {})=>{
        // folders are only displayed on the first page, therefore
        // we have to navigate the user to that page, in case a folder
        // was created successfully in order for them to see it
        if (created && query?.page !== '1') {
            setQuery({
                ...query,
                page: 1
            });
        }
        setShowEditFolderDialog((prev)=>!prev);
    };
    const handleBulkSelect = (checked, elements)=>{
        if (checked) {
            trackUsage('didSelectAllMediaLibraryElements');
        }
        selectAll(elements);
    };
    const handleChangeSort = (value)=>{
        trackUsage('didSortMediaLibraryElements', {
            location: 'upload',
            sort: value
        });
        setQuery({
            sort: value
        });
    };
    const handleEditFolder = (folder)=>{
        setFolderToEdit(folder);
        setShowEditFolderDialog(true);
    };
    const handleEditFolderClose = (payload)=>{
        setFolderToEdit(null);
        toggleEditFolderDialog(payload);
        if (currentFolderToEditRef.current) {
            currentFolderToEditRef.current.focus();
        }
    };
    const handleAssetDeleted = (numberOfAssets)=>{
        if (numberOfAssets === assetCount && assetsData?.pagination?.page === assetsData?.pagination?.pageCount && assetsData?.pagination?.page && assetsData.pagination.page > 1) {
            setQuery({
                ...query,
                page: assetsData.pagination.page - 1
            });
        }
    };
    const handleBulkActionSuccess = ()=>{
        selectAll();
        handleAssetDeleted(selected.length);
    };
    if (isLoading) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    if (assetsError || foldersError || aiAvailabilityStatus === 'error') {
        return /*#__PURE__*/ jsx(Page.Error, {});
    }
    return /*#__PURE__*/ jsxs(Layouts.Root, {
        children: [
            /*#__PURE__*/ jsxs(Page.Main, {
                children: [
                    /*#__PURE__*/ jsx(Header, {
                        breadcrumbs: !isCurrentFolderLoading ? getBreadcrumbDataML(currentFolder, {
                            pathname,
                            query
                        }) : null,
                        canCreate: canCreate,
                        onToggleEditFolderDialog: toggleEditFolderDialog,
                        onToggleUploadAssetDialog: toggleUploadAssetDialog,
                        folder: currentFolder
                    }),
                    /*#__PURE__*/ jsx(Layouts.Action, {
                        startActions: /*#__PURE__*/ jsxs(Fragment, {
                            children: [
                                canUpdate && isGridView && (assetCount > 0 || folderCount > 0) && /*#__PURE__*/ jsx(BoxWithHeight, {
                                    paddingLeft: 2,
                                    paddingRight: 2,
                                    background: "neutral0",
                                    hasRadius: true,
                                    borderColor: "neutral200",
                                    children: /*#__PURE__*/ jsx(Checkbox, {
                                        "aria-label": formatMessage({
                                            id: getTrad('bulk.select.label'),
                                            defaultMessage: 'Select all folders & assets'
                                        }),
                                        checked: indeterminateBulkSelect ? 'indeterminate' : (assetCount > 0 || folderCount > 0) && selected.length === assetCount + folderCount,
                                        onCheckedChange: (e)=>handleBulkSelect(e, [
                                                ...assets,
                                                ...folders
                                            ])
                                    })
                                }),
                                canRead && isGridView && /*#__PURE__*/ jsx(SortPicker, {
                                    value: query?.sort,
                                    onChangeSort: handleChangeSort
                                }),
                                canRead && /*#__PURE__*/ jsx(Filters, {})
                            ]
                        }),
                        endActions: /*#__PURE__*/ jsxs(Fragment, {
                            children: [
                                canConfigureView ? /*#__PURE__*/ jsx(ActionContainer, {
                                    paddingTop: 1,
                                    paddingBottom: 1,
                                    children: /*#__PURE__*/ jsx(IconButton, {
                                        tag: Link,
                                        to: {
                                            pathname: `${pathname}/configuration`,
                                            search: stringify(query, {
                                                encode: false
                                            })
                                        },
                                        label: formatMessage({
                                            id: 'app.links.configure-view',
                                            defaultMessage: 'Configure the view'
                                        }),
                                        children: /*#__PURE__*/ jsx(Cog, {})
                                    })
                                }) : null,
                                /*#__PURE__*/ jsx(ActionContainer, {
                                    paddingTop: 1,
                                    paddingBottom: 1,
                                    children: /*#__PURE__*/ jsx(IconButton, {
                                        label: isGridView ? formatMessage({
                                            id: getTrad('view-switch.list'),
                                            defaultMessage: 'List View'
                                        }) : formatMessage({
                                            id: getTrad('view-switch.grid'),
                                            defaultMessage: 'Grid View'
                                        }),
                                        onClick: ()=>setView(isGridView ? viewOptions.LIST : viewOptions.GRID),
                                        children: isGridView ? /*#__PURE__*/ jsx(List, {}) : /*#__PURE__*/ jsx(GridFour, {})
                                    })
                                }),
                                /*#__PURE__*/ jsx(SearchInput, {
                                    label: formatMessage({
                                        id: getTrad('search.label'),
                                        defaultMessage: 'Search for an asset'
                                    }),
                                    trackedEvent: "didSearchMediaLibraryElements",
                                    trackedEventDetails: {
                                        location: 'upload'
                                    }
                                })
                            ]
                        })
                    }),
                    /*#__PURE__*/ jsxs(Layouts.Content, {
                        children: [
                            selected.length > 0 && /*#__PURE__*/ jsx(BulkActions, {
                                currentFolder: currentFolder,
                                selected: selected,
                                onSuccess: handleBulkActionSuccess
                            }),
                            folderCount === 0 && assetCount === 0 && /*#__PURE__*/ jsx(EmptyOrNoPermissions, {
                                canCreate: canCreate,
                                canRead: canRead,
                                isFiltering: isFiltering,
                                onActionClick: toggleUploadAssetDialog
                            }),
                            canRead && !isGridView && (assetCount > 0 || folderCount > 0) && /*#__PURE__*/ jsx(TableList, {
                                assetCount: assetCount,
                                folderCount: folderCount,
                                indeterminate: indeterminateBulkSelect,
                                onChangeSort: handleChangeSort,
                                onChangeFolder: (folderID, folderPath)=>navigate(getFolderURL(pathname, query, {
                                        folder: folderID.toString(),
                                        folderPath
                                    })),
                                onEditAsset: setAssetToEdit,
                                onEditFolder: handleEditFolder,
                                onSelectOne: selectOne,
                                onSelectAll: handleBulkSelect,
                                rows: [
                                    ...folders,
                                    ...assets
                                ],
                                selected: selected,
                                shouldDisableBulkSelect: !canUpdate,
                                sortQuery: query?.sort ?? ''
                            }),
                            canRead && isGridView && /*#__PURE__*/ jsxs(Fragment, {
                                children: [
                                    folderCount > 0 && /*#__PURE__*/ jsx(FolderGridList, {
                                        title: (isFiltering && assetCount > 0 || !isFiltering) && formatMessage({
                                            id: getTrad('list.folders.title'),
                                            defaultMessage: 'Folders ({count})'
                                        }, {
                                            count: folderCount
                                        }) || '',
                                        children: folders.map((folder)=>{
                                            const selectedFolders = selected.filter(({ type })=>type === 'folder');
                                            const isSelected = !!selectedFolders.find((currentFolder)=>currentFolder.id === folder.id);
                                            const url = getFolderURL(pathname, query, {
                                                folder: folder?.id.toString(),
                                                folderPath: folder?.path
                                            });
                                            return /*#__PURE__*/ jsx(Grid.Item, {
                                                col: 3,
                                                m: 4,
                                                s: 6,
                                                xs: 12,
                                                direction: "column",
                                                alignItems: "stretch",
                                                children: /*#__PURE__*/ jsx(FolderCard, {
                                                    ref: folderToEdit && folder.id === folderToEdit.id ? currentFolderToEditRef : undefined,
                                                    ariaLabel: folder.name,
                                                    id: `folder-${folder.id}`,
                                                    to: url,
                                                    startAction: folder.isSelectable ? /*#__PURE__*/ jsx(FolderCardCheckbox, {
                                                        "data-testid": `folder-checkbox-${folder.id}`,
                                                        checked: isSelected,
                                                        onCheckedChange: ()=>selectOne(folder)
                                                    }) : null,
                                                    cardActions: /*#__PURE__*/ jsx(IconButton, {
                                                        label: formatMessage({
                                                            id: getTrad('list.folder.edit'),
                                                            defaultMessage: 'Edit folder'
                                                        }),
                                                        onClick: ()=>handleEditFolder(folder),
                                                        children: /*#__PURE__*/ jsx(Pencil, {})
                                                    }),
                                                    children: /*#__PURE__*/ jsx(FolderCardBody, {
                                                        children: /*#__PURE__*/ jsx(FolderCardBodyAction, {
                                                            to: url,
                                                            children: /*#__PURE__*/ jsxs(Flex, {
                                                                tag: "h2",
                                                                direction: "column",
                                                                alignItems: "start",
                                                                maxWidth: "100%",
                                                                children: [
                                                                    /*#__PURE__*/ jsxs(TypographyMaxWidth, {
                                                                        fontWeight: "semiBold",
                                                                        textColor: "neutral800",
                                                                        ellipsis: true,
                                                                        children: [
                                                                            folder.name,
                                                                            /*#__PURE__*/ jsx(VisuallyHidden, {
                                                                                children: ":"
                                                                            })
                                                                        ]
                                                                    }),
                                                                    /*#__PURE__*/ jsx(TypographyMaxWidth, {
                                                                        tag: "span",
                                                                        textColor: "neutral600",
                                                                        variant: "pi",
                                                                        ellipsis: true,
                                                                        children: formatMessage({
                                                                            id: getTrad('list.folder.subtitle'),
                                                                            defaultMessage: '{folderCount, plural, =0 {# folder} one {# folder} other {# folders}}, {filesCount, plural, =0 {# asset} one {# asset} other {# assets}}'
                                                                        }, {
                                                                            folderCount: folder.children?.count,
                                                                            filesCount: folder.files?.count
                                                                        })
                                                                    })
                                                                ]
                                                            })
                                                        })
                                                    })
                                                })
                                            }, `folder-${folder.id}`);
                                        })
                                    }),
                                    assetCount > 0 && folderCount > 0 && /*#__PURE__*/ jsx(Box, {
                                        paddingTop: 6,
                                        paddingBottom: 4,
                                        children: /*#__PURE__*/ jsx(Divider, {})
                                    }),
                                    assetCount > 0 && /*#__PURE__*/ jsx(AssetGridList, {
                                        assets: assets,
                                        onEditAsset: setAssetToEdit,
                                        onSelectAsset: selectOne,
                                        selectedAssets: selected.filter(({ type })=>type === 'asset'),
                                        title: (!isFiltering || isFiltering && folderCount > 0) && assetsData?.pagination?.page === 1 && formatMessage({
                                            id: getTrad('list.assets.title'),
                                            defaultMessage: 'Assets ({count})'
                                        }, {
                                            count: totalAssetCount
                                        }) || ''
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsxs(Pagination.Root, {
                                ...assetsData?.pagination,
                                children: [
                                    /*#__PURE__*/ jsx(Pagination.PageSize, {}),
                                    /*#__PURE__*/ jsx(Pagination.Links, {})
                                ]
                            })
                        ]
                    })
                ]
            }),
            showUploadAssetDialog && (isAiEnabled ? /*#__PURE__*/ jsx(AIUploadModal, {
                open: showUploadAssetDialog,
                onClose: toggleUploadAssetDialog,
                folderId: query?.folder ? Number(query.folder) : null
            }) : /*#__PURE__*/ jsx(UploadAssetDialog, {
                open: showUploadAssetDialog,
                onClose: toggleUploadAssetDialog,
                trackedLocation: "upload",
                folderId: query?.folder
            })),
            showEditFolderDialog && /*#__PURE__*/ jsx(EditFolderDialog, {
                open: showEditFolderDialog,
                onClose: ()=>handleEditFolderClose(),
                folder: folderToEdit,
                parentFolderId: query?.folder,
                location: "upload"
            }),
            assetToEdit && /*#__PURE__*/ jsx(EditAssetDialog, {
                onClose: (editedAsset)=>{
                    // The asset has been deleted
                    if (editedAsset === null) {
                        handleAssetDeleted(1);
                    }
                    setAssetToEdit(undefined);
                },
                open: !!assetToEdit,
                asset: assetToEdit,
                canUpdate: canUpdate,
                canCopyLink: canCopyLink,
                canDownload: canDownload,
                trackedLocation: "upload"
            })
        ]
    });
};

export { MediaLibrary };
//# sourceMappingURL=MediaLibrary.mjs.map
