import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { Flex, Modal, Loader, Button, Grid, Field, TextInput, VisuallyHidden } from '@strapi/design-system';
import { Formik, Form } from 'formik';
import isEqual from 'lodash/isEqual';
import { useIntl } from 'react-intl';
import { styled } from 'styled-components';
import * as yup from 'yup';
import { useEditAsset } from '../../hooks/useEditAsset.mjs';
import { useFolderStructure } from '../../hooks/useFolderStructure.mjs';
import { useTracking } from '../../hooks/useTracking.mjs';
import { findRecursiveFolderByValue } from '../../utils/findRecursiveFolderByValue.mjs';
import { formatBytes } from '../../utils/formatBytes.mjs';
import 'date-fns';
import { getTrad } from '../../utils/getTrad.mjs';
import 'qs';
import { getFileExtension } from '../../utils/getFileExtension.mjs';
import '../../utils/typeFromMime.mjs';
import '../../utils/urlYupSchema.mjs';
import { ContextInfo } from '../ContextInfo/ContextInfo.mjs';
import { SelectTree } from '../SelectTree/SelectTree.mjs';
import { DialogHeader } from './DialogHeader.mjs';
import { PreviewBox } from './PreviewBox/PreviewBox.mjs';
import { ReplaceMediaButton } from './ReplaceMediaButton.mjs';

const LoadingBody = styled(Flex)`
  /* 80px are coming from the Tabs component that is not included in the ModalBody */
  min-height: ${()=>`calc(60dvh + 8rem)`};
`;
const fileInfoSchema = yup.object({
    name: yup.string().required(),
    alternativeText: yup.string(),
    caption: yup.string(),
    folder: yup.number()
});
const EditAssetContent = ({ onClose, asset, canUpdate = false, canCopyLink = false, canDownload = false, trackedLocation, omitFields = [], omitActions = [] })=>{
    const { formatMessage, formatDate } = useIntl();
    const { trackUsage } = useTracking();
    const submitButtonRef = React.useRef(null);
    const [isCropping, setIsCropping] = React.useState(false);
    const [replacementFile, setReplacementFile] = React.useState();
    const { editAsset, isLoading } = useEditAsset();
    const { data: folderStructure, isLoading: folderStructureIsLoading } = useFolderStructure({
        enabled: true
    });
    const handleSubmit = async (values)=>{
        const nextAsset = {
            ...asset,
            ...values,
            folder: values.parent?.value
        };
        if (asset?.isLocal) {
            onClose(nextAsset);
        } else {
            const editedAsset = await editAsset(nextAsset, replacementFile);
            const assetType = asset?.mime?.split('/')[0];
            // if the folder parent was the root of Media Library, its id is null
            // we know it changed location if the new parent value exists
            const didChangeLocation = asset?.folder?.id ? asset.folder.id !== values.parent?.value : asset?.folder === null && !!values.parent?.value;
            trackUsage('didEditMediaLibraryElements', {
                location: trackedLocation,
                type: assetType,
                changeLocation: didChangeLocation
            });
            onClose(editedAsset);
        }
    };
    const handleStartCropping = ()=>{
        setIsCropping(true);
    };
    const handleCancelCropping = ()=>{
        setIsCropping(false);
    };
    const handleFinishCropping = ()=>{
        setIsCropping(false);
        onClose();
    };
    const formDisabled = !canUpdate || isCropping;
    const handleConfirmClose = ()=>{
        // eslint-disable-next-line no-alert
        const confirm = window.confirm(formatMessage({
            id: 'window.confirm.close-modal.file',
            defaultMessage: 'Are you sure? Your changes will be lost.'
        }));
        if (confirm) {
            onClose();
        }
    };
    const activeFolderId = asset?.folder?.id;
    const initialFormData = !folderStructureIsLoading && {
        name: asset?.name,
        alternativeText: asset?.alternativeText ?? undefined,
        caption: asset?.caption ?? undefined,
        parent: {
            value: activeFolderId ?? undefined,
            label: findRecursiveFolderByValue(folderStructure, activeFolderId)?.label ?? folderStructure[0].label
        }
    };
    const handleClose = (values)=>{
        if (!isEqual(initialFormData, values)) {
            handleConfirmClose();
        } else {
            onClose();
        }
    };
    if (folderStructureIsLoading) {
        return /*#__PURE__*/ jsxs(Fragment, {
            children: [
                /*#__PURE__*/ jsx(DialogHeader, {}),
                /*#__PURE__*/ jsx(LoadingBody, {
                    minHeight: "60vh",
                    justifyContent: "center",
                    paddingTop: 4,
                    paddingBottom: 4,
                    children: /*#__PURE__*/ jsx(Loader, {
                        children: formatMessage({
                            id: getTrad('content.isLoading'),
                            defaultMessage: 'Content is loading.'
                        })
                    })
                }),
                /*#__PURE__*/ jsx(Modal.Footer, {
                    children: /*#__PURE__*/ jsx(Button, {
                        onClick: ()=>handleClose(),
                        variant: "tertiary",
                        children: formatMessage({
                            id: 'cancel',
                            defaultMessage: 'Cancel'
                        })
                    })
                })
            ]
        });
    }
    return /*#__PURE__*/ jsx(Formik, {
        validationSchema: fileInfoSchema,
        validateOnChange: false,
        onSubmit: handleSubmit,
        initialValues: initialFormData,
        children: ({ values, errors, handleChange, setFieldValue })=>/*#__PURE__*/ jsxs(Fragment, {
                children: [
                    /*#__PURE__*/ jsx(DialogHeader, {}),
                    /*#__PURE__*/ jsx(Modal.Body, {
                        children: /*#__PURE__*/ jsxs(Grid.Root, {
                            gap: 4,
                            children: [
                                /*#__PURE__*/ jsx(Grid.Item, {
                                    xs: 12,
                                    col: 6,
                                    direction: "column",
                                    alignItems: "stretch",
                                    children: /*#__PURE__*/ jsx(PreviewBox, {
                                        asset: asset,
                                        canUpdate: canUpdate,
                                        canCopyLink: canCopyLink,
                                        canDownload: canDownload,
                                        onDelete: onClose,
                                        onCropFinish: handleFinishCropping,
                                        onCropStart: handleStartCropping,
                                        onCropCancel: handleCancelCropping,
                                        replacementFile: replacementFile,
                                        trackedLocation: trackedLocation
                                    })
                                }),
                                /*#__PURE__*/ jsx(Grid.Item, {
                                    xs: 12,
                                    col: 6,
                                    direction: "column",
                                    alignItems: "stretch",
                                    children: /*#__PURE__*/ jsxs(Form, {
                                        noValidate: true,
                                        children: [
                                            /*#__PURE__*/ jsxs(Flex, {
                                                direction: "column",
                                                alignItems: "stretch",
                                                gap: 3,
                                                children: [
                                                    /*#__PURE__*/ jsx(ContextInfo, {
                                                        blocks: [
                                                            {
                                                                label: formatMessage({
                                                                    id: getTrad('modal.file-details.size'),
                                                                    defaultMessage: 'Size'
                                                                }),
                                                                value: formatBytes(asset?.size ? asset.size : 0)
                                                            },
                                                            {
                                                                label: formatMessage({
                                                                    id: getTrad('modal.file-details.dimensions'),
                                                                    defaultMessage: 'Dimensions'
                                                                }),
                                                                value: asset?.height && asset.width ? `${asset.width}✕${asset.height}` : null
                                                            },
                                                            {
                                                                label: formatMessage({
                                                                    id: getTrad('modal.file-details.date'),
                                                                    defaultMessage: 'Date'
                                                                }),
                                                                value: formatDate(new Date(asset?.createdAt ? asset.createdAt : ''))
                                                            },
                                                            {
                                                                label: formatMessage({
                                                                    id: getTrad('modal.file-details.extension'),
                                                                    defaultMessage: 'Extension'
                                                                }),
                                                                value: getFileExtension(asset?.ext)
                                                            },
                                                            {
                                                                label: formatMessage({
                                                                    id: getTrad('modal.file-details.id'),
                                                                    defaultMessage: 'Asset ID'
                                                                }),
                                                                value: asset?.id ? asset.id : null
                                                            }
                                                        ]
                                                    }),
                                                    /*#__PURE__*/ jsxs(Field.Root, {
                                                        name: "name",
                                                        error: errors.name,
                                                        children: [
                                                            /*#__PURE__*/ jsx(Field.Label, {
                                                                children: formatMessage({
                                                                    id: getTrad('form.input.label.file-name'),
                                                                    defaultMessage: 'File name'
                                                                })
                                                            }),
                                                            /*#__PURE__*/ jsx(TextInput, {
                                                                value: values.name,
                                                                onChange: handleChange,
                                                                disabled: formDisabled,
                                                                type: "text"
                                                            }),
                                                            /*#__PURE__*/ jsx(Field.Error, {})
                                                        ]
                                                    }),
                                                    !omitFields?.includes('alternativeText') && /*#__PURE__*/ jsxs(Field.Root, {
                                                        name: "alternativeText",
                                                        hint: formatMessage({
                                                            id: getTrad('form.input.description.file-alt'),
                                                            defaultMessage: 'This text will be displayed if the asset can’t be shown.'
                                                        }),
                                                        error: errors.alternativeText,
                                                        children: [
                                                            /*#__PURE__*/ jsx(Field.Label, {
                                                                children: formatMessage({
                                                                    id: getTrad('form.input.label.file-alt'),
                                                                    defaultMessage: 'Alternative text'
                                                                })
                                                            }),
                                                            /*#__PURE__*/ jsx(TextInput, {
                                                                value: values.alternativeText,
                                                                onChange: handleChange,
                                                                disabled: formDisabled,
                                                                type: "text"
                                                            }),
                                                            /*#__PURE__*/ jsx(Field.Hint, {}),
                                                            /*#__PURE__*/ jsx(Field.Error, {})
                                                        ]
                                                    }),
                                                    !omitFields?.includes('caption') && /*#__PURE__*/ jsxs(Field.Root, {
                                                        name: "caption",
                                                        error: errors.caption,
                                                        children: [
                                                            /*#__PURE__*/ jsx(Field.Label, {
                                                                children: formatMessage({
                                                                    id: getTrad('form.input.label.file-caption'),
                                                                    defaultMessage: 'Caption'
                                                                })
                                                            }),
                                                            /*#__PURE__*/ jsx(TextInput, {
                                                                value: values.caption,
                                                                onChange: handleChange,
                                                                disabled: formDisabled,
                                                                type: "text"
                                                            })
                                                        ]
                                                    }),
                                                    /*#__PURE__*/ jsx(Flex, {
                                                        direction: "column",
                                                        alignItems: "stretch",
                                                        gap: 1,
                                                        children: /*#__PURE__*/ jsxs(Field.Root, {
                                                            name: "parent",
                                                            id: "asset-folder",
                                                            children: [
                                                                /*#__PURE__*/ jsx(Field.Label, {
                                                                    children: formatMessage({
                                                                        id: getTrad('form.input.label.file-location'),
                                                                        defaultMessage: 'Location'
                                                                    })
                                                                }),
                                                                /*#__PURE__*/ jsx(SelectTree, {
                                                                    name: "parent",
                                                                    defaultValue: values.parent,
                                                                    options: folderStructure,
                                                                    onChange: (value)=>{
                                                                        setFieldValue('parent', value);
                                                                    },
                                                                    menuPortalTarget: document.querySelector('body'),
                                                                    inputId: "asset-folder",
                                                                    isDisabled: formDisabled,
                                                                    error: errors?.parent,
                                                                    ariaErrorMessage: "folder-parent-error"
                                                                })
                                                            ]
                                                        })
                                                    })
                                                ]
                                            }),
                                            /*#__PURE__*/ jsx(VisuallyHidden, {
                                                children: /*#__PURE__*/ jsx("button", {
                                                    type: "submit",
                                                    tabIndex: -1,
                                                    ref: submitButtonRef,
                                                    disabled: formDisabled,
                                                    children: formatMessage({
                                                        id: 'submit',
                                                        defaultMessage: 'Submit'
                                                    })
                                                })
                                            })
                                        ]
                                    })
                                })
                            ]
                        })
                    }),
                    /*#__PURE__*/ jsxs(Modal.Footer, {
                        children: [
                            /*#__PURE__*/ jsx(Button, {
                                onClick: ()=>handleClose({
                                        ...values
                                    }),
                                variant: "tertiary",
                                children: formatMessage({
                                    id: 'global.cancel',
                                    defaultMessage: 'Cancel'
                                })
                            }),
                            /*#__PURE__*/ jsxs(Flex, {
                                gap: 2,
                                children: [
                                    !omitActions?.includes('replace') && /*#__PURE__*/ jsx(ReplaceMediaButton, {
                                        onSelectMedia: setReplacementFile,
                                        acceptedMime: asset?.mime ?? '',
                                        disabled: formDisabled,
                                        trackedLocation: trackedLocation
                                    }),
                                    /*#__PURE__*/ jsx(Button, {
                                        onClick: ()=>submitButtonRef.current?.click(),
                                        loading: isLoading,
                                        disabled: formDisabled,
                                        children: formatMessage({
                                            id: 'global.finish',
                                            defaultMessage: 'Finish'
                                        })
                                    })
                                ]
                            })
                        ]
                    })
                ]
            })
    });
};
const EditAssetDialog = ({ open, onClose, canUpdate = false, canCopyLink = false, canDownload = false, ...restProps })=>{
    return /*#__PURE__*/ jsx(Modal.Root, {
        open: open,
        onOpenChange: onClose,
        children: /*#__PURE__*/ jsx(Modal.Content, {
            children: /*#__PURE__*/ jsx(EditAssetContent, {
                onClose: onClose,
                canUpdate: canUpdate,
                canCopyLink: canCopyLink,
                canDownload: canDownload,
                ...restProps
            })
        })
    });
};

export { EditAssetContent, EditAssetDialog };
//# sourceMappingURL=EditAssetContent.mjs.map
