import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { Page } from '@strapi/admin/strapi-admin';
import { Flex, Tabs, Modal, Loader, Badge, Button, Divider, ScrollArea } from '@strapi/design-system';
import { useIntl } from 'react-intl';
import { styled } from 'styled-components';
import { useAssets } from '../../hooks/useAssets.mjs';
import { useFolders } from '../../hooks/useFolders.mjs';
import { useMediaLibraryPermissions } from '../../hooks/useMediaLibraryPermissions.mjs';
import { useModalQueryParams } from '../../hooks/useModalQueryParams.mjs';
import { useSelectionState } from '../../hooks/useSelectionState.mjs';
import { containsAssetFilter } from '../../utils/containsAssetFilter.mjs';
import 'byte-size';
import 'date-fns';
import { getAllowedFiles } from '../../utils/getAllowedFiles.mjs';
import { getTrad } from '../../utils/getTrad.mjs';
import 'qs';
import { moveElement } from '../../utils/moveElement.mjs';
import '../../utils/typeFromMime.mjs';
import '../../utils/urlYupSchema.mjs';
import { EditAssetContent } from '../EditAssetDialog/EditAssetContent.mjs';
import { EditFolderContent } from '../EditFolderDialog/EditFolderDialog.mjs';
import { BrowseStep } from './BrowseStep/BrowseStep.mjs';
import { DialogFooter } from './DialogFooter.mjs';
import { SelectedStep } from './SelectedStep/SelectedStep.mjs';

// TODO: find a better naming convention for the file that was an index file before
const LoadingBody = styled(Flex)`
  /* 80px are coming from the Tabs component that is not included in the ModalBody */
  min-height: ${()=>`calc(60dvh + 8rem)`};
`;
const AssetContent = ({ allowedTypes = [], folderId = null, onClose, onAddAsset, onAddFolder, onChangeFolder, onValidate, multiple = false, initiallySelectedAssets = [], trackedLocation })=>{
    const [assetToEdit, setAssetToEdit] = React.useState(undefined);
    const [folderToEdit, setFolderToEdit] = React.useState(undefined);
    const { formatMessage } = useIntl();
    const { canRead, canCreate, isLoading: isLoadingPermissions, canUpdate, canCopyLink, canDownload } = useMediaLibraryPermissions();
    const [{ queryObject }, { onChangeFilters, onChangePage, onChangePageSize, onChangeSort, onChangeSearch, onChangeFolder: onChangeFolderParam }] = useModalQueryParams({
        folder: folderId
    });
    const { data: { pagination, results: assets } = {}, isLoading: isLoadingAssets, error: errorAssets } = useAssets({
        skipWhen: !canRead,
        query: queryObject
    });
    const { data: folders, isLoading: isLoadingFolders, error: errorFolders } = useFolders({
        enabled: canRead && !containsAssetFilter(queryObject) && pagination?.page === 1,
        query: queryObject
    });
    const [selectedAssets, { selectOne, selectOnly, setSelections, selectMultiple, deselectMultiple }] = useSelectionState([
        'id'
    ], initiallySelectedAssets);
    const handleSelectAllAssets = ()=>{
        const allowedAssets = getAllowedFiles(allowedTypes, assets);
        if (!multiple) {
            return undefined;
        }
        // selected files in current folder
        const alreadySelected = allowedAssets.filter((asset)=>selectedAssets.findIndex((selectedAsset)=>selectedAsset.id === asset.id) !== -1);
        if (alreadySelected.length > 0) {
            deselectMultiple(alreadySelected);
        } else {
            selectMultiple(allowedAssets);
        }
    };
    const handleSelectAsset = (asset)=>{
        return multiple ? selectOne(asset) : selectOnly(asset);
    };
    const isLoading = isLoadingPermissions || isLoadingAssets || isLoadingFolders;
    const hasError = errorAssets || errorFolders;
    const [activeTab, setActiveTab] = React.useState(selectedAssets.length > 0 ? 'selected' : 'browse');
    if (isLoading) {
        return /*#__PURE__*/ jsxs(Fragment, {
            children: [
                /*#__PURE__*/ jsx(Modal.Header, {
                    children: /*#__PURE__*/ jsx(Modal.Title, {
                        children: formatMessage({
                            id: getTrad('header.actions.add-assets'),
                            defaultMessage: 'Add new assets'
                        })
                    })
                }),
                /*#__PURE__*/ jsx(LoadingBody, {
                    justifyContent: "center",
                    paddingTop: 4,
                    paddingBottom: 4,
                    children: /*#__PURE__*/ jsx(Loader, {
                        children: formatMessage({
                            id: getTrad('content.isLoading'),
                            defaultMessage: 'Content is loading.'
                        })
                    })
                }),
                /*#__PURE__*/ jsx(DialogFooter, {
                    onClose: onClose
                })
            ]
        });
    }
    if (hasError) {
        return /*#__PURE__*/ jsxs(Fragment, {
            children: [
                /*#__PURE__*/ jsx(Modal.Header, {
                    children: /*#__PURE__*/ jsx(Modal.Title, {
                        children: formatMessage({
                            id: getTrad('header.actions.add-assets'),
                            defaultMessage: 'Add new assets'
                        })
                    })
                }),
                /*#__PURE__*/ jsx(Page.Error, {}),
                /*#__PURE__*/ jsx(DialogFooter, {
                    onClose: onClose
                })
            ]
        });
    }
    if (!canRead) {
        return /*#__PURE__*/ jsxs(Fragment, {
            children: [
                /*#__PURE__*/ jsx(Modal.Header, {
                    children: /*#__PURE__*/ jsx(Modal.Title, {
                        children: formatMessage({
                            id: getTrad('header.actions.add-assets'),
                            defaultMessage: 'Add new assets'
                        })
                    })
                }),
                /*#__PURE__*/ jsx(Page.NoPermissions, {}),
                /*#__PURE__*/ jsx(DialogFooter, {
                    onClose: onClose
                })
            ]
        });
    }
    if (assetToEdit) {
        return /*#__PURE__*/ jsx(EditAssetContent, {
            onClose: ()=>setAssetToEdit(undefined),
            asset: assetToEdit,
            canUpdate: canUpdate,
            canCopyLink: canCopyLink,
            canDownload: canDownload,
            trackedLocation: trackedLocation
        });
    }
    if (folderToEdit) {
        return /*#__PURE__*/ jsx(EditFolderContent, {
            folder: folderToEdit,
            onClose: ()=>setFolderToEdit(undefined),
            location: "content-manager",
            parentFolderId: queryObject?.folder
        });
    }
    const handleMoveItem = (hoverIndex, destIndex)=>{
        const offset = destIndex - hoverIndex;
        const orderedAssetsClone = selectedAssets.slice();
        const nextAssets = moveElement(orderedAssetsClone, hoverIndex, offset);
        setSelections(nextAssets);
    };
    const handleFolderChange = (folderId, folderPath)=>{
        onChangeFolder(folderId);
        if (onChangeFolderParam) {
            onChangeFolderParam(folderId, folderPath);
        }
    };
    return /*#__PURE__*/ jsxs(Fragment, {
        children: [
            /*#__PURE__*/ jsx(Modal.Header, {
                children: /*#__PURE__*/ jsx(Modal.Title, {
                    children: formatMessage({
                        id: getTrad('header.actions.add-assets'),
                        defaultMessage: 'Add new assets'
                    })
                })
            }),
            /*#__PURE__*/ jsxs(TabsRoot, {
                variant: "simple",
                value: activeTab,
                onValueChange: setActiveTab,
                children: [
                    /*#__PURE__*/ jsxs(Flex, {
                        paddingLeft: 8,
                        paddingRight: 8,
                        paddingTop: 6,
                        justifyContent: "space-between",
                        children: [
                            /*#__PURE__*/ jsxs(Tabs.List, {
                                children: [
                                    /*#__PURE__*/ jsx(Tabs.Trigger, {
                                        value: "browse",
                                        children: formatMessage({
                                            id: getTrad('modal.nav.browse'),
                                            defaultMessage: 'Browse'
                                        })
                                    }),
                                    /*#__PURE__*/ jsxs(Tabs.Trigger, {
                                        value: "selected",
                                        children: [
                                            formatMessage({
                                                id: getTrad('modal.header.select-files'),
                                                defaultMessage: 'Selected files'
                                            }),
                                            /*#__PURE__*/ jsx(Badge, {
                                                marginLeft: 2,
                                                children: selectedAssets.length
                                            })
                                        ]
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsxs(Flex, {
                                gap: 2,
                                children: [
                                    /*#__PURE__*/ jsx(Button, {
                                        variant: "secondary",
                                        onClick: ()=>onAddFolder({
                                                folderId: queryObject?.folder
                                            }),
                                        children: formatMessage({
                                            id: getTrad('modal.upload-list.sub-header.add-folder'),
                                            defaultMessage: 'Add folder'
                                        })
                                    }),
                                    /*#__PURE__*/ jsx(Button, {
                                        onClick: ()=>onAddAsset({
                                                folderId: queryObject?.folder
                                            }),
                                        children: formatMessage({
                                            id: getTrad('modal.upload-list.sub-header.button'),
                                            defaultMessage: 'Add more assets'
                                        })
                                    })
                                ]
                            })
                        ]
                    }),
                    /*#__PURE__*/ jsx(Divider, {}),
                    /*#__PURE__*/ jsx(ScrollArea, {
                        children: /*#__PURE__*/ jsxs(Modal.Body, {
                            children: [
                                /*#__PURE__*/ jsx(Tabs.Content, {
                                    value: "browse",
                                    children: /*#__PURE__*/ jsx(BrowseStep, {
                                        allowedTypes: allowedTypes,
                                        assets: assets,
                                        canCreate: canCreate,
                                        canRead: canRead,
                                        folders: folders,
                                        onSelectAsset: handleSelectAsset,
                                        selectedAssets: selectedAssets,
                                        multiple: multiple,
                                        onSelectAllAsset: handleSelectAllAssets,
                                        onEditAsset: setAssetToEdit,
                                        onEditFolder: setFolderToEdit,
                                        pagination: pagination,
                                        queryObject: queryObject,
                                        onAddAsset: onAddAsset,
                                        onChangeFilters: (filters)=>onChangeFilters(filters),
                                        onChangeFolder: handleFolderChange,
                                        onChangePage: onChangePage,
                                        onChangePageSize: onChangePageSize,
                                        onChangeSort: (sort)=>onChangeSort(sort),
                                        onChangeSearch: onChangeSearch
                                    })
                                }),
                                /*#__PURE__*/ jsx(Tabs.Content, {
                                    value: "selected",
                                    children: /*#__PURE__*/ jsx(SelectedStep, {
                                        selectedAssets: selectedAssets,
                                        onSelectAsset: handleSelectAsset,
                                        onReorderAsset: handleMoveItem
                                    })
                                })
                            ]
                        })
                    })
                ]
            }),
            /*#__PURE__*/ jsx(DialogFooter, {
                onClose: onClose,
                onValidate: ()=>onValidate(selectedAssets)
            })
        ]
    });
};
const AssetDialog = ({ open = false, onClose, ...restProps })=>{
    return /*#__PURE__*/ jsx(Modal.Root, {
        open: open,
        onOpenChange: onClose,
        children: /*#__PURE__*/ jsx(Modal.Content, {
            children: /*#__PURE__*/ jsx(AssetContent, {
                onClose: onClose,
                ...restProps
            })
        })
    });
};
const TabsRoot = styled(Tabs.Root)`
  display: flex;
  flex-direction: column;
  overflow: hidden;
`;

export { AssetContent, AssetDialog };
//# sourceMappingURL=AssetDialog.mjs.map
