import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { ConfirmDialog } from '@strapi/admin/strapi-admin';
import { CardAction, Box, CardTimer, CardBody, Card, Grid, CardHeader, CardContent, Flex, Typography, CardTitle, CardBadge, CardSubtitle, Field, TextInput, Dialog, IconButton, Modal, CardAsset } from '@strapi/design-system';
import { Sparkle, Trash, Pencil } from '@strapi/icons';
import { useIntl } from 'react-intl';
import { styled } from 'styled-components';
import { AudioPreview } from '../../components/AssetCard/AudioPreview.mjs';
import { VideoPreview } from '../../components/AssetCard/VideoPreview.mjs';
import { EditAssetContent } from '../../components/EditAssetDialog/EditAssetContent.mjs';
import { AssetType, DocType } from '../../enums.mjs';
import { useMediaLibraryPermissions } from '../../hooks/useMediaLibraryPermissions.mjs';
import { useRemoveAsset } from '../../hooks/useRemoveAsset.mjs';
import { prefixFileUrlWithBackendUrl } from '../../utils/prefixFileUrlWithBackendUrl.mjs';
import { formatBytes } from '../../utils/formatBytes.mjs';
import { formatDuration } from '../../utils/formatDuration.mjs';
import { getTrad } from '../../utils/getTrad.mjs';
import 'qs';
import { getFileExtension } from '../../utils/getFileExtension.mjs';
import { typeFromMime } from '../../utils/typeFromMime.mjs';
import '../../utils/urlYupSchema.mjs';
import { useAIUploadModalContext } from './AIUploadModal.mjs';

const CardActionsContainer = styled(CardAction)`
  opacity: 0;
  z-index: 1;

  &:focus-within {
    opacity: 1;
  }
`;
const CardContainer = styled(Box)`
  background: ${({ theme })=>theme.colors.neutral0};
  border: 1px solid ${({ theme })=>theme.colors.neutral150};
  border-radius: ${({ theme })=>theme.borderRadius};

  &:hover {
    ${CardActionsContainer} {
      opacity: 1;
    }
  }
`;
/* -------------------------------------------------------------------------------------------------
 * AssetCardActions
 * -----------------------------------------------------------------------------------------------*/ const AssetCardActions = ({ asset })=>{
    const { formatMessage } = useIntl();
    const dispatch = useAIUploadModalContext('AssetCardActions', (s)=>s.dispatch);
    const state = useAIUploadModalContext('AssetCardActions', (s)=>s.state);
    const onClose = useAIUploadModalContext('AssetCardActions', (s)=>s.onClose);
    const { canUpdate, canCopyLink, canDownload } = useMediaLibraryPermissions();
    const [isEditModalOpen, setIsEditModalOpen] = React.useState(false);
    const { removeAsset } = useRemoveAsset(()=>{});
    const handleConfirmRemove = async (event)=>{
        event?.preventDefault();
        await removeAsset(asset.id);
        dispatch({
            type: 'remove_uploaded_asset',
            payload: {
                id: asset.id
            }
        });
        // Close modal if this was the last asset
        if (state.uploadedAssets.length === 1) {
            onClose();
        }
    };
    const handlePropagationClick = (event)=>{
        event.stopPropagation();
    };
    const handleEditAsset = (editedAsset)=>{
        if (editedAsset) {
            dispatch({
                type: 'edit_uploaded_asset',
                payload: {
                    editedAsset
                }
            });
            setIsEditModalOpen(false);
        }
    };
    return /*#__PURE__*/ jsxs(CardActionsContainer, {
        onClick: handlePropagationClick,
        position: "end",
        children: [
            /*#__PURE__*/ jsxs(Dialog.Root, {
                children: [
                    /*#__PURE__*/ jsx(Dialog.Trigger, {
                        children: /*#__PURE__*/ jsx(IconButton, {
                            label: formatMessage({
                                id: getTrad('control-card.remove-selection'),
                                defaultMessage: 'Remove from selection'
                            }),
                            children: /*#__PURE__*/ jsx(Trash, {})
                        })
                    }),
                    /*#__PURE__*/ jsx(ConfirmDialog, {
                        onConfirm: handleConfirmRemove
                    })
                ]
            }),
            /*#__PURE__*/ jsxs(Modal.Root, {
                open: isEditModalOpen,
                onOpenChange: setIsEditModalOpen,
                children: [
                    /*#__PURE__*/ jsx(Modal.Trigger, {
                        children: /*#__PURE__*/ jsx(IconButton, {
                            label: formatMessage({
                                id: getTrad('control-card.edit'),
                                defaultMessage: 'Edit'
                            }),
                            children: /*#__PURE__*/ jsx(Pencil, {})
                        })
                    }),
                    /*#__PURE__*/ jsx(Modal.Content, {
                        children: /*#__PURE__*/ jsx(EditAssetContent, {
                            // Is Local must be set to false to trigger the correct branch of logic in the EditAssetContent on submit
                            asset: {
                                ...asset,
                                isLocal: false,
                                folder: typeof asset.folder === 'number' ? {
                                    id: asset.folder
                                } : asset.folder
                            },
                            onClose: (arg)=>handleEditAsset(arg),
                            canUpdate: canUpdate,
                            canCopyLink: canCopyLink,
                            canDownload: canDownload,
                            omitFields: [
                                'caption',
                                'alternativeText'
                            ],
                            omitActions: [
                                'replace'
                            ]
                        })
                    })
                ]
            })
        ]
    });
};
const Extension = styled.span`
  text-transform: uppercase;
`;
const VideoPreviewWrapper = styled(Box)`
  position: relative;
  height: 100%;
  overflow: hidden;

  canvas,
  video {
    display: block;
    pointer-events: none;
    width: 100%;
    height: 100%;
    object-fit: cover;
    border-radius: ${({ theme })=>theme.borderRadius};
  }
`;
const VideoTimerOverlay = styled(CardTimer)`
  position: absolute;
  bottom: 0.5rem;
  right: 0.5rem;
`;
const AudioPreviewWrapper = styled(Box)`
  height: 100%;
  overflow: hidden;
  display: flex;
  align-items: center;
  justify-content: center;

  canvas,
  audio {
    display: block;
    max-width: 100%;
    max-height: 100%;
  }
`;
const Asset = ({ assetType, thumbnailUrl, assetUrl, asset })=>{
    const [duration, setDuration] = React.useState();
    const formattedDuration = duration ? formatDuration(duration) : undefined;
    switch(assetType){
        case AssetType.Image:
            return /*#__PURE__*/ jsx(CardAsset, {
                src: thumbnailUrl,
                size: "S",
                alt: asset.alternativeText || asset.name
            });
        case AssetType.Video:
            return /*#__PURE__*/ jsx(CardAsset, {
                size: "S",
                children: /*#__PURE__*/ jsxs(VideoPreviewWrapper, {
                    children: [
                        /*#__PURE__*/ jsx(VideoPreview, {
                            url: assetUrl,
                            mime: asset.mime || 'video/mp4',
                            onLoadDuration: setDuration,
                            alt: asset.alternativeText || asset.name
                        }),
                        formattedDuration && /*#__PURE__*/ jsx(VideoTimerOverlay, {
                            children: formattedDuration
                        })
                    ]
                })
            });
        case AssetType.Audio:
            return /*#__PURE__*/ jsx(CardAsset, {
                size: "S",
                children: /*#__PURE__*/ jsx(AudioPreviewWrapper, {
                    children: /*#__PURE__*/ jsx(AudioPreview, {
                        url: assetUrl,
                        alt: asset.alternativeText || asset.name
                    })
                })
            });
        default:
            return /*#__PURE__*/ jsx(CardAsset, {
                src: thumbnailUrl,
                size: "S",
                alt: asset.alternativeText || asset.name
            });
    }
};
/* -------------------------------------------------------------------------------------------------
 * AssetCard
 * -----------------------------------------------------------------------------------------------*/ const StyledCardBody = styled(CardBody)`
  display: flex;
  padding: ${({ theme })=>theme.spaces[2]} ${({ theme })=>theme.spaces[1]};
`;
const StyledCard = styled(Card)`
  width: 100%;
  height: 100%;
  border: none;
  box-shadow: none;
  border-radius: 0;
  padding: 0;
`;
const getAssetBadgeLabel = (assetType)=>{
    switch(assetType){
        case AssetType.Image:
            return {
                id: getTrad('settings.section.image.label'),
                defaultMessage: 'IMAGE'
            };
        case AssetType.Video:
            return {
                id: getTrad('settings.section.video.label'),
                defaultMessage: 'VIDEO'
            };
        case AssetType.Audio:
            return {
                id: getTrad('settings.section.audio.label'),
                defaultMessage: 'AUDIO'
            };
        case DocType.Pdf:
            return {
                id: getTrad('settings.section.pdf.label'),
                defaultMessage: 'PDF'
            };
        case DocType.Csv:
            return {
                id: getTrad('settings.section.csv.label'),
                defaultMessage: 'CSV'
            };
        case DocType.Xls:
            return {
                id: getTrad('settings.section.xls.label'),
                defaultMessage: 'XLS'
            };
        case DocType.Zip:
            return {
                id: getTrad('settings.section.zip.label'),
                defaultMessage: 'ZIP'
            };
        default:
            return {
                id: getTrad('settings.section.doc.label'),
                defaultMessage: 'DOC'
            };
    }
};
const AIAssetCard = ({ asset, onCaptionChange, onAltTextChange, wasAltTextChanged, wasCaptionChanged })=>{
    const { formatMessage } = useIntl();
    const assetType = typeFromMime(asset.mime || '');
    const thumbnailUrl = prefixFileUrlWithBackendUrl(asset?.formats?.thumbnail?.url || asset.url) || '';
    const assetUrl = prefixFileUrlWithBackendUrl(asset.url) || '';
    const subtitle = asset.height && asset.width ? ` - ${asset.width}x${asset.height}` : '';
    const formattedSize = asset.size ? formatBytes(asset.size) : '';
    const fullSubtitle = `${subtitle}${subtitle && formattedSize ? ' - ' : ''}${formattedSize}`;
    const [caption, setCaption] = React.useState(asset.caption || '');
    React.useEffect(()=>{
        onCaptionChange(caption);
    }, [
        caption,
        onCaptionChange
    ]);
    const [altText, setAltText] = React.useState(asset.alternativeText || '');
    React.useEffect(()=>{
        onAltTextChange(altText);
    }, [
        altText,
        onAltTextChange
    ]);
    return /*#__PURE__*/ jsx(CardContainer, {
        children: /*#__PURE__*/ jsxs(Grid.Root, {
            children: [
                /*#__PURE__*/ jsx(Grid.Item, {
                    m: 5,
                    xs: 12,
                    alignItems: "stretch",
                    children: /*#__PURE__*/ jsxs(StyledCard, {
                        width: "100%",
                        height: "100%",
                        shadow: "none",
                        borderRadius: 0,
                        padding: 0,
                        children: [
                            /*#__PURE__*/ jsxs(CardHeader, {
                                style: {
                                    borderStyle: 'none'
                                },
                                children: [
                                    /*#__PURE__*/ jsx(AssetCardActions, {
                                        asset: asset
                                    }),
                                    /*#__PURE__*/ jsx(Asset, {
                                        assetType: assetType,
                                        thumbnailUrl: thumbnailUrl,
                                        assetUrl: assetUrl,
                                        asset: asset
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsx(StyledCardBody, {
                                children: /*#__PURE__*/ jsxs(CardContent, {
                                    width: "100%",
                                    children: [
                                        /*#__PURE__*/ jsxs(Flex, {
                                            justifyContent: "space-between",
                                            alignItems: "start",
                                            children: [
                                                /*#__PURE__*/ jsx(Typography, {
                                                    tag: "h2",
                                                    children: /*#__PURE__*/ jsx(CardTitle, {
                                                        tag: "span",
                                                        children: asset.name
                                                    })
                                                }),
                                                /*#__PURE__*/ jsx(CardBadge, {
                                                    children: formatMessage(getAssetBadgeLabel(assetType))
                                                })
                                            ]
                                        }),
                                        /*#__PURE__*/ jsx(Flex, {
                                            children: /*#__PURE__*/ jsxs(CardSubtitle, {
                                                children: [
                                                    /*#__PURE__*/ jsx(Extension, {
                                                        children: getFileExtension(asset.ext)
                                                    }),
                                                    fullSubtitle
                                                ]
                                            })
                                        })
                                    ]
                                })
                            })
                        ]
                    })
                }),
                /*#__PURE__*/ jsx(Grid.Item, {
                    m: 7,
                    xs: 12,
                    flex: 1,
                    children: /*#__PURE__*/ jsxs(Flex, {
                        direction: "column",
                        height: "100%",
                        alignItems: "stretch",
                        flex: 1,
                        padding: 4,
                        gap: 2,
                        children: [
                            /*#__PURE__*/ jsxs(Field.Root, {
                                name: "caption",
                                children: [
                                    /*#__PURE__*/ jsx(Flex, {
                                        alignItems: "center",
                                        gap: 2,
                                        children: /*#__PURE__*/ jsx(Field.Label, {
                                            children: formatMessage({
                                                id: getTrad('form.input.label.file-caption'),
                                                defaultMessage: 'Caption'
                                            })
                                        })
                                    }),
                                    /*#__PURE__*/ jsx(TextInput, {
                                        value: caption,
                                        onChange: (e)=>setCaption(e.target.value),
                                        placeholder: formatMessage({
                                            id: getTrad('form.input.placeholder.file-caption'),
                                            defaultMessage: 'Enter caption'
                                        }),
                                        endAction: !wasCaptionChanged && asset.caption && /*#__PURE__*/ jsx(Sparkle, {
                                            width: "16px",
                                            height: "16px",
                                            fill: "#AC73E6"
                                        }),
                                        type: "text"
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsxs(Field.Root, {
                                name: "alternativeText",
                                hint: formatMessage({
                                    id: getTrad('form.input.description.file-alt'),
                                    defaultMessage: "This text will be displayed if the asset can't be shown."
                                }),
                                children: [
                                    /*#__PURE__*/ jsx(Flex, {
                                        alignItems: "center",
                                        gap: 2,
                                        children: /*#__PURE__*/ jsx(Field.Label, {
                                            children: formatMessage({
                                                id: getTrad('form.input.label.file-alt'),
                                                defaultMessage: 'Alternative text'
                                            })
                                        })
                                    }),
                                    /*#__PURE__*/ jsx(TextInput, {
                                        value: altText,
                                        onChange: (e)=>setAltText(e.target.value),
                                        placeholder: formatMessage({
                                            id: getTrad('form.input.placeholder.file-alt'),
                                            defaultMessage: 'Enter alternative text'
                                        }),
                                        endAction: !wasAltTextChanged && asset.alternativeText && /*#__PURE__*/ jsx(Sparkle, {
                                            width: "16px",
                                            height: "16px",
                                            fill: "#AC73E6"
                                        }),
                                        type: "text"
                                    }),
                                    /*#__PURE__*/ jsx(Field.Hint, {})
                                ]
                            })
                        ]
                    })
                })
            ]
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * AssetCardSkeletons
 * -----------------------------------------------------------------------------------------------*/ const SkeletonBox = styled(Box)`
  background: linear-gradient(
    90deg,
    ${({ theme })=>theme.colors.neutral100} 25%,
    ${({ theme })=>theme.colors.neutral150} 50%,
    ${({ theme })=>theme.colors.neutral100} 75%
  );
  background-size: 200% 100%;
  animation: loading 1.5s infinite;
  border-radius: ${({ theme })=>theme.borderRadius};
  width: ${({ width })=>width || '100%'};
  height: ${({ height })=>height || '1rem'};

  @keyframes loading {
    0% {
      background-position: 200% 0;
    }
    100% {
      background-position: -200% 0;
    }
  }
`;
const AIAssetCardSkeletons = ({ count = 1 })=>{
    const skeletons = Array.from({
        length: count
    }, (_, i)=>i);
    return skeletons.map((index)=>/*#__PURE__*/ jsx(Box, {
            background: "neutral0",
            borderColor: "neutral150",
            borderStyle: "solid",
            borderWidth: "1px",
            borderRadius: "4px",
            marginBottom: 4,
            children: /*#__PURE__*/ jsxs(Grid.Root, {
                children: [
                    /*#__PURE__*/ jsx(Grid.Item, {
                        col: 5,
                        alignItems: "stretch",
                        children: /*#__PURE__*/ jsxs(Card, {
                            height: "100%",
                            width: "100%",
                            borderStyle: "none",
                            shadow: "none",
                            borderRadius: 0,
                            padding: 2,
                            children: [
                                /*#__PURE__*/ jsx(Box, {
                                    height: "150px",
                                    padding: 2,
                                    children: /*#__PURE__*/ jsx(SkeletonBox, {
                                        height: "100%"
                                    })
                                }),
                                /*#__PURE__*/ jsx(CardBody, {
                                    style: {
                                        display: 'flex',
                                        padding: '8px 4px'
                                    },
                                    children: /*#__PURE__*/ jsxs(CardContent, {
                                        width: "100%",
                                        children: [
                                            /*#__PURE__*/ jsxs(Flex, {
                                                justifyContent: "space-between",
                                                alignItems: "start",
                                                marginBottom: 1,
                                                children: [
                                                    /*#__PURE__*/ jsx(SkeletonBox, {
                                                        width: "60%",
                                                        height: "18px"
                                                    }),
                                                    /*#__PURE__*/ jsx(SkeletonBox, {
                                                        width: "40px",
                                                        height: "16px"
                                                    })
                                                ]
                                            }),
                                            /*#__PURE__*/ jsx(SkeletonBox, {
                                                width: "80%",
                                                height: "14px"
                                            })
                                        ]
                                    })
                                })
                            ]
                        })
                    }),
                    /*#__PURE__*/ jsx(Grid.Item, {
                        m: 7,
                        xs: 12,
                        flex: 1,
                        children: /*#__PURE__*/ jsxs(Flex, {
                            direction: "column",
                            height: "100%",
                            alignItems: "stretch",
                            flex: 1,
                            padding: 4,
                            gap: 2,
                            children: [
                                /*#__PURE__*/ jsxs(Box, {
                                    children: [
                                        /*#__PURE__*/ jsx(SkeletonBox, {
                                            width: "60px",
                                            height: "16px",
                                            marginBottom: 1
                                        }),
                                        /*#__PURE__*/ jsx(SkeletonBox, {
                                            height: "32px"
                                        })
                                    ]
                                }),
                                /*#__PURE__*/ jsxs(Box, {
                                    children: [
                                        /*#__PURE__*/ jsx(SkeletonBox, {
                                            width: "100px",
                                            height: "16px",
                                            marginBottom: 1
                                        }),
                                        /*#__PURE__*/ jsx(SkeletonBox, {
                                            height: "32px"
                                        }),
                                        /*#__PURE__*/ jsx(Box, {
                                            marginTop: 1,
                                            children: /*#__PURE__*/ jsx(SkeletonBox, {
                                                width: "70%",
                                                height: "12px"
                                            })
                                        })
                                    ]
                                })
                            ]
                        })
                    })
                ]
            })
        }, index));
};

export { AIAssetCard, AIAssetCardSkeletons };
//# sourceMappingURL=AIAssetCard.mjs.map
