import { jsxs, jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { useAPIErrorHandler, useNotification, useRBAC, useQueryParams, useAdminUsers } from '@strapi/admin/strapi-admin';
import { unstable_useDocument } from '@strapi/content-manager/strapi-admin';
import { Field, VisuallyHidden, Combobox, ComboboxOption } from '@strapi/design-system';
import { useIntl } from 'react-intl';
import { useParams } from 'react-router-dom';
import { useTypedSelector } from '../../../../../modules/hooks.mjs';
import { useUpdateAssigneeMutation } from '../../../../../services/content-manager.mjs';
import { buildValidParams } from '../../../../../utils/api.mjs';
import { getDisplayName } from '../../../../../utils/users.mjs';
import { ASSIGNEE_ATTRIBUTE_NAME } from './constants.mjs';

const AssigneeSelect = ({ isCompact })=>{
    const { collectionType = '', id, slug: model = '' } = useParams();
    const permissions = useTypedSelector((state)=>state.admin_app.permissions);
    const { formatMessage } = useIntl();
    const { _unstableFormatAPIError: formatAPIError } = useAPIErrorHandler();
    const { toggleNotification } = useNotification();
    const { allowedActions: { canRead }, isLoading: isLoadingPermissions } = useRBAC(permissions.settings?.users);
    const [{ query }] = useQueryParams();
    const params = React.useMemo(()=>buildValidParams(query), [
        query
    ]);
    const { data, isLoading: isLoadingUsers, isError } = useAdminUsers(undefined, {
        skip: isLoadingPermissions || !canRead
    });
    const { document } = unstable_useDocument({
        collectionType,
        model,
        documentId: id
    }, {
        skip: !id && collectionType !== 'single-types'
    });
    const users = data?.users || [];
    const currentAssignee = document ? document[ASSIGNEE_ATTRIBUTE_NAME] : null;
    const [updateAssignee, { error, isLoading: isMutating }] = useUpdateAssigneeMutation();
    if (!collectionType || !model || !document?.documentId) {
        return null;
    }
    const handleChange = async (assigneeId)=>{
        // a simple way to avoid erroneous updates
        if (currentAssignee?.id === assigneeId) {
            return;
        }
        const res = await updateAssignee({
            slug: collectionType,
            model,
            id: document.documentId,
            params,
            data: {
                id: assigneeId ? parseInt(assigneeId, 10) : null
            }
        });
        if ('data' in res) {
            toggleNotification({
                type: 'success',
                message: formatMessage({
                    id: 'content-manager.reviewWorkflows.assignee.notification.saved',
                    defaultMessage: 'Assignee updated'
                })
            });
        }
        if (isCompact && 'error' in res) {
            toggleNotification({
                type: 'danger',
                message: formatAPIError(res.error)
            });
        }
    };
    const isDisabled = !isLoadingPermissions && !isLoadingUsers && users.length === 0 || !document.documentId;
    const isLoading = isLoadingUsers || isLoadingPermissions || isMutating;
    const assigneeLabel = formatMessage({
        id: 'content-manager.reviewWorkflows.assignee.label',
        defaultMessage: 'Assignee'
    });
    const assigneeClearLabel = formatMessage({
        id: 'content-manager.reviewWorkflows.assignee.clear',
        defaultMessage: 'Clear assignee'
    });
    const assigneePlaceholder = formatMessage({
        id: 'content-manager.reviewWorkflows.assignee.placeholder',
        defaultMessage: 'Select…'
    });
    if (isCompact) {
        return /*#__PURE__*/ jsxs(Field.Root, {
            name: ASSIGNEE_ATTRIBUTE_NAME,
            id: ASSIGNEE_ATTRIBUTE_NAME,
            children: [
                /*#__PURE__*/ jsx(VisuallyHidden, {
                    children: /*#__PURE__*/ jsx(Field.Label, {
                        children: assigneeLabel
                    })
                }),
                /*#__PURE__*/ jsx(Combobox, {
                    clearLabel: assigneeClearLabel,
                    disabled: isDisabled,
                    value: currentAssignee ? currentAssignee.id.toString() : null,
                    onChange: handleChange,
                    onClear: ()=>handleChange(null),
                    placeholder: assigneePlaceholder,
                    loading: isLoading || isLoadingPermissions || isMutating,
                    size: "S",
                    children: users.map((user)=>{
                        return /*#__PURE__*/ jsx(ComboboxOption, {
                            value: user.id.toString(),
                            textValue: getDisplayName(user),
                            children: getDisplayName(user)
                        }, user.id);
                    })
                })
            ]
        });
    }
    return /*#__PURE__*/ jsxs(Field.Root, {
        name: ASSIGNEE_ATTRIBUTE_NAME,
        id: ASSIGNEE_ATTRIBUTE_NAME,
        error: (isError && canRead && formatMessage({
            id: 'content-manager.reviewWorkflows.assignee.error',
            defaultMessage: 'An error occurred while fetching users'
        }) || error && formatAPIError(error)) ?? undefined,
        children: [
            /*#__PURE__*/ jsx(Field.Label, {
                children: assigneeLabel
            }),
            /*#__PURE__*/ jsx(Combobox, {
                clearLabel: assigneeClearLabel,
                disabled: !isLoadingPermissions && !isLoading && users.length === 0 || !document.documentId,
                value: currentAssignee ? currentAssignee.id.toString() : null,
                onChange: handleChange,
                onClear: ()=>handleChange(null),
                placeholder: assigneePlaceholder,
                loading: isLoading || isLoadingPermissions || isMutating,
                children: users.map((user)=>{
                    return /*#__PURE__*/ jsx(ComboboxOption, {
                        value: user.id.toString(),
                        textValue: getDisplayName(user),
                        children: getDisplayName(user)
                    }, user.id);
                })
            }),
            /*#__PURE__*/ jsx(Field.Error, {})
        ]
    });
};

export { AssigneeSelect };
//# sourceMappingURL=AssigneeSelect.mjs.map
