import { jsx, jsxs } from 'react/jsx-runtime';
import 'react';
import { useNotification, useRBAC, Page, Layouts, useAPIErrorHandler } from '@strapi/admin/strapi-admin';
import { useAIAvailability } from '@strapi/admin/strapi-admin/ee';
import { Flex, Box, Typography, Field, Toggle, EmptyStateLayout, Tooltip } from '@strapi/design-system';
import { Sparkle } from '@strapi/icons';
import { EmptyDocuments } from '@strapi/icons/symbols';
import { useIntl } from 'react-intl';
import { CreateLocale } from '../components/CreateLocale.mjs';
import { LocaleTable } from '../components/LocaleTable.mjs';
import { PERMISSIONS } from '../constants.mjs';
import { useGetLocalesQuery } from '../services/locales.mjs';
import { useGetSettingsQuery, useUpdatei18nSettingsMutation } from '../services/settings.mjs';
import { getTranslation } from '../utils/getTranslation.mjs';

const SettingsErrrorTooltip = ({ children, error })=>{
    const { _unstableFormatAPIError: formatAPIError } = useAPIErrorHandler();
    if (error) {
        return /*#__PURE__*/ jsx(Tooltip, {
            label: formatAPIError(error),
            style: {
                maxWidth: '200px'
            },
            children: children
        });
    }
    return children;
};
const SettingsPage = ()=>{
    const { formatMessage } = useIntl();
    const { toggleNotification } = useNotification();
    const { data: locales, isLoading: isLoadingLocales, error } = useGetLocalesQuery();
    const { isLoading: isLoadingRBAC, allowedActions: { canUpdate, canCreate, canDelete } } = useRBAC(PERMISSIONS);
    const isAIAvailable = useAIAvailability();
    // Settings state management
    const { data: settings, isLoading: isLoadingSettings, error: settingsError } = useGetSettingsQuery();
    const [updateSettings] = useUpdatei18nSettingsMutation();
    const handleToggleChange = async (checked)=>{
        try {
            await updateSettings({
                aiLocalizations: checked
            }).unwrap();
            toggleNotification({
                type: 'success',
                message: formatMessage({
                    id: 'notification.form.success.fields'
                })
            });
        } catch (err) {
            console.error(err);
            toggleNotification({
                type: 'danger',
                message: formatMessage({
                    id: 'notification.error',
                    defaultMessage: 'An error occurred'
                })
            });
        }
    };
    const isLoading = isLoadingLocales || isLoadingRBAC || isLoadingSettings;
    if (isLoading) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    if (error || !Array.isArray(locales)) {
        return /*#__PURE__*/ jsx(Page.Error, {});
    }
    return /*#__PURE__*/ jsxs(Page.Main, {
        tabIndex: -1,
        children: [
            /*#__PURE__*/ jsx(Layouts.Header, {
                primaryAction: /*#__PURE__*/ jsx(CreateLocale, {
                    disabled: !canCreate
                }),
                title: formatMessage({
                    id: getTranslation('plugin.name'),
                    defaultMessage: 'Internationalization'
                }),
                subtitle: formatMessage({
                    id: getTranslation('Settings.list.description'),
                    defaultMessage: 'Configure the settings'
                })
            }),
            /*#__PURE__*/ jsxs(Layouts.Content, {
                children: [
                    isAIAvailable && /*#__PURE__*/ jsxs(Flex, {
                        background: "neutral0",
                        padding: 6,
                        marginBottom: 6,
                        shadow: "filterShadow",
                        hasRadius: true,
                        children: [
                            /*#__PURE__*/ jsxs(Flex, {
                                direction: "column",
                                alignItems: "stretch",
                                gap: 1,
                                flex: 1,
                                children: [
                                    /*#__PURE__*/ jsxs(Flex, {
                                        gap: 1,
                                        children: [
                                            /*#__PURE__*/ jsx(Box, {
                                                color: "alternative700",
                                                children: /*#__PURE__*/ jsx(Sparkle, {})
                                            }),
                                            /*#__PURE__*/ jsx(Typography, {
                                                variant: "delta",
                                                tag: "h2",
                                                children: formatMessage({
                                                    id: getTranslation('Settings.aiLocalizations.label'),
                                                    defaultMessage: 'AI Translations'
                                                })
                                            })
                                        ]
                                    }),
                                    /*#__PURE__*/ jsx(Typography, {
                                        variant: "pi",
                                        textColor: "neutral600",
                                        fontSize: "14px",
                                        children: formatMessage({
                                            id: getTranslation('Settings.aiLocalizations.description'),
                                            defaultMessage: 'Everytime you save in the Content Manager, our AI will use your default locale to translate all other locales automatically.'
                                        })
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsx(Field.Root, {
                                name: "aiLocalizations",
                                minWidth: "200px",
                                children: /*#__PURE__*/ jsx(SettingsErrrorTooltip, {
                                    error: settingsError,
                                    children: /*#__PURE__*/ jsx(Toggle, {
                                        disabled: Boolean(settingsError),
                                        checked: settings?.data?.aiLocalizations ?? false,
                                        offLabel: formatMessage({
                                            id: 'app.components.ToggleCheckbox.disabled-label',
                                            defaultMessage: 'Disabled'
                                        }),
                                        onLabel: formatMessage({
                                            id: 'app.components.ToggleCheckbox.enabled-label',
                                            defaultMessage: 'Enabled'
                                        }),
                                        onChange: (e)=>handleToggleChange(e.target.checked)
                                    })
                                })
                            })
                        ]
                    }),
                    locales.length > 0 ? /*#__PURE__*/ jsx(LocaleTable, {
                        locales: locales,
                        canDelete: canDelete,
                        canUpdate: canUpdate
                    }) : /*#__PURE__*/ jsx(EmptyStateLayout, {
                        icon: /*#__PURE__*/ jsx(EmptyDocuments, {
                            width: undefined,
                            height: undefined
                        }),
                        content: formatMessage({
                            id: getTranslation('Settings.list.empty.title'),
                            defaultMessage: 'There are no locales'
                        }),
                        action: /*#__PURE__*/ jsx(CreateLocale, {
                            disabled: !canCreate,
                            variant: "secondary"
                        })
                    })
                ]
            })
        ]
    });
};
const ProtectedSettingsPage = ()=>{
    return /*#__PURE__*/ jsx(Page.Protect, {
        permissions: PERMISSIONS.read,
        children: /*#__PURE__*/ jsx(SettingsPage, {})
    });
};

export { ProtectedSettingsPage, SettingsPage };
//# sourceMappingURL=SettingsPage.mjs.map
