'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var query = require('@reduxjs/toolkit/query');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var ee = require('@strapi/admin/strapi-admin/ee');
var strapiAdmin$1 = require('@strapi/content-manager/strapi-admin');
var designSystem = require('@strapi/design-system');
var icons = require('@strapi/icons');
var reactIntl = require('react-intl');
var reactRouterDom = require('react-router-dom');
var styledComponents = require('styled-components');
var useAILocalizationJobsPolling = require('../hooks/useAILocalizationJobsPolling.js');
var useI18n = require('../hooks/useI18n.js');
var aiLocalizationJobs = require('../services/aiLocalizationJobs.js');
var locales = require('../services/locales.js');
var relations = require('../services/relations.js');
var settings = require('../services/settings.js');
var clean = require('../utils/clean.js');
var getTranslation = require('../utils/getTranslation.js');
var strings = require('../utils/strings.js');
var BulkLocaleActionModal = require('./BulkLocaleActionModal.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

const statusVariants = {
    draft: 'secondary',
    published: 'success',
    modified: 'alternative'
};
const LocaleOption = ({ isDraftAndPublishEnabled, locale, status, entryExists })=>{
    const { formatMessage } = reactIntl.useIntl();
    if (!entryExists) {
        return formatMessage({
            id: getTranslation.getTranslation('CMEditViewLocalePicker.locale.create'),
            defaultMessage: 'Create <bold>{locale}</bold> locale'
        }, {
            bold: (locale)=>/*#__PURE__*/ jsxRuntime.jsx("b", {
                    children: locale
                }),
            locale: locale.name
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
        width: "100%",
        gap: 1,
        justifyContent: "space-between",
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                children: locale.name
            }),
            isDraftAndPublishEnabled ? /*#__PURE__*/ jsxRuntime.jsx(designSystem.Status, {
                display: "flex",
                paddingLeft: "6px",
                paddingRight: "6px",
                paddingTop: "2px",
                paddingBottom: "2px",
                size: "S",
                variant: statusVariants[status],
                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                    tag: "span",
                    variant: "pi",
                    fontWeight: "bold",
                    children: strings.capitalize(status)
                })
            }) : null
        ]
    });
};
const LocaleOptionStartIcon = ({ entryWithLocaleExists, translationStatus, index })=>{
    const isAiAvailable = ee.useAIAvailability();
    if (!entryWithLocaleExists) {
        return /*#__PURE__*/ jsxRuntime.jsx(icons.Plus, {});
    }
    if (isAiAvailable && index !== 0 && translationStatus === 'failed') {
        return /*#__PURE__*/ jsxRuntime.jsx(icons.WarningCircle, {
            fill: "warning600"
        });
    }
    return null;
};
const LocalePickerAction = ({ document, meta, model, collectionType, documentId })=>{
    const { formatMessage } = reactIntl.useIntl();
    const [{ query }, setQuery] = strapiAdmin.useQueryParams();
    const { hasI18n, canCreate, canRead } = useI18n.useI18n();
    const { data: locales$1 = [] } = locales.useGetLocalesQuery();
    const currentDesiredLocale = query.plugins?.i18n?.locale;
    const { schema } = strapiAdmin$1.unstable_useDocument({
        model,
        collectionType,
        documentId,
        params: {
            locale: currentDesiredLocale
        }
    });
    const { data: jobData } = aiLocalizationJobs.useGetAILocalizationJobsByDocumentQuery({
        documentId: documentId,
        model: model,
        collectionType: collectionType
    });
    const { data: settings$1 } = settings.useGetSettingsQuery();
    const isAiAvailable = ee.useAIAvailability();
    const setValues = strapiAdmin.useForm('LocalePickerAction', (state)=>state.setValues);
    const handleSelect = React__namespace.useCallback((value)=>{
        setQuery({
            plugins: {
                ...query.plugins,
                i18n: {
                    locale: value
                }
            }
        }, 'push', true);
    }, [
        query.plugins,
        setQuery
    ]);
    const nonTranslatedFields = React__namespace.useMemo(()=>{
        if (!schema?.attributes) return [];
        return Object.keys(schema.attributes).filter((field)=>{
            const attribute = schema.attributes[field];
            return attribute?.pluginOptions?.i18n?.localized === false;
        });
    }, [
        schema?.attributes
    ]);
    const sourceLocaleData = React__namespace.useMemo(()=>{
        if (!Array.isArray(locales$1) || !meta?.availableLocales) return null;
        const defaultLocale = locales$1.find((locale)=>locale.isDefault);
        const existingLocales = meta.availableLocales.map((loc)=>loc.locale);
        const sourceLocaleCode = defaultLocale && existingLocales.includes(defaultLocale.code) && defaultLocale.code !== currentDesiredLocale ? defaultLocale.code : existingLocales.find((locale)=>locale !== currentDesiredLocale);
        if (!sourceLocaleCode) return null;
        // Find the document data from availableLocales (now includes non-translatable fields)
        const sourceLocaleDoc = meta.availableLocales.find((loc)=>loc.locale === sourceLocaleCode);
        return sourceLocaleDoc ? {
            locale: sourceLocaleCode,
            data: sourceLocaleDoc
        } : null;
    }, [
        locales$1,
        meta?.availableLocales,
        currentDesiredLocale
    ]);
    /**
   * Prefilling form with non-translatable fields from already existing locale
   */ React__namespace.useEffect(()=>{
        // Only run when creating a new locale (no document ID yet) and when we have non-translatable fields
        if (!document?.id && nonTranslatedFields.length > 0 && sourceLocaleData?.data) {
            const dataToSet = nonTranslatedFields.reduce((acc, field)=>{
                acc[field] = sourceLocaleData.data[field];
                return acc;
            }, {});
            if (Object.keys(dataToSet).length > 0) {
                setValues(dataToSet);
            }
        }
    }, [
        document?.id,
        nonTranslatedFields,
        sourceLocaleData?.data,
        setValues
    ]);
    React__namespace.useEffect(()=>{
        if (!Array.isArray(locales$1) || !hasI18n) {
            return;
        }
        /**
     * Handle the case where the current locale query param doesn't exist
     * in the list of available locales, so we redirect to the default locale.
     */ const doesLocaleExist = locales$1.find((loc)=>loc.code === currentDesiredLocale);
        const defaultLocale = locales$1.find((locale)=>locale.isDefault);
        if (!doesLocaleExist && defaultLocale?.code) {
            handleSelect(defaultLocale.code);
        }
    }, [
        handleSelect,
        hasI18n,
        locales$1,
        currentDesiredLocale
    ]);
    const currentLocale = Array.isArray(locales$1) ? locales$1.find((locale)=>locale.code === currentDesiredLocale) : undefined;
    // Use meta.availableLocales instead of document.localizations
    // meta.availableLocales contains all locales for the document, even when creating new locales
    const availableLocales = meta?.availableLocales ?? [];
    const documentLocalizations = document?.localizations ?? [];
    // Prefer meta.availableLocales as it's more reliable, fallback to document.localizations
    const allLocalizations = availableLocales.length > 0 ? availableLocales : documentLocalizations;
    const allCurrentLocales = [
        {
            status: getDocumentStatus(document, meta),
            locale: currentLocale?.code
        },
        ...allLocalizations
    ];
    if (!hasI18n || !Array.isArray(locales$1) || locales$1.length === 0) {
        return null;
    }
    const displayedLocales = locales$1.filter((locale)=>{
        /**
     * If you can read we allow you to see the locale exists
     * otherwise the locale is hidden.
     */ return canRead.includes(locale.code);
    });
    const localesSortingDefaultFirst = displayedLocales.sort((a, b)=>a.isDefault ? -1 : b.isDefault ? 1 : 0);
    return {
        label: formatMessage({
            id: getTranslation.getTranslation('Settings.locales.modal.locales.label'),
            defaultMessage: 'Locales'
        }),
        options: localesSortingDefaultFirst.map((locale, index)=>{
            const entryWithLocaleExists = allCurrentLocales.some((doc)=>doc.locale === locale.code);
            const currentLocaleDoc = allCurrentLocales.find((doc)=>'locale' in doc ? doc.locale === locale.code : false);
            const permissionsToCheck = currentLocaleDoc ? canRead : canCreate;
            if (isAiAvailable && settings$1?.data?.aiLocalizations) {
                return {
                    _render: ()=>/*#__PURE__*/ jsxRuntime.jsxs(React__namespace.Fragment, {
                            children: [
                                /*#__PURE__*/ jsxRuntime.jsx(designSystem.SingleSelectOption, {
                                    disabled: !permissionsToCheck.includes(locale.code),
                                    startIcon: /*#__PURE__*/ jsxRuntime.jsx(LocaleOptionStartIcon, {
                                        entryWithLocaleExists: entryWithLocaleExists,
                                        translationStatus: jobData?.data?.status,
                                        index: index
                                    }),
                                    value: locale.code,
                                    children: /*#__PURE__*/ jsxRuntime.jsx(LocaleOption, {
                                        isDraftAndPublishEnabled: !!schema?.options?.draftAndPublish,
                                        locale: locale,
                                        status: currentLocaleDoc?.status,
                                        entryExists: entryWithLocaleExists
                                    })
                                }, locale.code),
                                localesSortingDefaultFirst.length > 1 && index === 0 && /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                    paddingRight: 4,
                                    paddingLeft: 4,
                                    paddingTop: 2,
                                    paddingBottom: 2,
                                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                        variant: "sigma",
                                        children: formatMessage({
                                            id: getTranslation.getTranslation('CMEditViewLocalePicker.locale.ai-translations'),
                                            defaultMessage: 'AI Translations'
                                        })
                                    })
                                })
                            ]
                        }, index)
                };
            }
            return {
                disabled: !permissionsToCheck.includes(locale.code),
                value: locale.code,
                label: /*#__PURE__*/ jsxRuntime.jsx(LocaleOption, {
                    isDraftAndPublishEnabled: !!schema?.options?.draftAndPublish,
                    locale: locale,
                    status: currentLocaleDoc?.status,
                    entryExists: entryWithLocaleExists,
                    translationStatus: jobData?.data?.status
                }),
                startIcon: /*#__PURE__*/ jsxRuntime.jsx(LocaleOptionStartIcon, {
                    entryWithLocaleExists: entryWithLocaleExists
                })
            };
        }),
        customizeContent: ()=>currentLocale?.name,
        onSelect: handleSelect,
        value: currentLocale
    };
};
const getDocumentStatus = (document, meta)=>{
    const docStatus = document?.status;
    const statuses = meta?.availableStatus ?? [];
    /**
   * Creating an entry
   */ if (!docStatus) {
        return 'draft';
    }
    /**
   * We're viewing a draft, but the document could have a published version
   */ if (docStatus === 'draft' && statuses.find((doc)=>doc.publishedAt !== null)) {
        return 'published';
    }
    return docStatus;
};
/* -------------------------------------------------------------------------------------------------
 * AISettingsStatusAction
 * -----------------------------------------------------------------------------------------------*/ const AITranslationStatusIcon = styledComponents.styled(designSystem.Status)`
  display: flex;
  gap: ${({ theme })=>theme.spaces[1]};
  justify-content: center;
  align-items: center;
  height: 100%;

  // Disabled state
  ${({ $isAISettingEnabled, theme })=>!$isAISettingEnabled && `
    background-color: ${theme.colors.neutral150};
  `}

  svg {
    ${({ $isAISettingEnabled, theme })=>!$isAISettingEnabled && `
        fill: ${theme.colors.neutral300};
      `}
  }
`;
const SpinningLoader = styledComponents.styled(icons.Loader)`
  @keyframes spin {
    from {
      transform: rotate(0deg);
    }
    to {
      transform: rotate(360deg);
    }
  }

  animation: spin 2s linear infinite;
`;
const AITranslationStatusAction = ({ documentId, model, collectionType })=>{
    const { formatMessage } = reactIntl.useIntl();
    const isAIAvailable = ee.useAIAvailability();
    const { data: settings$1 } = settings.useGetSettingsQuery();
    const isAISettingEnabled = settings$1?.data?.aiLocalizations;
    const { hasI18n } = useI18n.useI18n();
    // Poll for AI localizations jobs when AI is enabled and we have a documentId
    const { status } = useAILocalizationJobsPolling.useAILocalizationJobsPolling({
        documentId,
        model,
        collectionType
    });
    const statusVariant = (()=>{
        if (status === 'failed') {
            return 'warning';
        }
        if (isAISettingEnabled) {
            return 'alternative';
        }
        return 'neutral';
    })();
    // Do not display this action when i18n is not available
    if (!hasI18n) {
        return null;
    }
    // Do not display this action when AI is not available
    if (!isAIAvailable) {
        return null;
    }
    return {
        _status: {
            message: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                height: "100%",
                "aria-label": formatMessage({
                    id: getTranslation.getTranslation('CMEditViewAITranslation.status-aria-label'),
                    defaultMessage: 'AI Translation Status'
                }),
                children: /*#__PURE__*/ jsxRuntime.jsx(AITranslationStatusIcon, {
                    $isAISettingEnabled: Boolean(isAISettingEnabled),
                    variant: statusVariant,
                    size: "S",
                    children: status === 'processing' ? /*#__PURE__*/ jsxRuntime.jsx(SpinningLoader, {}) : /*#__PURE__*/ jsxRuntime.jsx(icons.Sparkle, {})
                })
            }),
            tooltip: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                direction: "column",
                padding: 4,
                alignItems: "flex-start",
                width: "25rem",
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        variant: "pi",
                        fontWeight: "600",
                        children: formatMessage({
                            id: getTranslation.getTranslation('CMEditViewAITranslation.status-title'),
                            defaultMessage: '{enabled, select, true {AI translation enabled} false {AI translation disabled} other {AI translation disabled}}'
                        }, {
                            enabled: isAISettingEnabled
                        })
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        variant: "pi",
                        paddingTop: 1,
                        paddingBottom: 3,
                        children: formatMessage({
                            id: getTranslation.getTranslation('CMEditViewAITranslation.status-description'),
                            defaultMessage: 'Our AI translates content in all locales each time you save a modification in the default locale.'
                        })
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Link, {
                        fontSize: "inherit",
                        tag: reactRouterDom.NavLink,
                        to: "/settings/internationalization",
                        style: {
                            alignSelf: 'flex-end'
                        },
                        children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                            variant: "pi",
                            textAlign: "right",
                            children: formatMessage({
                                id: getTranslation.getTranslation('CMEditViewAITranslation.settings-link'),
                                defaultMessage: '{enabled, select, true {Disable it in settings} false {Enable it in settings} other {Enable it in settings}}'
                            }, {
                                enabled: isAISettingEnabled
                            })
                        })
                    })
                ]
            })
        }
    };
};
/* -------------------------------------------------------------------------------------------------
 * FillFromAnotherLocaleAction
 * -----------------------------------------------------------------------------------------------*/ const FillFromAnotherLocaleAction = ({ documentId, meta, model, collectionType })=>{
    const { formatMessage } = reactIntl.useIntl();
    const [{ query }] = strapiAdmin.useQueryParams();
    const { hasI18n } = useI18n.useI18n();
    const currentDesiredLocale = query.plugins?.i18n?.locale;
    const [localeSelected, setLocaleSelected] = React__namespace.useState(null);
    const setValues = strapiAdmin.useForm('FillFromAnotherLocale', (state)=>state.setValues);
    const { getDocument } = strapiAdmin$1.unstable_useDocumentActions();
    const { schema, components } = strapiAdmin$1.unstable_useDocument({
        model,
        documentId,
        collectionType,
        params: {
            locale: currentDesiredLocale
        }
    });
    const { data: locales$1 = [] } = locales.useGetLocalesQuery();
    const isAIAvailable = ee.useAIAvailability();
    const { data: settings$1 } = settings.useGetSettingsQuery();
    const isAISettingEnabled = settings$1?.data?.aiLocalizations;
    const availableLocales = Array.isArray(locales$1) ? locales$1.filter((locale)=>meta?.availableLocales.some((l)=>l.locale === locale.code)) : [];
    const fillFromLocale = (onClose)=>async ()=>{
            const response = await getDocument({
                collectionType,
                model,
                documentId,
                params: {
                    locale: localeSelected
                }
            });
            if (!response || !schema) {
                return;
            }
            const { data } = response;
            const cleanedData = clean.cleanData(data, schema, components);
            setValues(cleanedData);
            onClose();
        };
    if (!hasI18n) {
        return null;
    }
    // Do not display this action when AI is available and AI translations are enabled
    if (isAIAvailable && isAISettingEnabled) {
        return null;
    }
    return {
        type: 'icon',
        icon: /*#__PURE__*/ jsxRuntime.jsx(icons.Earth, {}),
        disabled: availableLocales.length === 0,
        label: formatMessage({
            id: getTranslation.getTranslation('CMEditViewCopyLocale.copy-text'),
            defaultMessage: 'Fill in from another locale'
        }),
        dialog: {
            type: 'dialog',
            title: formatMessage({
                id: getTranslation.getTranslation('CMEditViewCopyLocale.dialog.title'),
                defaultMessage: 'Confirmation'
            }),
            content: ({ onClose })=>/*#__PURE__*/ jsxRuntime.jsxs(jsxRuntime.Fragment, {
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Dialog.Body, {
                            children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                                direction: "column",
                                gap: 3,
                                children: [
                                    /*#__PURE__*/ jsxRuntime.jsx(icons.WarningCircle, {
                                        width: "24px",
                                        height: "24px",
                                        fill: "danger600"
                                    }),
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                        textAlign: "center",
                                        children: formatMessage({
                                            id: getTranslation.getTranslation('CMEditViewCopyLocale.dialog.body'),
                                            defaultMessage: 'Your current content will be erased and filled by the content of the selected locale:'
                                        })
                                    }),
                                    /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Field.Root, {
                                        width: "100%",
                                        children: [
                                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Label, {
                                                children: formatMessage({
                                                    id: getTranslation.getTranslation('CMEditViewCopyLocale.dialog.field.label'),
                                                    defaultMessage: 'Locale'
                                                })
                                            }),
                                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.SingleSelect, {
                                                value: localeSelected,
                                                placeholder: formatMessage({
                                                    id: getTranslation.getTranslation('CMEditViewCopyLocale.dialog.field.placeholder'),
                                                    defaultMessage: 'Select one locale...'
                                                }),
                                                // @ts-expect-error – the DS will handle numbers, but we're not allowing the API.
                                                onChange: (value)=>setLocaleSelected(value),
                                                children: availableLocales.map((locale)=>/*#__PURE__*/ jsxRuntime.jsx(designSystem.SingleSelectOption, {
                                                        value: locale.code,
                                                        children: locale.name
                                                    }, locale.code))
                                            })
                                        ]
                                    })
                                ]
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Dialog.Footer, {
                            children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                                gap: 2,
                                width: "100%",
                                children: [
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                        flex: "auto",
                                        variant: "tertiary",
                                        onClick: onClose,
                                        children: formatMessage({
                                            id: getTranslation.getTranslation('CMEditViewCopyLocale.cancel-text'),
                                            defaultMessage: 'No, cancel'
                                        })
                                    }),
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                        flex: "auto",
                                        variant: "success",
                                        onClick: fillFromLocale(onClose),
                                        children: formatMessage({
                                            id: getTranslation.getTranslation('CMEditViewCopyLocale.submit-text'),
                                            defaultMessage: 'Yes, fill in'
                                        })
                                    })
                                ]
                            })
                        })
                    ]
                })
        }
    };
};
/* -------------------------------------------------------------------------------------------------
 * DeleteLocaleAction
 * -----------------------------------------------------------------------------------------------*/ const DeleteLocaleAction = ({ document, documentId, model, collectionType })=>{
    const { formatMessage } = reactIntl.useIntl();
    const navigate = reactRouterDom.useNavigate();
    const { toggleNotification } = strapiAdmin.useNotification();
    const { delete: deleteAction, isLoading } = strapiAdmin$1.unstable_useDocumentActions();
    const { hasI18n, canDelete } = useI18n.useI18n();
    // Get the current locale object, using the URL instead of document so it works while creating
    const [{ query }] = strapiAdmin.useQueryParams();
    const { data: locales$1 = [] } = locales.useGetLocalesQuery();
    const currentDesiredLocale = query.plugins?.i18n?.locale;
    const locale = !('error' in locales$1) && locales$1.find((loc)=>loc.code === currentDesiredLocale);
    if (!hasI18n) {
        return null;
    }
    return {
        disabled: document?.locale && !canDelete.includes(document.locale) || !document || !document.id,
        position: [
            'header',
            'table-row'
        ],
        label: formatMessage({
            id: getTranslation.getTranslation('actions.delete.label'),
            defaultMessage: 'Delete entry ({locale})'
        }, {
            locale: locale && locale.name
        }),
        icon: /*#__PURE__*/ jsxRuntime.jsx(StyledTrash, {}),
        variant: 'danger',
        dialog: {
            type: 'dialog',
            title: formatMessage({
                id: getTranslation.getTranslation('actions.delete.dialog.title'),
                defaultMessage: 'Confirmation'
            }),
            content: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                direction: "column",
                gap: 2,
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(icons.WarningCircle, {
                        width: "24px",
                        height: "24px",
                        fill: "danger600"
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        tag: "p",
                        variant: "omega",
                        textAlign: "center",
                        children: formatMessage({
                            id: getTranslation.getTranslation('actions.delete.dialog.body'),
                            defaultMessage: 'Are you sure?'
                        })
                    })
                ]
            }),
            loading: isLoading,
            onConfirm: async ()=>{
                const unableToDelete = // We are unable to delete a collection type without a document ID
                // & unable to delete generally if there is no document locale
                collectionType !== 'single-types' && !documentId || !document?.locale;
                if (unableToDelete) {
                    console.error("You're trying to delete a document without an id or locale, this is likely a bug with Strapi. Please open an issue.");
                    toggleNotification({
                        message: formatMessage({
                            id: getTranslation.getTranslation('actions.delete.error'),
                            defaultMessage: 'An error occurred while trying to delete the document locale.'
                        }),
                        type: 'danger'
                    });
                    return;
                }
                const res = await deleteAction({
                    documentId,
                    model,
                    collectionType,
                    params: {
                        locale: document.locale
                    }
                });
                if (!('error' in res)) {
                    navigate({
                        pathname: `../${collectionType}/${model}`
                    }, {
                        replace: true
                    });
                }
            }
        }
    };
};
/* -------------------------------------------------------------------------------------------------
 * BulkLocaleAction
 *
 * This component is used to handle bulk publish and unpublish actions on locales.
 * -----------------------------------------------------------------------------------------------*/ const BulkLocaleAction = ({ document, documentId, model, collectionType, action, meta })=>{
    const locale = document?.locale ?? null;
    const [{ query: query$1 }] = strapiAdmin.useQueryParams();
    const params = React__namespace.useMemo(()=>strapiAdmin$1.buildValidParams(query$1), [
        query$1
    ]);
    const isOnPublishedTab = query$1.status === 'published';
    const { formatMessage } = reactIntl.useIntl();
    const { hasI18n, canPublish } = useI18n.useI18n();
    const { toggleNotification } = strapiAdmin.useNotification();
    const { _unstableFormatAPIError: formatAPIError } = strapiAdmin.useAPIErrorHandler();
    const [selectedRows, setSelectedRows] = React__namespace.useState([]);
    const [isDraftRelationConfirmationOpen, setIsDraftRelationConfirmationOpen] = React__namespace.useState(false);
    const { publishMany: publishManyAction, unpublishMany: unpublishManyAction } = strapiAdmin$1.unstable_useDocumentActions();
    const { schema, validate } = strapiAdmin$1.unstable_useDocument({
        model,
        collectionType,
        documentId,
        params: {
            locale
        }
    }, {
        // No need to fetch the document, the data is already available in the `document` prop
        skip: true
    });
    const { data: localesMetadata = [] } = locales.useGetLocalesQuery(hasI18n ? undefined : query.skipToken);
    const headers = [
        {
            label: formatMessage({
                id: 'global.name',
                defaultMessage: 'Name'
            }),
            name: 'name'
        },
        {
            label: formatMessage({
                id: getTranslation.getTranslation('CMEditViewBulkLocale.status'),
                defaultMessage: 'Status'
            }),
            name: 'status'
        },
        {
            label: formatMessage({
                id: getTranslation.getTranslation('CMEditViewBulkLocale.publication-status'),
                defaultMessage: 'Publication Status'
            }),
            name: 'publication-status'
        }
    ];
    // Extract the rows for the bulk locale publish modal and any validation
    // errors per locale
    const [rows, validationErrors] = React__namespace.useMemo(()=>{
        if (!document) {
            return [
                [],
                {}
            ];
        }
        const metaLocalizations = (meta?.availableLocales ?? []).map((locale)=>({
                locale: locale.locale,
                status: locale.status ?? 'draft'
            }));
        const documentLocalizations = (document.localizations ?? []).map((doc)=>({
                locale: doc.locale ?? undefined,
                status: doc.status ?? 'draft'
            }));
        const localesMap = new Map();
        metaLocalizations.forEach(({ locale, status })=>{
            if (locale) {
                localesMap.set(locale, {
                    locale,
                    status
                });
            }
        });
        documentLocalizations.forEach(({ locale, status })=>{
            if (locale) {
                localesMap.set(locale, {
                    locale,
                    status
                });
            }
        });
        // Build the rows for the bulk locale publish modal by combining the current
        // document with all the available locales from the document meta
        const locales = [];
        if (document?.locale) {
            locales.push({
                locale: document.locale,
                status: document.status ?? 'draft'
            });
        }
        locales.push(...Array.from(localesMap.entries()).filter(([locale])=>locale !== document?.locale).map(([, value])=>value));
        if (locales.length === 0 && document?.locale) {
            locales.push({
                locale: document.locale,
                status: document.status ?? 'draft'
            });
        }
        // Build the validation errors for each locale.
        const allDocuments = [
            document,
            ...document.localizations ?? []
        ];
        const errors = allDocuments.reduce((errs, document)=>{
            if (!document) {
                return errs;
            }
            // Validate each locale entry via the useDocument validate function and store any errors in a dictionary
            const validation = validate(document);
            if (validation !== null) {
                errs[document.locale] = validation;
            }
            return errs;
        }, {});
        return [
            locales,
            errors
        ];
    }, [
        document,
        meta?.availableLocales,
        validate
    ]);
    const isBulkPublish = action === 'bulk-publish';
    const localesForAction = selectedRows.reduce((acc, selectedRow)=>{
        const isValidLocale = // Validation errors are irrelevant if we are trying to unpublish
        !isBulkPublish || !Object.keys(validationErrors).includes(selectedRow.locale);
        const shouldAddLocale = isBulkPublish ? selectedRow.status !== 'published' && isValidLocale : selectedRow.status !== 'draft' && isValidLocale;
        if (shouldAddLocale) {
            acc.push(selectedRow.locale);
        }
        return acc;
    }, []);
    // TODO skipping this for now as there is a bug with the draft relation count that will be worked on separately
    // see https://www.notion.so/strapi/Count-draft-relations-56901b492efb45ab90d42fe975b32bd8?pvs=4
    const enableDraftRelationsCount = false;
    const { data: draftRelationsCount = 0, isLoading: isDraftRelationsLoading, error: isDraftRelationsError } = relations.useGetManyDraftRelationCountQuery({
        model,
        documentIds: [
            documentId
        ],
        locale: localesForAction
    }, {
        skip: !enableDraftRelationsCount
    });
    React__namespace.useEffect(()=>{
        if (isDraftRelationsError) {
            toggleNotification({
                type: 'danger',
                message: formatAPIError(isDraftRelationsError)
            });
        }
    }, [
        isDraftRelationsError,
        toggleNotification,
        formatAPIError
    ]);
    if (!schema?.options?.draftAndPublish) {
        return null;
    }
    if (!hasI18n) {
        return null;
    }
    if (!documentId) {
        return null;
    }
    // This document action can be enabled given that draft and publish and i18n are
    // enabled and we can publish the current locale.
    const publish = async ()=>{
        await publishManyAction({
            model,
            documentIds: [
                documentId
            ],
            params: {
                ...params,
                locale: localesForAction
            }
        });
        setSelectedRows([]);
    };
    const unpublish = async ()=>{
        await unpublishManyAction({
            model,
            documentIds: [
                documentId
            ],
            params: {
                ...params,
                locale: localesForAction
            }
        });
        setSelectedRows([]);
    };
    const handleAction = async ()=>{
        if (draftRelationsCount > 0) {
            setIsDraftRelationConfirmationOpen(true);
        } else if (isBulkPublish) {
            await publish();
        } else {
            await unpublish();
        }
    };
    if (isDraftRelationConfirmationOpen) {
        return {
            label: formatMessage({
                id: 'app.components.ConfirmDialog.title',
                defaultMessage: 'Confirmation'
            }),
            variant: 'danger',
            dialog: {
                onCancel: ()=>{
                    setIsDraftRelationConfirmationOpen(false);
                },
                onConfirm: async ()=>{
                    await publish();
                    setIsDraftRelationConfirmationOpen(false);
                },
                type: 'dialog',
                title: formatMessage({
                    id: getTranslation.getTranslation('actions.publish.dialog.title'),
                    defaultMessage: 'Confirmation'
                }),
                content: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                    direction: "column",
                    alignItems: "center",
                    gap: 2,
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(icons.WarningCircle, {
                            width: "2.4rem",
                            height: "2.4rem",
                            fill: "danger600"
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                            textAlign: "center",
                            children: formatMessage({
                                id: getTranslation.getTranslation('CMEditViewBulkLocale.draft-relation-warning'),
                                defaultMessage: 'Some locales are related to draft entries. Publishing them could leave broken links in your app.'
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                            textAlign: "center",
                            children: formatMessage({
                                id: getTranslation.getTranslation('CMEditViewBulkLocale.continue-confirmation'),
                                defaultMessage: 'Are you sure you want to continue?'
                            })
                        })
                    ]
                })
            }
        };
    }
    const hasPermission = selectedRows.map(({ locale })=>locale).every((locale)=>canPublish.includes(locale));
    return {
        label: formatMessage({
            id: getTranslation.getTranslation(`CMEditViewBulkLocale.${isBulkPublish ? 'publish' : 'unpublish'}-title`),
            defaultMessage: `${isBulkPublish ? 'Publish' : 'Unpublish'} Multiple Locales`
        }),
        variant: isBulkPublish ? 'secondary' : 'danger',
        icon: isBulkPublish ? /*#__PURE__*/ jsxRuntime.jsx(icons.ListPlus, {}) : /*#__PURE__*/ jsxRuntime.jsx(icons.Cross, {}),
        disabled: isOnPublishedTab || canPublish.length === 0,
        position: [
            'panel'
        ],
        dialog: {
            type: 'modal',
            title: formatMessage({
                id: getTranslation.getTranslation(`CMEditViewBulkLocale.${isBulkPublish ? 'publish' : 'unpublish'}-title`),
                defaultMessage: `${isBulkPublish ? 'Publish' : 'Unpublish'} Multiple Locales`
            }),
            content: ()=>{
                return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Table.Root, {
                    headers: headers,
                    rows: rows.map((row)=>({
                            ...row,
                            id: row.locale
                        })),
                    selectedRows: selectedRows,
                    onSelectedRowsChange: (tableSelectedRows)=>setSelectedRows(tableSelectedRows),
                    children: /*#__PURE__*/ jsxRuntime.jsx(BulkLocaleActionModal.BulkLocaleActionModal, {
                        validationErrors: validationErrors,
                        headers: headers,
                        rows: rows,
                        localesMetadata: localesMetadata,
                        action: action ?? 'bulk-publish'
                    })
                });
            },
            footer: ()=>/*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Footer, {
                    justifyContent: "flex-end",
                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                        loading: isDraftRelationsLoading,
                        disabled: !hasPermission || localesForAction.length === 0,
                        variant: "default",
                        onClick: handleAction,
                        children: formatMessage({
                            id: isBulkPublish ? 'app.utils.publish' : 'app.utils.unpublish',
                            defaultMessage: isBulkPublish ? 'Publish' : 'Unpublish'
                        })
                    })
                })
        }
    };
};
/* -------------------------------------------------------------------------------------------------
 * BulkLocalePublishAction
 * -----------------------------------------------------------------------------------------------*/ const BulkLocalePublishAction = (props)=>{
    return BulkLocaleAction({
        action: 'bulk-publish',
        ...props
    });
};
/* -------------------------------------------------------------------------------------------------
 * BulkLocaleUnpublishAction
 * -----------------------------------------------------------------------------------------------*/ const BulkLocaleUnpublishAction = (props)=>{
    return BulkLocaleAction({
        action: 'bulk-unpublish',
        ...props
    });
};
/**
 * Because the icon system is completely broken, we have to do
 * this to remove the fill from the cog.
 */ const StyledTrash = styledComponents.styled(icons.Trash)`
  path {
    fill: currentColor;
  }
`;

exports.AITranslationStatusAction = AITranslationStatusAction;
exports.BulkLocalePublishAction = BulkLocalePublishAction;
exports.BulkLocaleUnpublishAction = BulkLocaleUnpublishAction;
exports.DeleteLocaleAction = DeleteLocaleAction;
exports.FillFromAnotherLocaleAction = FillFromAnotherLocaleAction;
exports.LocalePickerAction = LocalePickerAction;
//# sourceMappingURL=CMHeaderActions.js.map
