'use strict';

var node_path = require('node:path');
var handlebars = require('handlebars');
var fs = require('fs-extra');

// Starts the Plop CLI programmatically
const runCLI = async ()=>{
    const { Plop, run } = await import('plop');
    Plop.prepare({
        configPath: node_path.join(__dirname, 'plopfile.js')
    }, (env)=>{
        Plop.execute(env, [], (env, argv)=>{
            const options = {
                ...env,
                dest: node_path.join(process.cwd(), 'src')
            };
            return run(options, argv, true); // Pass the third argument 'true' for passArgsBeforeDashes
        });
    });
};
const generate = async (generatorName, options, { dir = process.cwd(), plopFile = 'plopfile.js' } = {})=>{
    // Resolve the absolute path to the plopfile (generator definitions)
    const plopfilePath = node_path.join(__dirname, plopFile);
    // Dynamically require the plopfile module.
    // Note: This allows loading either CommonJS or transpiled ESM plopfiles.
    // eslint-disable-next-line @typescript-eslint/no-var-requires
    const plopModule = require(plopfilePath);
    // Internal objects to store registered generators and helpers.
    // These will be populated by the plopfile when it is executed.
    const generators = {};
    const helpers = {};
    // Minimal mock Plop API implementation, exposing only the methods needed by our plopfile.
    // This allows the plopfile to register generators and helpers as it would in a real Plop environment.
    const plopApi = {
        setGenerator (name, config) {
            generators[name] = config;
        },
        setHelper (name, fn) {
            helpers[name] = fn;
        },
        getDestBasePath () {
            return node_path.join(dir, 'src');
        },
        setWelcomeMessage () {}
    };
    // Execute the plopfile, passing in our API.
    // This will populate the `generators` and `helpers` objects.
    // Support both CommonJS and ESM default exports.
    if (typeof plopModule.default === 'function') {
        plopModule.default(plopApi);
    } else {
        plopModule(plopApi);
    }
    const generator = generators[generatorName];
    if (!generator) {
        throw new Error(`Generator "${generatorName}" not found`);
    }
    registerHandlebarsHelpers(helpers);
    const actions = typeof generator.actions === 'function' ? generator.actions(options) : generator.actions || [];
    await executeActions(actions, options, dir);
    return {
        success: true
    };
};
const registerHandlebarsHelpers = (helpers)=>{
    Object.entries(helpers).forEach(([name, fn])=>handlebars.registerHelper(name, fn));
};
// Executes generator actions: add or modify files as specified
const executeActions = async (actions, options, dir)=>{
    for (const action of actions){
        const outputPath = handlebars.compile(action.path)(options);
        const fullPath = node_path.join(dir, 'src', outputPath);
        if (action.type === 'add' && action.templateFile) {
            const templatePath = node_path.join(__dirname, action.templateFile);
            const templateContent = await fs.readFile(templatePath, 'utf8');
            const compiled = handlebars.compile(templateContent);
            const output = compiled({
                ...options,
                ...action.data
            });
            await fs.ensureDir(node_path.dirname(fullPath));
            await fs.writeFile(fullPath, output);
        }
        if (action.type === 'modify') {
            if (await fs.pathExists(fullPath)) {
                const content = await fs.readFile(fullPath, 'utf8');
                const modified = action.transform ? action.transform(content) : content;
                await fs.writeFile(fullPath, modified);
            }
        }
    }
};

exports.generate = generate;
exports.runCLI = runCLI;
//# sourceMappingURL=index.js.map
