import ___default from 'lodash';
import { defaults } from 'lodash/fp';
import { arrays, errors } from '@strapi/utils';
import { createUser, hasSuperAdminRole } from '../domain/user.mjs';
import { password } from '../validation/common-validators.mjs';
import { getService } from '../utils/index.mjs';
import constants from './constants.mjs';

const { SUPER_ADMIN_CODE } = constants;
const { ValidationError } = errors;
const sanitizeUserRoles = (role)=>___default.pick(role, [
        'id',
        'name',
        'description',
        'code'
    ]);
const getSessionManager = ()=>{
    const manager = strapi.sessionManager;
    return manager ?? null;
};
/**
 * Remove private user fields
 * @param  user - user to sanitize
 */ const sanitizeUser = (user)=>{
    return {
        ...___default.omit(user, [
            'password',
            'resetPasswordToken',
            'registrationToken',
            'roles'
        ]),
        roles: user.roles && user.roles.map(sanitizeUserRoles)
    };
};
/**
 * Create and save a user in database
 * @param attributes A partial user object
 */ const create = async (// isActive is added in the controller, it's not sent by the API.
attributes)=>{
    const userInfo = {
        registrationToken: getService('token').createToken(),
        ...attributes
    };
    if (___default.has(attributes, 'password')) {
        userInfo.password = await getService('auth').hashPassword(attributes.password);
    }
    const user = createUser(userInfo);
    const createdUser = await strapi.db.query('admin::user').create({
        data: user,
        populate: [
            'roles'
        ]
    });
    getService('metrics').sendDidInviteUser();
    strapi.eventHub.emit('user.create', {
        user: sanitizeUser(createdUser)
    });
    return createdUser;
};
/**
 * Update a user in database
 * @param id query params to find the user to update
 * @param attributes A partial user object
 */ const updateById = async (id, attributes)=>{
    // Check at least one super admin remains
    if (___default.has(attributes, 'roles')) {
        const lastAdminUser = await isLastSuperAdminUser(id);
        const superAdminRole = await getService('role').getSuperAdminWithUsersCount();
        const willRemoveSuperAdminRole = !arrays.includesString(attributes.roles, superAdminRole.id);
        if (lastAdminUser && willRemoveSuperAdminRole) {
            throw new ValidationError('You must have at least one user with super admin role.');
        }
    }
    // cannot disable last super admin
    if (attributes.isActive === false) {
        const lastAdminUser = await isLastSuperAdminUser(id);
        if (lastAdminUser) {
            throw new ValidationError('You must have at least one user with super admin role.');
        }
    }
    // hash password if a new one is sent
    if (___default.has(attributes, 'password')) {
        const hashedPassword = await getService('auth').hashPassword(attributes.password);
        const updatedUser = await strapi.db.query('admin::user').update({
            where: {
                id
            },
            data: {
                ...attributes,
                password: hashedPassword
            },
            populate: [
                'roles'
            ]
        });
        strapi.eventHub.emit('user.update', {
            user: sanitizeUser(updatedUser)
        });
        return updatedUser;
    }
    const updatedUser = await strapi.db.query('admin::user').update({
        where: {
            id
        },
        data: attributes,
        populate: [
            'roles'
        ]
    });
    if (updatedUser) {
        strapi.eventHub.emit('user.update', {
            user: sanitizeUser(updatedUser)
        });
    }
    return updatedUser;
};
/**
 * Reset a user password by email. (Used in admin:reset CLI)
 * @param email - user email
 * @param password - new password
 */ const resetPasswordByEmail = async (email, password$1)=>{
    const user = await strapi.db.query('admin::user').findOne({
        where: {
            email
        },
        populate: [
            'roles'
        ]
    });
    if (!user) {
        throw new Error(`User not found for email: ${email}`);
    }
    try {
        await password.validate(password$1);
    } catch (error) {
        throw new ValidationError('Invalid password. Expected a minimum of 8 characters with at least one number and one uppercase letter');
    }
    await updateById(user.id, {
        password: password$1
    });
};
/**
 * Check if a user is the last super admin
 * @param userId user's id to look for
 */ const isLastSuperAdminUser = async (userId)=>{
    const user = await findOne(userId);
    if (!user) return false;
    const superAdminRole = await getService('role').getSuperAdminWithUsersCount();
    return superAdminRole.usersCount === 1 && hasSuperAdminRole(user);
};
/**
 * Check if a user is the first super admin
 * @param userId user's id to look for
 */ const isFirstSuperAdminUser = async (userId)=>{
    const currentUser = await findOne(userId);
    if (!currentUser || !hasSuperAdminRole(currentUser)) return false;
    const [oldestUser] = await strapi.db.query('admin::user').findMany({
        populate: {
            roles: {
                where: {
                    code: {
                        $eq: SUPER_ADMIN_CODE
                    }
                }
            }
        },
        orderBy: {
            createdAt: 'asc'
        },
        limit: 1,
        select: [
            'id'
        ]
    });
    return oldestUser.id === currentUser.id;
};
/**
 * Check if a user with specific attributes exists in the database
 * @param attributes A partial user object
 */ const exists = async (attributes = {})=>{
    return await strapi.db.query('admin::user').count({
        where: attributes
    }) > 0;
};
/**
 * Returns a user registration info
 * @param registrationToken - a user registration token
 * @returns - Returns user email, firstname and lastname
 */ const findRegistrationInfo = async (registrationToken)=>{
    const user = await strapi.db.query('admin::user').findOne({
        where: {
            registrationToken
        }
    });
    if (!user) {
        return undefined;
    }
    return ___default.pick(user, [
        'email',
        'firstname',
        'lastname'
    ]);
};
/**
 * Registers a user based on a registrationToken and some informations to update
 * @param params
 * @param params.registrationToken registration token
 * @param params.userInfo user info
 */ const register = async ({ registrationToken, userInfo })=>{
    const matchingUser = await strapi.db.query('admin::user').findOne({
        where: {
            registrationToken
        }
    });
    if (!matchingUser) {
        throw new ValidationError('Invalid registration info');
    }
    return getService('user').updateById(matchingUser.id, {
        password: userInfo.password,
        firstname: userInfo.firstname,
        lastname: userInfo.lastname,
        registrationToken: null,
        isActive: true
    });
};
/**
 * Find one user
 */ const findOne = async (id, populate = [
    'roles'
])=>{
    return strapi.db.query('admin::user').findOne({
        where: {
            id
        },
        populate
    });
};
/**
 * Find one user by its email
 * @param email
 * @param populate
 * @returns
 */ const findOneByEmail = async (email, populate = [])=>{
    return strapi.db.query('admin::user').findOne({
        where: {
            email: {
                $eqi: email
            }
        },
        populate
    });
};
/** Find many users (paginated)
 * @param params
 */ const findPage = async (params = {})=>{
    const query = strapi.get('query-params').transform('admin::user', defaults({
        populate: [
            'roles'
        ]
    }, params));
    return strapi.db.query('admin::user').findPage(query);
};
/** Delete a user
 * @param id id of the user to delete
 */ const deleteById = async (id)=>{
    // Check at least one super admin remains
    const userToDelete = await strapi.db.query('admin::user').findOne({
        where: {
            id
        },
        populate: [
            'roles'
        ]
    });
    if (!userToDelete) {
        return null;
    }
    if (userToDelete) {
        if (userToDelete.roles.some((r)=>r.code === SUPER_ADMIN_CODE)) {
            const superAdminRole = await getService('role').getSuperAdminWithUsersCount();
            if (superAdminRole.usersCount === 1) {
                throw new ValidationError('You must have at least one user with super admin role.');
            }
        }
    }
    const deletedUser = await strapi.db.query('admin::user').delete({
        where: {
            id
        },
        populate: [
            'roles'
        ]
    });
    // Invalidate all sessions for the deleted user
    const sessionManager = getSessionManager();
    if (sessionManager && sessionManager.hasOrigin('admin')) {
        await sessionManager('admin').invalidateRefreshToken(String(id));
    }
    strapi.eventHub.emit('user.delete', {
        user: sanitizeUser(deletedUser)
    });
    return deletedUser;
};
/** Delete a user
 * @param ids ids of the users to delete
 */ const deleteByIds = async (ids)=>{
    // Check at least one super admin remains
    const superAdminRole = await getService('role').getSuperAdminWithUsersCount();
    const nbOfSuperAdminToDelete = await strapi.db.query('admin::user').count({
        where: {
            id: ids,
            roles: {
                id: superAdminRole.id
            }
        }
    });
    if (superAdminRole.usersCount === nbOfSuperAdminToDelete) {
        throw new ValidationError('You must have at least one user with super admin role.');
    }
    const deletedUsers = [];
    for (const id of ids){
        const deletedUser = await strapi.db.query('admin::user').delete({
            where: {
                id
            },
            populate: [
                'roles'
            ]
        });
        // Invalidate all sessions for the deleted user
        const sessionManager = getSessionManager();
        if (sessionManager && sessionManager.hasOrigin('admin')) {
            await sessionManager('admin').invalidateRefreshToken(String(id));
        }
        deletedUsers.push(deletedUser);
    }
    strapi.eventHub.emit('user.delete', {
        users: deletedUsers.map((deletedUser)=>sanitizeUser(deletedUser))
    });
    return deletedUsers;
};
/** Count the users that don't have any associated roles
 */ const countUsersWithoutRole = async ()=>{
    return strapi.db.query('admin::user').count({
        where: {
            roles: {
                id: {
                    $null: true
                }
            }
        }
    });
};
/**
 * Count the number of users based on search params
 * @param params params used for the query
 */ const count = async (where = {})=>{
    return strapi.db.query('admin::user').count({
        where
    });
};
/**
 * Assign some roles to several users
 */ const assignARoleToAll = async (roleId)=>{
    const users = await strapi.db.query('admin::user').findMany({
        select: [
            'id'
        ],
        where: {
            roles: {
                id: {
                    $null: true
                }
            }
        }
    });
    await Promise.all(users.map((user)=>{
        return strapi.db.query('admin::user').update({
            where: {
                id: user.id
            },
            data: {
                roles: [
                    roleId
                ]
            }
        });
    }));
};
/** Display a warning if some users don't have at least one role
 */ const displayWarningIfUsersDontHaveRole = async ()=>{
    const count = await countUsersWithoutRole();
    if (count > 0) {
        strapi.log.warn(`Some users (${count}) don't have any role.`);
    }
};
/** Returns an array of interface languages currently used by users
 */ const getLanguagesInUse = async ()=>{
    const users = await strapi.db.query('admin::user').findMany({
        select: [
            'preferedLanguage'
        ]
    });
    return users.map((user)=>user.preferedLanguage || 'en');
};
var user = {
    create,
    updateById,
    exists,
    findRegistrationInfo,
    register,
    sanitizeUser,
    findOne,
    findOneByEmail,
    findPage,
    deleteById,
    deleteByIds,
    countUsersWithoutRole,
    count,
    assignARoleToAll,
    displayWarningIfUsersDontHaveRole,
    resetPasswordByEmail,
    getLanguagesInUse,
    isFirstSuperAdminUser
};

export { user as default };
//# sourceMappingURL=user.mjs.map
