import * as _ from 'lodash';
import { errors } from '@strapi/utils';
import { validateUserCreationInput, validateUserUpdateInput, validateUsersDeleteInput } from '../validation/user.mjs';
import { getService } from '../utils/index.mjs';

const { ApplicationError } = errors;
var user = {
    async create (ctx) {
        const { body } = ctx.request;
        const cleanData = {
            ...body,
            email: _.get(body, `email`, ``).toLowerCase()
        };
        await validateUserCreationInput(cleanData);
        const attributes = _.pick(cleanData, [
            'firstname',
            'lastname',
            'email',
            'roles',
            'preferedLanguage'
        ]);
        const userAlreadyExists = await getService('user').exists({
            email: attributes.email
        });
        if (userAlreadyExists) {
            throw new ApplicationError('Email already taken');
        }
        const createdUser = await getService('user').create(attributes);
        const userInfo = getService('user').sanitizeUser(createdUser);
        // Note: We need to assign manually the registrationToken to the
        // final user payload so that it's not removed in the sanitation process.
        Object.assign(userInfo, {
            registrationToken: createdUser.registrationToken
        });
        // Send 201 created
        ctx.created({
            data: userInfo
        });
    },
    async find (ctx) {
        const userService = getService('user');
        const permissionsManager = strapi.service('admin::permission').createPermissionsManager({
            ability: ctx.state.userAbility,
            model: 'admin::user'
        });
        await permissionsManager.validateQuery(ctx.query);
        const sanitizedQuery = await permissionsManager.sanitizeQuery(ctx.query);
        // @ts-expect-error update the service type
        const { results, pagination } = await userService.findPage(sanitizedQuery);
        ctx.body = {
            data: {
                results: results.map((user)=>userService.sanitizeUser(user)),
                pagination
            }
        };
    },
    async findOne (ctx) {
        const { id } = ctx.params;
        const user = await getService('user').findOne(id);
        if (!user) {
            return ctx.notFound('User does not exist');
        }
        ctx.body = {
            data: getService('user').sanitizeUser(user)
        };
    },
    async update (ctx) {
        const { id } = ctx.params;
        const { body: input } = ctx.request;
        await validateUserUpdateInput(input);
        if (_.has(input, 'email')) {
            const uniqueEmailCheck = await getService('user').exists({
                id: {
                    $ne: id
                },
                email: input.email
            });
            if (uniqueEmailCheck) {
                throw new ApplicationError('A user with this email address already exists');
            }
        }
        const updatedUser = await getService('user').updateById(id, input);
        if (!updatedUser) {
            return ctx.notFound('User does not exist');
        }
        ctx.body = {
            data: getService('user').sanitizeUser(updatedUser)
        };
    },
    async deleteOne (ctx) {
        const { id } = ctx.params;
        const user = ctx.state.user;
        if (user && user.id === id) {
            throw new ApplicationError('You cannot delete your own user');
        }
        const deletedUser = await getService('user').deleteById(id);
        if (!deletedUser) {
            return ctx.notFound('User not found');
        }
        return ctx.deleted({
            data: getService('user').sanitizeUser(deletedUser)
        });
    },
    /**
   * Delete several users
   * @param ctx - koa context
   */ async deleteMany (ctx) {
        const { body } = ctx.request;
        const user = ctx.state.user;
        await validateUsersDeleteInput(body);
        const idsSet = new Set(body.ids);
        // Prevent self-deletion
        if (user && idsSet.has(user.id)) {
            throw new ApplicationError('You cannot delete your own user');
        }
        const users = await getService('user').deleteByIds(body.ids);
        const sanitizedUsers = users.map(getService('user').sanitizeUser);
        return ctx.deleted({
            data: sanitizedUsers
        });
    }
};

export { user as default };
//# sourceMappingURL=user.mjs.map
