import { jsx, jsxs } from 'react/jsx-runtime';
import 'react';
import { Box, Flex, Typography } from '@strapi/design-system';
import { Lightning } from '@strapi/icons';
import { useIntl } from 'react-intl';
import { styled } from 'styled-components';
import { useIsDesktop } from '../../hooks/useMediaQuery.mjs';
import { tours } from '../GuidedTour/Tours.mjs';
import { NavLink } from './NavLink.mjs';

const NavLinkBadgeLock = styled(NavLink.Badge)`
  background-color: transparent;
`;
const NavLinkBadgeCounter = styled(NavLink.Badge)`
  span {
    color: ${({ theme })=>theme.colors.neutral0};
  }
`;
/**
 * Will attach a guided tour tooltip to the right links. (mostly used for the finish step to indicate the next tour)
 * @param to: The target link
 * @param children: The original link to be wrapped in a guided tour tooltip
 * @returns The link wrapped in a guided tour tooltip or the original link if no guided tour needs to be attached
 */ const GuidedTourTooltip = ({ to, children })=>{
    const normalizedTo = to.toString().replace(/\//g, '');
    switch(normalizedTo){
        // We attach the final step of the content type builder tour on content manager link because it's the next tour (Content Type Builder -> Content Manager).
        case 'content-manager':
            return /*#__PURE__*/ jsx(tours.contentTypeBuilder.Finish, {
                children: children
            });
        // We attach the final step of the api tokens tour on the home link because it was the last tour (API Tokens -> Go back to homepage).
        case '':
            return /*#__PURE__*/ jsx(tours.apiTokens.Finish, {
                children: children
            });
        // We attach the final step of the content manager tour on the settings link because it's the next tour (Content Manager -> API tokens).
        case 'settings':
            return /*#__PURE__*/ jsx(tours.contentManager.Finish, {
                children: children
            });
        // If the link doesn't match any of the above, we return the original link.
        default:
            return children;
    }
};
const MainNavIcons = ({ listLinks, mobileLinks, handleClickOnLink })=>{
    const { formatMessage } = useIntl();
    const isDesktop = useIsDesktop();
    return listLinks.length > 0 ? listLinks.map((link)=>{
        const LinkIcon = link.icon;
        const badgeContentLock = link?.licenseOnly ? /*#__PURE__*/ jsx(Lightning, {
            fill: "primary600"
        }) : undefined;
        const badgeContentNumeric = link.notificationsCount && link.notificationsCount > 0 ? link.notificationsCount.toString() : undefined;
        const labelValue = formatMessage(link.intlLabel);
        const linkMobile = mobileLinks.find((mobileLink)=>mobileLink.to === link.to);
        const LinkElement = ()=>/*#__PURE__*/ jsxs(NavLink.Link, {
                to: link.to,
                onClick: ()=>handleClickOnLink(link.to),
                "aria-label": labelValue,
                children: [
                    /*#__PURE__*/ jsx(NavLink.Icon, {
                        label: labelValue,
                        children: /*#__PURE__*/ jsx(LinkIcon, {
                            width: "20",
                            height: "20",
                            fill: "neutral500"
                        })
                    }),
                    badgeContentLock ? /*#__PURE__*/ jsx(NavLinkBadgeLock, {
                        label: "locked",
                        textColor: "neutral500",
                        paddingLeft: 0,
                        paddingRight: 0,
                        children: badgeContentLock
                    }) : badgeContentNumeric ? /*#__PURE__*/ jsx(NavLinkBadgeCounter, {
                        label: badgeContentNumeric,
                        backgroundColor: "primary600",
                        width: "2.3rem",
                        color: "neutral0",
                        children: badgeContentNumeric
                    }) : null
                ]
            });
        return isDesktop || !isDesktop && linkMobile ? /*#__PURE__*/ jsx(Flex, {
            tag: "li",
            children: /*#__PURE__*/ jsx(GuidedTourTooltip, {
                to: link.to,
                children: /*#__PURE__*/ jsx(NavLink.Tooltip, {
                    position: isDesktop ? 'right' : 'bottom',
                    label: labelValue,
                    children: /*#__PURE__*/ jsx(LinkElement, {})
                })
            })
        }, link.to) : null;
    }) : null;
};
const IconContainer = styled(Box)`
  svg {
    display: block;
  }
`;
const MainNavBurgerMenuLinks = ({ listLinks, handleClickOnLink })=>{
    const { formatMessage } = useIntl();
    return listLinks.length > 0 ? listLinks.map((link)=>{
        const LinkIcon = link.icon;
        const badgeContentLock = link?.licenseOnly ? /*#__PURE__*/ jsx(Lightning, {
            fill: "primary600"
        }) : undefined;
        const badgeContentNumeric = link.notificationsCount && link.notificationsCount > 0 ? link.notificationsCount.toString() : undefined;
        const labelValue = formatMessage(link.intlLabel);
        const navigationTarget = link.navigationLink || link.to;
        return /*#__PURE__*/ jsx(Flex, {
            paddingTop: 3,
            alignItems: "center",
            tag: "li",
            children: /*#__PURE__*/ jsxs(NavLink.Link, {
                to: navigationTarget,
                onClick: ()=>handleClickOnLink(navigationTarget),
                "aria-label": labelValue,
                children: [
                    /*#__PURE__*/ jsx(IconContainer, {
                        marginRight: "0.6rem",
                        children: /*#__PURE__*/ jsx(LinkIcon, {
                            width: "20",
                            height: "20",
                            fill: "neutral500"
                        })
                    }),
                    /*#__PURE__*/ jsx(Typography, {
                        marginLeft: 3,
                        children: labelValue
                    }),
                    badgeContentLock ? /*#__PURE__*/ jsx(NavLinkBadgeLock, {
                        label: "locked",
                        textColor: "neutral500",
                        paddingLeft: 0,
                        paddingRight: 0,
                        children: badgeContentLock
                    }) : badgeContentNumeric ? /*#__PURE__*/ jsx(NavLinkBadgeCounter, {
                        label: badgeContentNumeric,
                        backgroundColor: "primary600",
                        width: "2.3rem",
                        color: "neutral0",
                        children: badgeContentNumeric
                    }) : null
                ]
            })
        }, navigationTarget);
    }) : null;
};

export { MainNavBurgerMenuLinks, MainNavIcons };
//# sourceMappingURL=MainNavLinks.mjs.map
