'use strict';

var os = require('os');
var path = require('path');
var _ = require('lodash');
var isDocker = require('is-docker');
var ciEnv = require('ci-info');
var tsUtils = require('@strapi/typescript-utils');
var strapiUtils = require('@strapi/utils');
var adminUserHash = require('./admin-user-hash.js');

const defaultQueryOpts = {
    timeout: 1000,
    headers: {
        'Content-Type': 'application/json'
    }
};
/**
 * Add properties from the package.json strapi key in the metadata
 */ const addPackageJsonStrapiMetadata = (metadata, strapi)=>{
    const { packageJsonStrapi = {} } = strapi.config;
    _.defaults(metadata, packageJsonStrapi);
};
/**
 * Create a send function for event with all the necessary metadata
 */ var createSender = ((strapi)=>{
    const { uuid, installId: installIdFromPackageJson } = strapi.config;
    const installId = strapiUtils.generateInstallId(uuid, installIdFromPackageJson);
    const serverRootPath = strapi.dirs.app.root;
    const adminRootPath = path.join(strapi.dirs.app.root, 'src', 'admin');
    const anonymousUserProperties = {
        environment: strapi.config.environment,
        os: os.type(),
        osPlatform: os.platform(),
        osArch: os.arch(),
        osRelease: os.release(),
        nodeVersion: process.versions.node
    };
    const anonymousGroupProperties = {
        docker: process.env.DOCKER || isDocker(),
        isCI: ciEnv.isCI,
        version: strapi.config.get('info.strapi'),
        useTypescriptOnServer: tsUtils.isUsingTypeScriptSync(serverRootPath),
        useTypescriptOnAdmin: tsUtils.isUsingTypeScriptSync(adminRootPath),
        projectId: uuid,
        isHostedOnStrapiCloud: strapiUtils.env('STRAPI_HOSTING', null) === 'strapi.cloud',
        aiLicenseKey: strapiUtils.env('STRAPI_ADMIN_AI_LICENSE', null)
    };
    addPackageJsonStrapiMetadata(anonymousGroupProperties, strapi);
    return async (event, payload = {}, opts = {})=>{
        const userId = adminUserHash.generateAdminUserHash(strapi);
        const reqParams = {
            method: 'POST',
            body: JSON.stringify({
                event,
                userId,
                installId,
                eventProperties: payload.eventProperties,
                userProperties: userId ? {
                    ...anonymousUserProperties,
                    ...payload.userProperties
                } : {},
                groupProperties: {
                    ...anonymousGroupProperties,
                    projectType: strapi.EE ? 'Enterprise' : 'Community',
                    ...payload.groupProperties
                }
            }),
            ..._.merge({
                headers: {
                    'X-Strapi-Event': event
                }
            }, defaultQueryOpts, opts)
        };
        try {
            const analyticsUrl = strapiUtils.env('STRAPI_ANALYTICS_URL', 'https://analytics.strapi.io');
            const res = await strapi.fetch(`${analyticsUrl}/api/v2/track`, reqParams);
            return res.ok;
        } catch (err) {
            return false;
        }
    };
});

module.exports = createSender;
//# sourceMappingURL=sender.js.map
