'use strict';

var path = require('path');
var fse = require('fs-extra');
var fp = require('lodash/fp');
var resolve = require('resolve.exports');
var strapiUtils = require('@strapi/utils');
var loadConfigFile = require('../../utils/load-config-file.js');
var loadFiles = require('../../utils/load-files.js');
var getEnabledPlugins = require('./get-enabled-plugins.js');
var getUserPluginsConfig = require('./get-user-plugins-config.js');
var index = require('../../domain/content-type/index.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var resolve__namespace = /*#__PURE__*/_interopNamespaceDefault(resolve);

const defaultPlugin = {
    bootstrap () {},
    destroy () {},
    register () {},
    config: {
        default: {},
        validator () {}
    },
    routes: [],
    controllers: {},
    services: {},
    policies: {},
    middlewares: {},
    contentTypes: {}
};
const applyUserExtension = async (plugins)=>{
    const extensionsDir = strapi.dirs.dist.extensions;
    if (!await fse.pathExists(extensionsDir)) {
        return;
    }
    const extendedSchemas = await loadFiles.loadFiles(extensionsDir, '**/content-types/**/schema.json');
    const strapiServers = await loadFiles.loadFiles(extensionsDir, '**/strapi-server.js');
    for (const pluginName of Object.keys(plugins)){
        const plugin = plugins[pluginName];
        // first: load json schema
        for (const ctName of Object.keys(plugin.contentTypes)){
            const extendedSchema = fp.get([
                pluginName,
                'content-types',
                ctName,
                'schema'
            ], extendedSchemas);
            if (extendedSchema) {
                plugin.contentTypes[ctName].schema = {
                    ...plugin.contentTypes[ctName].schema,
                    ...extendedSchema
                };
            }
        }
        // second: execute strapi-server extension
        const strapiServer = fp.get([
            pluginName,
            'strapi-server'
        ], strapiServers);
        if (strapiServer) {
            plugins[pluginName] = await strapiServer(plugin);
        }
    }
};
const applyUserConfig = async (plugins)=>{
    const userPluginsConfig = await getUserPluginsConfig.getUserPluginsConfig();
    for (const pluginName of Object.keys(plugins)){
        const plugin = plugins[pluginName];
        const userPluginConfig = fp.getOr({}, `${pluginName}.config`, userPluginsConfig);
        const defaultConfig = typeof plugin.config.default === 'function' ? plugin.config.default({
            env: strapiUtils.env
        }) : plugin.config.default;
        const config = fp.defaultsDeep(defaultConfig, userPluginConfig);
        try {
            plugin.config.validator(config);
        } catch (e) {
            if (e instanceof Error) {
                throw new Error(`Error regarding ${pluginName} config: ${e.message}`);
            }
            throw e;
        }
        plugin.config = config;
    }
};
async function loadPlugins(strapi1) {
    const plugins = {};
    const enabledPlugins = await getEnabledPlugins.getEnabledPlugins(strapi1);
    strapi1.config.set('enabledPlugins', enabledPlugins);
    for (const pluginName of Object.keys(enabledPlugins)){
        const enabledPlugin = enabledPlugins[pluginName];
        let serverEntrypointPath;
        let resolvedExport = './strapi-server.js';
        try {
            resolvedExport = (resolve__namespace.exports(enabledPlugin.packageInfo, 'strapi-server', {
                require: true
            }) ?? './strapi-server.js').toString();
        } catch (e) {
        // no export map or missing strapi-server export => fallback to default
        }
        try {
            serverEntrypointPath = path.join(enabledPlugin.pathToPlugin, resolvedExport);
        } catch (e) {
            throw new Error(`Error loading the plugin ${pluginName} because ${pluginName} is not installed. Please either install the plugin or remove its configuration.`);
        }
        // only load plugins with a server entrypoint
        if (!await fse.pathExists(serverEntrypointPath)) {
            continue;
        }
        const pluginServer = loadConfigFile.loadConfigFile(serverEntrypointPath);
        plugins[pluginName] = {
            ...defaultPlugin,
            ...pluginServer,
            contentTypes: formatContentTypes(pluginName, pluginServer.contentTypes ?? {}),
            config: fp.defaults(defaultPlugin.config, pluginServer.config),
            routes: pluginServer.routes ?? defaultPlugin.routes
        };
    }
    // TODO: validate plugin format
    await applyUserConfig(plugins);
    await applyUserExtension(plugins);
    for (const pluginName of Object.keys(plugins)){
        strapi1.get('plugins').add(pluginName, plugins[pluginName]);
    }
}
const formatContentTypes = (pluginName, contentTypes)=>{
    Object.values(contentTypes).forEach((definition)=>{
        const { schema } = definition;
        Object.assign(schema, {
            plugin: pluginName,
            collectionName: schema.collectionName || `${pluginName}_${schema.info.singularName}`.toLowerCase(),
            globalId: index.getGlobalId(schema, pluginName)
        });
    });
    return contentTypes;
};

module.exports = loadPlugins;
//# sourceMappingURL=index.js.map
