'use strict';

var strapiUtils = require('@strapi/utils');
var z = require('zod/v4');
var mappers = require('./mappers.js');
var common = require('./common.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var z__namespace = /*#__PURE__*/_interopNamespaceDefault(z);

/**
 * A validator for core content-type routes.
 *
 * Provides validation schemas and utilities for handling content-type-specific route validation.
 * Extends the base AbstractRouteValidator with schema-aware validation for Strapi content types.
 *
 * @example
 * ```ts
 * const strapi = // ... strapi instance
 * const uid = 'api::article.article'
 * const validator = new CoreContentTypeRouteValidator(strapi, uid);
 *
 * // Get validation schema for document
 * const documentSchema = validator.document;
 *
 * // Validate query parameters with schema awareness
 * const querySchema = validator.queryParams(['fields', 'populate', 'sort']);
 * ```
 */ class CoreContentTypeRouteValidator extends common.AbstractCoreRouteValidator {
    /**
   * Generates a validation schema for document IDs
   *
   * @returns A schema that validates UUIDs
   *
   * @example
   * ```ts
   * const validator = new CoreContentTypeRouteValidator(strapi, uid);
   * const idSchema = validator.documentID;
   * ```
   */ get documentID() {
        return z__namespace.uuid().describe('The document ID, represented by a UUID');
    }
    /**
   * Generates a comprehensive validation schema for a single document.
   *
   * Combines scalar fields and populatable fields into a single schema.
   *
   * @returns A schema for validating complete documents
   *
   * @example
   * ```ts
   * const validator = new CoreContentTypeRouteValidator(strapi, uid);
   * const docSchema = validator.document;
   * ```
   */ get document() {
        const entries = Object.entries({
            ...this._scalarFields,
            ...this._populatableFields
        });
        const sanitizedAttributes = entries// Remove passwords from the attribute list
        .filter(([, attribute])=>![
                'password'
            ].includes(attribute.type));
        // Merge all attributes into a single schema
        const attributesSchema = mappers.createAttributesSchema(sanitizedAttributes);
        return z__namespace.object({
            documentId: this.documentID,
            id: z__namespace.union([
                z__namespace.string(),
                z__namespace.number()
            ])
        }).extend(attributesSchema.shape);
    }
    /**
   * Generates a validation schema for an array of documents
   *
   * @returns A schema for validating arrays of documents
   *
   * @example
   * ```ts
   * const validator = new CoreContentTypeRouteValidator(strapi, uid);
   * const docsSchema = validator.documents;
   * ```
   */ get documents() {
        return z__namespace.array(this.document);
    }
    /**
   * Schema-aware fields validation that restricts to actual model fields
   */ get schemaAwareQueryFields() {
        return this.scalarFieldsArray.readonly().describe(`The fields to return, this doesn't include populatable fields like relations, components, files, or dynamic zones`);
    }
    /**
   * Schema-aware populate validation that restricts to actual populatable fields
   */ get schemaAwareQueryPopulate() {
        const wildcardPopulate = z__namespace.literal('*').readonly().describe('Populate all the first level relations, components, files, and dynamic zones for the entry');
        const singleFieldPopulate = this.populatableFieldsEnum.readonly().describe('Populate a single relation, component, file, or dynamic zone');
        const multiPopulate = this.populatableFieldsArray.describe('Populate a selection of multiple relations, components, files, or dynamic zones');
        return z__namespace.union([
            wildcardPopulate,
            singleFieldPopulate,
            multiPopulate
        ]);
    }
    /**
   * Schema-aware sort validation that restricts to actual model fields
   */ get schemaAwareQuerySort() {
        const orderDirection = z__namespace.enum([
            'asc',
            'desc'
        ]);
        // TODO: Handle nested sorts but very low priority, very little usage
        return z__namespace.union([
            this.scalarFieldsEnum,
            this.scalarFieldsArray,
            this.fieldRecord(orderDirection),
            z__namespace.array(this.fieldRecord(orderDirection))
        ]).describe('Sort the result');
    }
    /**
   * Schema-aware filters validation that restricts to actual model fields
   */ get schemaAwareFilters() {
        return z__namespace.record(this.scalarFieldsEnum, z__namespace.any()).describe('Filters to apply to the query');
    }
    get locale() {
        return z__namespace.string().describe('Select a locale');
    }
    get status() {
        return z__namespace.enum([
            'draft',
            'published'
        ]).describe('Fetch documents based on their status. Default to "published" if not specified.');
    }
    get data() {
        const isWritableAttribute = ([attributeName])=>{
            return strapiUtils.contentTypes.isWritableAttribute(this._schema, attributeName);
        };
        const entries = Object.entries({
            ...this._scalarFields,
            ...this._populatableFields
        });
        const sanitizedAttributes = entries// Remove non-writable attributes
        .filter(isWritableAttribute);
        return mappers.createAttributesInputSchema(sanitizedAttributes);
    }
    get query() {
        return z__namespace.string();
    }
    get body() {
        return z__namespace.object({
            data: this.data
        });
    }
    get partialBody() {
        return z__namespace.object({
            data: this.data.partial()
        });
    }
    /**
   * Creates validation schemas for query parameters
   *
   * @param params - Array of query parameters to validate ('fields', 'populate', 'sort', ...)
   * @returns Object containing validation schemas for requested parameters
   *
   * @example
   * ```ts
   * const validator = new CoreContentTypeRouteValidator(strapi, uid);
   * const querySchemas = validator.queryParams(['fields', 'populate']);
   * ```
   */ queryParams(params) {
        const map = {
            fields: ()=>this.schemaAwareQueryFields.optional(),
            populate: ()=>this.schemaAwareQueryPopulate.optional(),
            sort: ()=>this.schemaAwareQuerySort.optional(),
            filters: ()=>this.schemaAwareFilters.optional(),
            locale: ()=>this.locale.optional(),
            pagination: ()=>this.pagination.optional(),
            status: ()=>this.status.optional(),
            _q: ()=>this.query.optional()
        };
        return params.reduce((acc, param)=>({
                ...acc,
                [param]: map[param]()
            }), {});
    }
}

exports.CoreContentTypeRouteValidator = CoreContentTypeRouteValidator;
//# sourceMappingURL=content-type.js.map
