import type { UID } from '@strapi/types';
import * as z from 'zod/v4';
import { AbstractCoreRouteValidator } from './common';
export type QueryParam = 'fields' | 'populate' | 'sort' | 'status' | 'locale' | 'pagination' | 'filters' | '_q';
/**
 * A validator for core content-type routes.
 *
 * Provides validation schemas and utilities for handling content-type-specific route validation.
 * Extends the base AbstractRouteValidator with schema-aware validation for Strapi content types.
 *
 * @example
 * ```ts
 * const strapi = // ... strapi instance
 * const uid = 'api::article.article'
 * const validator = new CoreContentTypeRouteValidator(strapi, uid);
 *
 * // Get validation schema for document
 * const documentSchema = validator.document;
 *
 * // Validate query parameters with schema awareness
 * const querySchema = validator.queryParams(['fields', 'populate', 'sort']);
 * ```
 */
export declare class CoreContentTypeRouteValidator extends AbstractCoreRouteValidator<UID.ContentType> {
    /**
     * Generates a validation schema for document IDs
     *
     * @returns A schema that validates UUIDs
     *
     * @example
     * ```ts
     * const validator = new CoreContentTypeRouteValidator(strapi, uid);
     * const idSchema = validator.documentID;
     * ```
     */
    get documentID(): z.ZodUUID;
    /**
     * Generates a comprehensive validation schema for a single document.
     *
     * Combines scalar fields and populatable fields into a single schema.
     *
     * @returns A schema for validating complete documents
     *
     * @example
     * ```ts
     * const validator = new CoreContentTypeRouteValidator(strapi, uid);
     * const docSchema = validator.document;
     * ```
     */
    get document(): z.ZodObject<{
        documentId: z.ZodUUID;
        id: z.ZodUnion<readonly [z.ZodString, z.ZodNumber]>;
    }, z.core.$strip>;
    /**
     * Generates a validation schema for an array of documents
     *
     * @returns A schema for validating arrays of documents
     *
     * @example
     * ```ts
     * const validator = new CoreContentTypeRouteValidator(strapi, uid);
     * const docsSchema = validator.documents;
     * ```
     */
    get documents(): z.ZodArray<z.ZodObject<{
        documentId: z.ZodUUID;
        id: z.ZodUnion<readonly [z.ZodString, z.ZodNumber]>;
    }, z.core.$strip>>;
    /**
     * Schema-aware fields validation that restricts to actual model fields
     */
    protected get schemaAwareQueryFields(): z.ZodReadonly<z.ZodArray<z.ZodEnum<{
        [x: string]: string;
    }>>>;
    /**
     * Schema-aware populate validation that restricts to actual populatable fields
     */
    protected get schemaAwareQueryPopulate(): z.ZodUnion<readonly [z.ZodReadonly<z.ZodLiteral<"*">>, z.ZodReadonly<z.ZodEnum<{
        [x: string]: string;
    }>>, z.ZodArray<z.ZodEnum<{
        [x: string]: string;
    }>>]>;
    /**
     * Schema-aware sort validation that restricts to actual model fields
     */
    protected get schemaAwareQuerySort(): z.ZodUnion<readonly [z.ZodEnum<{
        [x: string]: string;
    }>, z.ZodArray<z.ZodEnum<{
        [x: string]: string;
    }>>, z.ZodRecord<z.ZodEnum<{
        [x: string]: string;
    }>, z.ZodType<unknown, unknown, z.core.$ZodTypeInternals<unknown, unknown>>>, z.ZodArray<z.ZodRecord<z.ZodEnum<{
        [x: string]: string;
    }>, z.ZodType<unknown, unknown, z.core.$ZodTypeInternals<unknown, unknown>>>>]>;
    /**
     * Schema-aware filters validation that restricts to actual model fields
     */
    protected get schemaAwareFilters(): z.ZodRecord<z.ZodEnum<{
        [x: string]: string;
    }>, z.ZodAny>;
    get locale(): z.ZodString;
    get status(): z.ZodEnum<{
        draft: "draft";
        published: "published";
    }>;
    get data(): z.ZodObject<{}, z.core.$strip>;
    get query(): z.ZodString;
    get body(): z.ZodObject<{
        data: z.ZodObject<{}, z.core.$strip>;
    }, z.core.$strip>;
    get partialBody(): z.ZodObject<{
        data: z.ZodObject<{}, z.core.$strip>;
    }, z.core.$strip>;
    /**
     * Creates validation schemas for query parameters
     *
     * @param params - Array of query parameters to validate ('fields', 'populate', 'sort', ...)
     * @returns Object containing validation schemas for requested parameters
     *
     * @example
     * ```ts
     * const validator = new CoreContentTypeRouteValidator(strapi, uid);
     * const querySchemas = validator.queryParams(['fields', 'populate']);
     * ```
     */
    queryParams(params: QueryParam[]): Partial<Record<QueryParam, z.Schema>>;
}
//# sourceMappingURL=content-type.d.ts.map