import { augmentSchema, maybeRequired, maybeWithDefault, maybeReadonly, maybeWithMinMax, relations } from '@strapi/utils';
import { z } from 'zod/v4';
import { BOOLEAN_LITERAL_VALUES } from './constants.mjs';
import { CoreComponentRouteValidator } from './component.mjs';
import { CoreContentTypeRouteValidator } from './content-type.mjs';
import { safeSchemaCreation } from './utils.mjs';

/**
 * Converts a BigInteger attribute to a Zod schema.
 * @param attribute - The BigInteger attribute object from the Strapi schema.
 * @returns A Zod schema representing the BigInteger field.
 */ const bigIntegerToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        (schema)=>min !== undefined ? schema.min(min) : schema,
        (schema)=>max !== undefined ? schema.max(max) : schema,
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A biginteger field');
};
/**
 * Converts a blocks attribute to a Zod schema.
 * @returns A Zod schema representing the blocks field.
 */ const blocksToSchema = ()=>{
    return z.array(z.any()).describe('A blocks field');
};
/**
 * Converts a boolean attribute to a Zod schema.
 * @param attribute - The Boolean attribute object from the Strapi schema.
 * @returns A Zod schema representing the boolean field.
 */ const booleanToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.boolean().nullable(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A boolean field');
};
/**
 * Converts a component attribute to a Zod schema.
 * @param attribute - The Component attribute object from the Strapi schema.
 * @returns A Zod schema representing the component field.
 */ const componentToSchema = (attribute)=>{
    const { writable, required, min, max, component, repeatable } = attribute;
    const componentSchema = safeSchemaCreation(component, ()=>new CoreComponentRouteValidator(strapi, component).entry);
    const baseSchema = repeatable ? z.array(componentSchema) : componentSchema;
    const schema = augmentSchema(baseSchema, [
        (schema)=>min !== undefined && schema instanceof z.ZodArray ? schema.min(min) : schema,
        (schema)=>max !== undefined && schema instanceof z.ZodArray ? schema.max(max) : schema,
        maybeRequired(required),
        maybeReadonly(writable)
    ]);
    return schema.describe('A component field');
};
/**
 * Converts a date attribute to a Zod schema.
 * @param attribute - The Date attribute object from the Strapi schema.
 * @returns A Zod schema representing the date field.
 */ const dateToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A date field');
};
/**
 * Converts a datetime attribute to a Zod schema.
 * @param attribute - The DateTime attribute object from the Strapi schema.
 * @returns A Zod schema representing the datetime field.
 */ const datetimeToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A datetime field');
};
/**
 * Converts a decimal attribute to a Zod schema.
 * @param attribute - The Decimal attribute object from the Strapi schema.
 * @returns A Zod schema representing the decimal field.
 */ const decimalToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const schema = augmentSchema(z.number(), [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A decimal field');
};
/**
 * Converts a dynamic zone attribute to a Zod schema.
 * @param attribute - The DynamicZone attribute object from the Strapi schema.
 * @returns A Zod schema representing the dynamic zone field.
 */ const dynamicZoneToSchema = (attribute)=>{
    const { writable, required, min, max } = attribute;
    const baseSchema = z.array(z.any());
    const schema = augmentSchema(baseSchema, [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeReadonly(writable)
    ]);
    return schema.describe('A dynamic zone field');
};
/**
 * Converts an email attribute to a Zod schema.
 * @param attribute - The Email attribute object from the Strapi schema.
 * @returns A Zod schema representing the email field.
 */ const emailToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, minLength, maxLength } = attribute;
    const baseSchema = z.email();
    const schema = augmentSchema(baseSchema, [
        maybeWithMinMax(minLength, maxLength),
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('An email field');
};
/**
 * Converts an enumeration attribute to a Zod schema.
 * @param attribute - The Enumeration attribute object from the Strapi schema.
 * @returns A Zod schema representing the enumeration field.
 */ const enumToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, enum: enumValues } = attribute;
    const baseSchema = z.enum(enumValues);
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('An enum field');
};
/**
 * Converts a float attribute to a Zod schema.
 * @param attribute - The Float attribute object from the Strapi schema.
 * @returns A Zod schema representing the float field.
 */ const floatToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const schema = augmentSchema(z.number(), [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeReadonly(writable),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A float field');
};
/**
 * Converts an integer attribute to a Zod schema.
 * @param attribute - The Integer attribute object from the Strapi schema.
 * @returns A Zod schema representing the integer field.
 */ const integerToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const baseSchema = z.number().int();
    const schema = augmentSchema(baseSchema, [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeReadonly(writable),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('An integer field');
};
/**
 * Converts a JSON attribute to a Zod schema.
 * @param attribute - The JSON attribute object from the Strapi schema.
 * @returns A Zod schema representing the JSON field.
 */ const jsonToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.any(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A JSON field');
};
/**
 * Converts a media attribute to a Zod schema.
 * @param attribute - The Media attribute object from the Strapi schema.
 * @returns A Zod schema representing the media field.
 */ const mediaToSchema = (attribute)=>{
    const { writable, required, multiple } = attribute;
    const uploadPlugin = strapi.plugin('upload');
    // @ts-expect-error there is a mismatch between a raw module and a loader module
    const fileSchema = uploadPlugin.contentTypes.file;
    const mediaSchema = safeSchemaCreation(fileSchema.uid, ()=>new CoreContentTypeRouteValidator(strapi, fileSchema.uid).document);
    const baseSchema = multiple ? z.array(mediaSchema) : mediaSchema;
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeReadonly(writable)
    ]);
    return schema.describe('A media field');
};
/**
 * Converts a relation attribute to a Zod schema.
 * @param attribute - The Relation attribute object from the Strapi schema.
 * @returns A Zod schema representing the relational field.
 */ const relationToSchema = (attribute)=>{
    if (!('target' in attribute)) {
        return z.any();
    }
    const { writable, required, target } = attribute;
    const targetSchema = safeSchemaCreation(target, ()=>new CoreContentTypeRouteValidator(strapi, target).document);
    const baseSchema = relations.isAnyToMany(attribute) ? z.array(targetSchema) : targetSchema;
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeReadonly(writable)
    ]);
    return schema.describe('A relational field');
};
/**
 * Converts a string, text, rich text, or password attribute to a Zod schema.
 * @param attribute - The String, Text, RichText, or Password attribute object from the Strapi schema.
 * @returns A Zod schema representing the field.
 */ const stringToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeWithMinMax(minLength, maxLength),
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe(`A ${attribute.type} field`);
};
/**
 * Converts a time attribute to a Zod schema.
 * @param attribute - The Time attribute object from the Strapi schema.
 * @returns A Zod schema representing the time field.
 */ const timeToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A time field');
};
/**
 * Converts a timestamp attribute to a Zod schema.
 * @param attribute - The Timestamp attribute object from the Strapi schema.
 * @returns A Zod schema representing the timestamp field.
 */ const timestampToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const baseSchema = z.union([
        z.string(),
        z.number()
    ]);
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A timestamp field');
};
/**
 * Converts a UID attribute to a Zod schema.
 * @param attribute - The UID attribute object from the Strapi schema.
 * @returns A Zod schema representing the UID field.
 */ const uidToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeWithMinMax(minLength, maxLength),
        maybeRequired(required),
        maybeWithDefault(defaultValue),
        maybeReadonly(writable)
    ]);
    return schema.describe('A UID field');
};
/**
 * Converts a BigInteger attribute to a Zod schema for input validation.
 * @param attribute - The BigInteger attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the BigInteger field.
 */ const bigIntegerToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        (schema)=>min !== undefined ? schema.min(min) : schema,
        (schema)=>max !== undefined ? schema.max(max) : schema,
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A biginteger field');
};
/**
 * Converts a blocks attribute to a Zod schema for input validation.
 * @returns A Zod schema for input validation of the blocks field.
 */ const blocksToInputSchema = ()=>{
    // TODO: better support blocks data structure
    return z.array(z.any()).describe('A blocks field');
};
/**
 * Converts a boolean attribute to a Zod schema for input validation.
 * @param attribute - The Boolean attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the boolean field.
 */ const booleanToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const baseSchema = z.enum(BOOLEAN_LITERAL_VALUES).nullable();
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A boolean field');
};
/**
 * Converts a component attribute to a Zod schema for input validation.
 * @param attribute - The Component attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the component field.
 */ const componentToInputSchema = (attribute)=>{
    const { required, repeatable, min, max } = attribute;
    const baseSchema = repeatable ? z.array(z.any()) : z.any();
    const schema = augmentSchema(baseSchema, [
        (schema)=>min !== undefined && schema instanceof z.ZodArray ? schema.min(min) : schema,
        (schema)=>max !== undefined && schema instanceof z.ZodArray ? schema.max(max) : schema,
        maybeRequired(required)
    ]);
    return schema.describe('A component field');
};
/**
 * Converts a date attribute to a Zod schema for input validation.
 * @param attribute - The Date attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the date field.
 */ const dateToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A date field');
};
/**
 * Converts a datetime attribute to a Zod schema for input validation.
 * @param attribute - The DateTime attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the datetime field.
 */ const datetimeToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A datetime field');
};
/**
 * Converts a decimal attribute to a Zod schema for input validation.
 * @param attribute - The Decimal attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the decimal field.
 */ const decimalToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const schema = augmentSchema(z.number(), [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A decimal field');
};
/**
 * Converts a dynamic zone attribute to a Zod schema for input validation.
 * @param attribute - The DynamicZone attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the dynamic zone field.
 */ const dynamicZoneToInputSchema = (attribute)=>{
    const { required, min, max } = attribute;
    const baseSchema = z.array(z.any());
    const schema = augmentSchema(baseSchema, [
        maybeWithMinMax(min, max),
        maybeRequired(required)
    ]);
    return schema.describe('A dynamic zone field');
};
/**
 * Converts an email attribute to a Zod schema for input validation.
 * @param attribute - The Email attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the email field.
 */ const emailToInputSchema = (attribute)=>{
    const { required, default: defaultValue, minLength, maxLength } = attribute;
    const baseSchema = z.email();
    const schema = augmentSchema(baseSchema, [
        maybeWithMinMax(minLength, maxLength),
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('An email field');
};
/**
 * Converts an enumeration attribute to a Zod schema for input validation.
 * @param attribute - The Enumeration attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the enumeration field.
 */ const enumerationToInputSchema = (attribute)=>{
    const { required, default: defaultValue, enum: enumValues } = attribute;
    const baseSchema = z.enum(enumValues);
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('An enum field');
};
/**
 * Converts a float attribute to a Zod schema for input validation.
 * @param attribute - The Float attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the float field.
 */ const floatToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const schema = augmentSchema(z.number(), [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A float field');
};
/**
 * Converts an integer attribute to a Zod schema for input validation.
 * @param attribute - The Integer attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the integer field.
 */ const integerToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const baseSchema = z.number().int();
    const schema = augmentSchema(baseSchema, [
        maybeWithMinMax(min, max),
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A float field');
};
/**
 * Converts a JSON attribute to a Zod schema for input validation.
 * @param attribute - The JSON attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the JSON field.
 */ const jsonToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.any(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A JSON field');
};
/**
 * Converts a media attribute to a Zod schema for input validation.
 * @param attribute - The Media attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the media field.
 */ const mediaToInputSchema = (attribute)=>{
    const { required, multiple } = attribute;
    const baseSchema = multiple ? z.array(z.any()) : z.any();
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required)
    ]);
    return schema.describe('A media field');
};
/**
 * Converts a relation attribute to a Zod schema for input validation.
 * @param attribute - The Relation attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the relational field.
 */ const relationToInputSchema = (attribute)=>{
    const { required } = attribute;
    const isToMany = relations.isAnyToMany(attribute);
    const uuid = z.string().uuid();
    const baseSchema = isToMany ? z.array(uuid) : uuid;
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required)
    ]);
    return schema.describe('A relational field');
};
/**
 * Converts a string, text, rich text, or password attribute to a Zod schema for input validation.
 * @param attribute - The String, Text, RichText, or Password attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the field.
 */ const textToInputSchema = (attribute)=>{
    const { required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeWithMinMax(minLength, maxLength),
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe(`A ${attribute.type} field`);
};
/**
 * Converts a time attribute to a Zod schema for input validation.
 * @param attribute - The Time attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the time field.
 */ const timeToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A time field');
};
/**
 * Converts a timestamp attribute to a Zod schema for input validation.
 * @param attribute - The Timestamp attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the timestamp field.
 */ const timestampToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const baseSchema = z.union([
        z.string(),
        z.number()
    ]);
    const schema = augmentSchema(baseSchema, [
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A timestamp field');
};
/**
 * Converts a UID attribute to a Zod schema for input validation.
 * @param attribute - The UID attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the UID field.
 */ const uidToInputSchema = (attribute)=>{
    const { required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = augmentSchema(z.string(), [
        maybeWithMinMax(minLength, maxLength),
        maybeRequired(required),
        maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A UID field');
};

export { bigIntegerToInputSchema, bigIntegerToSchema, blocksToInputSchema, blocksToSchema, booleanToInputSchema, booleanToSchema, componentToInputSchema, componentToSchema, dateToInputSchema, dateToSchema, datetimeToInputSchema, datetimeToSchema, decimalToInputSchema, decimalToSchema, dynamicZoneToInputSchema, dynamicZoneToSchema, emailToInputSchema, emailToSchema, enumToSchema, enumerationToInputSchema, floatToInputSchema, floatToSchema, integerToInputSchema, integerToSchema, jsonToInputSchema, jsonToSchema, mediaToInputSchema, mediaToSchema, relationToInputSchema, relationToSchema, stringToSchema, textToInputSchema, timeToInputSchema, timeToSchema, timestampToInputSchema, timestampToSchema, uidToInputSchema, uidToSchema };
//# sourceMappingURL=attributes.mjs.map
