'use strict';

var strapiUtils = require('@strapi/utils');
var z = require('zod/v4');
var constants = require('./constants.js');
var component = require('./component.js');
var contentType = require('./content-type.js');
var utils = require('./utils.js');

/**
 * Converts a BigInteger attribute to a Zod schema.
 * @param attribute - The BigInteger attribute object from the Strapi schema.
 * @returns A Zod schema representing the BigInteger field.
 */ const bigIntegerToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        (schema)=>min !== undefined ? schema.min(min) : schema,
        (schema)=>max !== undefined ? schema.max(max) : schema,
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A biginteger field');
};
/**
 * Converts a blocks attribute to a Zod schema.
 * @returns A Zod schema representing the blocks field.
 */ const blocksToSchema = ()=>{
    return z.z.array(z.z.any()).describe('A blocks field');
};
/**
 * Converts a boolean attribute to a Zod schema.
 * @param attribute - The Boolean attribute object from the Strapi schema.
 * @returns A Zod schema representing the boolean field.
 */ const booleanToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.boolean().nullable(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A boolean field');
};
/**
 * Converts a component attribute to a Zod schema.
 * @param attribute - The Component attribute object from the Strapi schema.
 * @returns A Zod schema representing the component field.
 */ const componentToSchema = (attribute)=>{
    const { writable, required, min, max, component: component$1, repeatable } = attribute;
    const componentSchema = utils.safeSchemaCreation(component$1, ()=>new component.CoreComponentRouteValidator(strapi, component$1).entry);
    const baseSchema = repeatable ? z.z.array(componentSchema) : componentSchema;
    const schema = strapiUtils.augmentSchema(baseSchema, [
        (schema)=>min !== undefined && schema instanceof z.z.ZodArray ? schema.min(min) : schema,
        (schema)=>max !== undefined && schema instanceof z.z.ZodArray ? schema.max(max) : schema,
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A component field');
};
/**
 * Converts a date attribute to a Zod schema.
 * @param attribute - The Date attribute object from the Strapi schema.
 * @returns A Zod schema representing the date field.
 */ const dateToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A date field');
};
/**
 * Converts a datetime attribute to a Zod schema.
 * @param attribute - The DateTime attribute object from the Strapi schema.
 * @returns A Zod schema representing the datetime field.
 */ const datetimeToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A datetime field');
};
/**
 * Converts a decimal attribute to a Zod schema.
 * @param attribute - The Decimal attribute object from the Strapi schema.
 * @returns A Zod schema representing the decimal field.
 */ const decimalToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.number(), [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A decimal field');
};
/**
 * Converts a dynamic zone attribute to a Zod schema.
 * @param attribute - The DynamicZone attribute object from the Strapi schema.
 * @returns A Zod schema representing the dynamic zone field.
 */ const dynamicZoneToSchema = (attribute)=>{
    const { writable, required, min, max } = attribute;
    const baseSchema = z.z.array(z.z.any());
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A dynamic zone field');
};
/**
 * Converts an email attribute to a Zod schema.
 * @param attribute - The Email attribute object from the Strapi schema.
 * @returns A Zod schema representing the email field.
 */ const emailToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, minLength, maxLength } = attribute;
    const baseSchema = z.z.email();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeWithMinMax(minLength, maxLength),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('An email field');
};
/**
 * Converts an enumeration attribute to a Zod schema.
 * @param attribute - The Enumeration attribute object from the Strapi schema.
 * @returns A Zod schema representing the enumeration field.
 */ const enumToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, enum: enumValues } = attribute;
    const baseSchema = z.z.enum(enumValues);
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('An enum field');
};
/**
 * Converts a float attribute to a Zod schema.
 * @param attribute - The Float attribute object from the Strapi schema.
 * @returns A Zod schema representing the float field.
 */ const floatToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.number(), [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeReadonly(writable),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A float field');
};
/**
 * Converts an integer attribute to a Zod schema.
 * @param attribute - The Integer attribute object from the Strapi schema.
 * @returns A Zod schema representing the integer field.
 */ const integerToSchema = (attribute)=>{
    const { writable, required, min, max, default: defaultValue } = attribute;
    const baseSchema = z.z.number().int();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeReadonly(writable),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('An integer field');
};
/**
 * Converts a JSON attribute to a Zod schema.
 * @param attribute - The JSON attribute object from the Strapi schema.
 * @returns A Zod schema representing the JSON field.
 */ const jsonToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.any(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A JSON field');
};
/**
 * Converts a media attribute to a Zod schema.
 * @param attribute - The Media attribute object from the Strapi schema.
 * @returns A Zod schema representing the media field.
 */ const mediaToSchema = (attribute)=>{
    const { writable, required, multiple } = attribute;
    const uploadPlugin = strapi.plugin('upload');
    // @ts-expect-error there is a mismatch between a raw module and a loader module
    const fileSchema = uploadPlugin.contentTypes.file;
    const mediaSchema = utils.safeSchemaCreation(fileSchema.uid, ()=>new contentType.CoreContentTypeRouteValidator(strapi, fileSchema.uid).document);
    const baseSchema = multiple ? z.z.array(mediaSchema) : mediaSchema;
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A media field');
};
/**
 * Converts a relation attribute to a Zod schema.
 * @param attribute - The Relation attribute object from the Strapi schema.
 * @returns A Zod schema representing the relational field.
 */ const relationToSchema = (attribute)=>{
    if (!('target' in attribute)) {
        return z.z.any();
    }
    const { writable, required, target } = attribute;
    const targetSchema = utils.safeSchemaCreation(target, ()=>new contentType.CoreContentTypeRouteValidator(strapi, target).document);
    const baseSchema = strapiUtils.relations.isAnyToMany(attribute) ? z.z.array(targetSchema) : targetSchema;
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A relational field');
};
/**
 * Converts a string, text, rich text, or password attribute to a Zod schema.
 * @param attribute - The String, Text, RichText, or Password attribute object from the Strapi schema.
 * @returns A Zod schema representing the field.
 */ const stringToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeWithMinMax(minLength, maxLength),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe(`A ${attribute.type} field`);
};
/**
 * Converts a time attribute to a Zod schema.
 * @param attribute - The Time attribute object from the Strapi schema.
 * @returns A Zod schema representing the time field.
 */ const timeToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A time field');
};
/**
 * Converts a timestamp attribute to a Zod schema.
 * @param attribute - The Timestamp attribute object from the Strapi schema.
 * @returns A Zod schema representing the timestamp field.
 */ const timestampToSchema = (attribute)=>{
    const { writable, required, default: defaultValue } = attribute;
    const baseSchema = z.z.union([
        z.z.string(),
        z.z.number()
    ]);
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A timestamp field');
};
/**
 * Converts a UID attribute to a Zod schema.
 * @param attribute - The UID attribute object from the Strapi schema.
 * @returns A Zod schema representing the UID field.
 */ const uidToSchema = (attribute)=>{
    const { writable, required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeWithMinMax(minLength, maxLength),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue),
        strapiUtils.maybeReadonly(writable)
    ]);
    return schema.describe('A UID field');
};
/**
 * Converts a BigInteger attribute to a Zod schema for input validation.
 * @param attribute - The BigInteger attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the BigInteger field.
 */ const bigIntegerToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        (schema)=>min !== undefined ? schema.min(min) : schema,
        (schema)=>max !== undefined ? schema.max(max) : schema,
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A biginteger field');
};
/**
 * Converts a blocks attribute to a Zod schema for input validation.
 * @returns A Zod schema for input validation of the blocks field.
 */ const blocksToInputSchema = ()=>{
    // TODO: better support blocks data structure
    return z.z.array(z.z.any()).describe('A blocks field');
};
/**
 * Converts a boolean attribute to a Zod schema for input validation.
 * @param attribute - The Boolean attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the boolean field.
 */ const booleanToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const baseSchema = z.z.enum(constants.BOOLEAN_LITERAL_VALUES).nullable();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A boolean field');
};
/**
 * Converts a component attribute to a Zod schema for input validation.
 * @param attribute - The Component attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the component field.
 */ const componentToInputSchema = (attribute)=>{
    const { required, repeatable, min, max } = attribute;
    const baseSchema = repeatable ? z.z.array(z.z.any()) : z.z.any();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        (schema)=>min !== undefined && schema instanceof z.z.ZodArray ? schema.min(min) : schema,
        (schema)=>max !== undefined && schema instanceof z.z.ZodArray ? schema.max(max) : schema,
        strapiUtils.maybeRequired(required)
    ]);
    return schema.describe('A component field');
};
/**
 * Converts a date attribute to a Zod schema for input validation.
 * @param attribute - The Date attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the date field.
 */ const dateToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A date field');
};
/**
 * Converts a datetime attribute to a Zod schema for input validation.
 * @param attribute - The DateTime attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the datetime field.
 */ const datetimeToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A datetime field');
};
/**
 * Converts a decimal attribute to a Zod schema for input validation.
 * @param attribute - The Decimal attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the decimal field.
 */ const decimalToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.number(), [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A decimal field');
};
/**
 * Converts a dynamic zone attribute to a Zod schema for input validation.
 * @param attribute - The DynamicZone attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the dynamic zone field.
 */ const dynamicZoneToInputSchema = (attribute)=>{
    const { required, min, max } = attribute;
    const baseSchema = z.z.array(z.z.any());
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required)
    ]);
    return schema.describe('A dynamic zone field');
};
/**
 * Converts an email attribute to a Zod schema for input validation.
 * @param attribute - The Email attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the email field.
 */ const emailToInputSchema = (attribute)=>{
    const { required, default: defaultValue, minLength, maxLength } = attribute;
    const baseSchema = z.z.email();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeWithMinMax(minLength, maxLength),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('An email field');
};
/**
 * Converts an enumeration attribute to a Zod schema for input validation.
 * @param attribute - The Enumeration attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the enumeration field.
 */ const enumerationToInputSchema = (attribute)=>{
    const { required, default: defaultValue, enum: enumValues } = attribute;
    const baseSchema = z.z.enum(enumValues);
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('An enum field');
};
/**
 * Converts a float attribute to a Zod schema for input validation.
 * @param attribute - The Float attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the float field.
 */ const floatToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.number(), [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A float field');
};
/**
 * Converts an integer attribute to a Zod schema for input validation.
 * @param attribute - The Integer attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the integer field.
 */ const integerToInputSchema = (attribute)=>{
    const { required, min, max, default: defaultValue } = attribute;
    const baseSchema = z.z.number().int();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeWithMinMax(min, max),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A float field');
};
/**
 * Converts a JSON attribute to a Zod schema for input validation.
 * @param attribute - The JSON attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the JSON field.
 */ const jsonToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.any(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A JSON field');
};
/**
 * Converts a media attribute to a Zod schema for input validation.
 * @param attribute - The Media attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the media field.
 */ const mediaToInputSchema = (attribute)=>{
    const { required, multiple } = attribute;
    const baseSchema = multiple ? z.z.array(z.z.any()) : z.z.any();
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required)
    ]);
    return schema.describe('A media field');
};
/**
 * Converts a relation attribute to a Zod schema for input validation.
 * @param attribute - The Relation attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the relational field.
 */ const relationToInputSchema = (attribute)=>{
    const { required } = attribute;
    const isToMany = strapiUtils.relations.isAnyToMany(attribute);
    const uuid = z.z.string().uuid();
    const baseSchema = isToMany ? z.z.array(uuid) : uuid;
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required)
    ]);
    return schema.describe('A relational field');
};
/**
 * Converts a string, text, rich text, or password attribute to a Zod schema for input validation.
 * @param attribute - The String, Text, RichText, or Password attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the field.
 */ const textToInputSchema = (attribute)=>{
    const { required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeWithMinMax(minLength, maxLength),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe(`A ${attribute.type} field`);
};
/**
 * Converts a time attribute to a Zod schema for input validation.
 * @param attribute - The Time attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the time field.
 */ const timeToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A time field');
};
/**
 * Converts a timestamp attribute to a Zod schema for input validation.
 * @param attribute - The Timestamp attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the timestamp field.
 */ const timestampToInputSchema = (attribute)=>{
    const { required, default: defaultValue } = attribute;
    const baseSchema = z.z.union([
        z.z.string(),
        z.z.number()
    ]);
    const schema = strapiUtils.augmentSchema(baseSchema, [
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A timestamp field');
};
/**
 * Converts a UID attribute to a Zod schema for input validation.
 * @param attribute - The UID attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the UID field.
 */ const uidToInputSchema = (attribute)=>{
    const { required, default: defaultValue, minLength, maxLength } = attribute;
    const schema = strapiUtils.augmentSchema(z.z.string(), [
        strapiUtils.maybeWithMinMax(minLength, maxLength),
        strapiUtils.maybeRequired(required),
        strapiUtils.maybeWithDefault(defaultValue)
    ]);
    return schema.describe('A UID field');
};

exports.bigIntegerToInputSchema = bigIntegerToInputSchema;
exports.bigIntegerToSchema = bigIntegerToSchema;
exports.blocksToInputSchema = blocksToInputSchema;
exports.blocksToSchema = blocksToSchema;
exports.booleanToInputSchema = booleanToInputSchema;
exports.booleanToSchema = booleanToSchema;
exports.componentToInputSchema = componentToInputSchema;
exports.componentToSchema = componentToSchema;
exports.dateToInputSchema = dateToInputSchema;
exports.dateToSchema = dateToSchema;
exports.datetimeToInputSchema = datetimeToInputSchema;
exports.datetimeToSchema = datetimeToSchema;
exports.decimalToInputSchema = decimalToInputSchema;
exports.decimalToSchema = decimalToSchema;
exports.dynamicZoneToInputSchema = dynamicZoneToInputSchema;
exports.dynamicZoneToSchema = dynamicZoneToSchema;
exports.emailToInputSchema = emailToInputSchema;
exports.emailToSchema = emailToSchema;
exports.enumToSchema = enumToSchema;
exports.enumerationToInputSchema = enumerationToInputSchema;
exports.floatToInputSchema = floatToInputSchema;
exports.floatToSchema = floatToSchema;
exports.integerToInputSchema = integerToInputSchema;
exports.integerToSchema = integerToSchema;
exports.jsonToInputSchema = jsonToInputSchema;
exports.jsonToSchema = jsonToSchema;
exports.mediaToInputSchema = mediaToInputSchema;
exports.mediaToSchema = mediaToSchema;
exports.relationToInputSchema = relationToInputSchema;
exports.relationToSchema = relationToSchema;
exports.stringToSchema = stringToSchema;
exports.textToInputSchema = textToInputSchema;
exports.timeToInputSchema = timeToInputSchema;
exports.timeToSchema = timeToSchema;
exports.timestampToInputSchema = timestampToInputSchema;
exports.timestampToSchema = timestampToSchema;
exports.uidToInputSchema = uidToInputSchema;
exports.uidToSchema = uidToSchema;
//# sourceMappingURL=attributes.js.map
