/**
 * @file This module provides a set of functions to convert Strapi schema attributes into Zod schemas.
 * It handles various attribute types, including primitive types, components, dynamic zones, media, and relations.
 * The module also provides functions to create input schemas for these attributes, which are used for validation
 * of incoming data.
 */
import { type Schema, UID } from '@strapi/types';
import { z } from 'zod/v4';
/**
 * Converts a BigInteger attribute to a Zod schema.
 * @param attribute - The BigInteger attribute object from the Strapi schema.
 * @returns A Zod schema representing the BigInteger field.
 */
export declare const bigIntegerToSchema: (attribute: Schema.Attribute.BigInteger) => z.Schema;
/**
 * Converts a blocks attribute to a Zod schema.
 * @returns A Zod schema representing the blocks field.
 */
export declare const blocksToSchema: () => z.Schema;
/**
 * Converts a boolean attribute to a Zod schema.
 * @param attribute - The Boolean attribute object from the Strapi schema.
 * @returns A Zod schema representing the boolean field.
 */
export declare const booleanToSchema: (attribute: Schema.Attribute.Boolean) => z.Schema;
/**
 * Converts a component attribute to a Zod schema.
 * @param attribute - The Component attribute object from the Strapi schema.
 * @returns A Zod schema representing the component field.
 */
export declare const componentToSchema: (attribute: Schema.Attribute.Component<UID.Component, boolean>) => z.Schema;
/**
 * Converts a date attribute to a Zod schema.
 * @param attribute - The Date attribute object from the Strapi schema.
 * @returns A Zod schema representing the date field.
 */
export declare const dateToSchema: (attribute: Schema.Attribute.Date) => z.Schema;
/**
 * Converts a datetime attribute to a Zod schema.
 * @param attribute - The DateTime attribute object from the Strapi schema.
 * @returns A Zod schema representing the datetime field.
 */
export declare const datetimeToSchema: (attribute: Schema.Attribute.DateTime) => z.Schema;
/**
 * Converts a decimal attribute to a Zod schema.
 * @param attribute - The Decimal attribute object from the Strapi schema.
 * @returns A Zod schema representing the decimal field.
 */
export declare const decimalToSchema: (attribute: Schema.Attribute.Decimal) => z.Schema;
/**
 * Converts a dynamic zone attribute to a Zod schema.
 * @param attribute - The DynamicZone attribute object from the Strapi schema.
 * @returns A Zod schema representing the dynamic zone field.
 */
export declare const dynamicZoneToSchema: (attribute: Schema.Attribute.DynamicZone) => z.Schema;
/**
 * Converts an email attribute to a Zod schema.
 * @param attribute - The Email attribute object from the Strapi schema.
 * @returns A Zod schema representing the email field.
 */
export declare const emailToSchema: (attribute: Schema.Attribute.Email) => z.Schema;
/**
 * Converts an enumeration attribute to a Zod schema.
 * @param attribute - The Enumeration attribute object from the Strapi schema.
 * @returns A Zod schema representing the enumeration field.
 */
export declare const enumToSchema: (attribute: Schema.Attribute.Enumeration<string[]>) => z.Schema;
/**
 * Converts a float attribute to a Zod schema.
 * @param attribute - The Float attribute object from the Strapi schema.
 * @returns A Zod schema representing the float field.
 */
export declare const floatToSchema: (attribute: Schema.Attribute.Float) => z.Schema;
/**
 * Converts an integer attribute to a Zod schema.
 * @param attribute - The Integer attribute object from the Strapi schema.
 * @returns A Zod schema representing the integer field.
 */
export declare const integerToSchema: (attribute: Schema.Attribute.Integer) => z.Schema;
/**
 * Converts a JSON attribute to a Zod schema.
 * @param attribute - The JSON attribute object from the Strapi schema.
 * @returns A Zod schema representing the JSON field.
 */
export declare const jsonToSchema: (attribute: Schema.Attribute.JSON) => z.Schema;
/**
 * Converts a media attribute to a Zod schema.
 * @param attribute - The Media attribute object from the Strapi schema.
 * @returns A Zod schema representing the media field.
 */
export declare const mediaToSchema: (attribute: Schema.Attribute.Media<Schema.Attribute.MediaKind | undefined, boolean>) => z.Schema;
/**
 * Converts a relation attribute to a Zod schema.
 * @param attribute - The Relation attribute object from the Strapi schema.
 * @returns A Zod schema representing the relational field.
 */
export declare const relationToSchema: (attribute: Schema.Attribute.Relation) => z.Schema;
/**
 * Converts a string, text, rich text, or password attribute to a Zod schema.
 * @param attribute - The String, Text, RichText, or Password attribute object from the Strapi schema.
 * @returns A Zod schema representing the field.
 */
export declare const stringToSchema: (attribute: Schema.Attribute.String | Schema.Attribute.Text | Schema.Attribute.RichText | Schema.Attribute.Password) => z.Schema;
/**
 * Converts a time attribute to a Zod schema.
 * @param attribute - The Time attribute object from the Strapi schema.
 * @returns A Zod schema representing the time field.
 */
export declare const timeToSchema: (attribute: Schema.Attribute.Time) => z.Schema;
/**
 * Converts a timestamp attribute to a Zod schema.
 * @param attribute - The Timestamp attribute object from the Strapi schema.
 * @returns A Zod schema representing the timestamp field.
 */
export declare const timestampToSchema: (attribute: Schema.Attribute.Timestamp) => z.Schema;
/**
 * Converts a UID attribute to a Zod schema.
 * @param attribute - The UID attribute object from the Strapi schema.
 * @returns A Zod schema representing the UID field.
 */
export declare const uidToSchema: (attribute: Schema.Attribute.UID) => z.Schema;
/**
 * Converts a BigInteger attribute to a Zod schema for input validation.
 * @param attribute - The BigInteger attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the BigInteger field.
 */
export declare const bigIntegerToInputSchema: (attribute: Schema.Attribute.BigInteger) => z.ZodString;
/**
 * Converts a blocks attribute to a Zod schema for input validation.
 * @returns A Zod schema for input validation of the blocks field.
 */
export declare const blocksToInputSchema: () => z.ZodArray<z.ZodAny>;
/**
 * Converts a boolean attribute to a Zod schema for input validation.
 * @param attribute - The Boolean attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the boolean field.
 */
export declare const booleanToInputSchema: (attribute: Schema.Attribute.Boolean) => z.ZodNullable<z.ZodEnum<{
    true: "true";
    0: "0";
    t: "t";
    1: "1";
    f: "f";
    false: "false";
}>>;
/**
 * Converts a component attribute to a Zod schema for input validation.
 * @param attribute - The Component attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the component field.
 */
export declare const componentToInputSchema: (attribute: Schema.Attribute.Component<UID.Component, boolean>) => z.ZodAny | z.ZodArray<z.ZodAny>;
/**
 * Converts a date attribute to a Zod schema for input validation.
 * @param attribute - The Date attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the date field.
 */
export declare const dateToInputSchema: (attribute: Schema.Attribute.Date) => z.ZodString;
/**
 * Converts a datetime attribute to a Zod schema for input validation.
 * @param attribute - The DateTime attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the datetime field.
 */
export declare const datetimeToInputSchema: (attribute: Schema.Attribute.DateTime) => z.ZodString;
/**
 * Converts a decimal attribute to a Zod schema for input validation.
 * @param attribute - The Decimal attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the decimal field.
 */
export declare const decimalToInputSchema: (attribute: Schema.Attribute.Decimal) => z.ZodNumber;
/**
 * Converts a dynamic zone attribute to a Zod schema for input validation.
 * @param attribute - The DynamicZone attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the dynamic zone field.
 */
export declare const dynamicZoneToInputSchema: (attribute: Schema.Attribute.DynamicZone) => z.ZodArray<z.ZodAny>;
/**
 * Converts an email attribute to a Zod schema for input validation.
 * @param attribute - The Email attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the email field.
 */
export declare const emailToInputSchema: (attribute: Schema.Attribute.Email) => z.ZodEmail;
/**
 * Converts an enumeration attribute to a Zod schema for input validation.
 * @param attribute - The Enumeration attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the enumeration field.
 */
export declare const enumerationToInputSchema: (attribute: Schema.Attribute.Enumeration<string[]>) => z.ZodEnum<{
    [x: string]: string;
}>;
/**
 * Converts a float attribute to a Zod schema for input validation.
 * @param attribute - The Float attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the float field.
 */
export declare const floatToInputSchema: (attribute: Schema.Attribute.Float) => z.ZodNumber;
/**
 * Converts an integer attribute to a Zod schema for input validation.
 * @param attribute - The Integer attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the integer field.
 */
export declare const integerToInputSchema: (attribute: Schema.Attribute.Integer) => z.ZodNumber;
/**
 * Converts a JSON attribute to a Zod schema for input validation.
 * @param attribute - The JSON attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the JSON field.
 */
export declare const jsonToInputSchema: (attribute: Schema.Attribute.JSON) => z.ZodAny;
/**
 * Converts a media attribute to a Zod schema for input validation.
 * @param attribute - The Media attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the media field.
 */
export declare const mediaToInputSchema: (attribute: Schema.Attribute.Media<Schema.Attribute.MediaKind | undefined, boolean>) => z.ZodAny | z.ZodArray<z.ZodAny>;
/**
 * Converts a relation attribute to a Zod schema for input validation.
 * @param attribute - The Relation attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the relational field.
 */
export declare const relationToInputSchema: (attribute: Schema.Attribute.Relation) => z.ZodString | z.ZodArray<z.ZodString>;
/**
 * Converts a string, text, rich text, or password attribute to a Zod schema for input validation.
 * @param attribute - The String, Text, RichText, or Password attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the field.
 */
export declare const textToInputSchema: (attribute: Schema.Attribute.String | Schema.Attribute.Text | Schema.Attribute.RichText | Schema.Attribute.Password) => z.ZodString;
/**
 * Converts a time attribute to a Zod schema for input validation.
 * @param attribute - The Time attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the time field.
 */
export declare const timeToInputSchema: (attribute: Schema.Attribute.Time) => z.ZodString;
/**
 * Converts a timestamp attribute to a Zod schema for input validation.
 * @param attribute - The Timestamp attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the timestamp field.
 */
export declare const timestampToInputSchema: (attribute: Schema.Attribute.Timestamp) => z.ZodUnion<readonly [z.ZodString, z.ZodNumber]>;
/**
 * Converts a UID attribute to a Zod schema for input validation.
 * @param attribute - The UID attribute object from the Strapi schema.
 * @returns A Zod schema for input validation of the UID field.
 */
export declare const uidToInputSchema: (attribute: Schema.Attribute.UID) => z.ZodString;
//# sourceMappingURL=attributes.d.ts.map