'use strict';

var fp = require('lodash/fp');
var strapiUtils = require('@strapi/utils');

/**
 *
 * Returns a collection type controller to handle default core-api actions
 */ const createCollectionTypeController = ({ contentType })=>{
    const uid = contentType.uid;
    // TODO: transform into a class
    return {
        /**
     * Retrieve records.
     */ async find (ctx) {
            await this.validateQuery(ctx);
            const sanitizedQuery = await this.sanitizeQuery(ctx);
            const { results, pagination } = await strapi.service(uid).find(sanitizedQuery);
            const sanitizedResults = await this.sanitizeOutput(results, ctx);
            return this.transformResponse(sanitizedResults, {
                pagination
            });
        },
        /**
     * Retrieve a record
     */ async findOne (ctx) {
            const { id } = ctx.params;
            await this.validateQuery(ctx);
            const sanitizedQuery = await this.sanitizeQuery(ctx);
            const entity = await strapi.service(uid).findOne(id, sanitizedQuery);
            const sanitizedEntity = await this.sanitizeOutput(entity, ctx);
            return this.transformResponse(sanitizedEntity);
        },
        /**
     * Create a record.
     */ async create (ctx) {
            await this.validateQuery(ctx);
            const sanitizedQuery = await this.sanitizeQuery(ctx);
            const { body = {} } = ctx.request;
            if (!fp.isObject(body.data)) {
                throw new strapiUtils.errors.ValidationError('Missing "data" payload in the request body');
            }
            await this.validateInput(body.data, ctx);
            const sanitizedInputData = await this.sanitizeInput(body.data, ctx);
            const entity = await strapi.service(uid).create({
                ...sanitizedQuery,
                data: sanitizedInputData
            });
            const sanitizedEntity = await this.sanitizeOutput(entity, ctx);
            ctx.status = 201;
            return this.transformResponse(sanitizedEntity);
        },
        /**
     * Update a record.
     */ async update (ctx) {
            const { id } = ctx.params;
            await this.validateQuery(ctx);
            const sanitizedQuery = await this.sanitizeQuery(ctx);
            const { body = {} } = ctx.request;
            if (!fp.isObject(body.data)) {
                throw new strapiUtils.errors.ValidationError('Missing "data" payload in the request body');
            }
            await this.validateInput(body.data, ctx);
            const sanitizedInputData = await this.sanitizeInput(body.data, ctx);
            const entity = await strapi.service(uid).update(id, {
                ...sanitizedQuery,
                data: sanitizedInputData
            });
            const sanitizedEntity = await this.sanitizeOutput(entity, ctx);
            return this.transformResponse(sanitizedEntity);
        },
        /**
     * Destroy a record.
     */ async delete (ctx) {
            const { id } = ctx.params;
            await this.validateQuery(ctx);
            const sanitizedQuery = await this.sanitizeQuery(ctx);
            await strapi.service(uid).delete(id, sanitizedQuery);
            ctx.status = 204;
        }
    };
};

exports.createCollectionTypeController = createCollectionTypeController;
//# sourceMappingURL=collection-type.js.map
