import os from 'os';
import path from 'path';
import _ from 'lodash';
import { omit } from 'lodash/fp';
import dotenv from 'dotenv';
import { strings } from '@strapi/utils';
import { getConfigUrls, getAbsoluteServerUrl, getAbsoluteAdminUrl } from './urls.mjs';
import loadConfigDir from './config-loader.mjs';
import { getDirs } from './get-dirs.mjs';
import { version } from '../package.json.mjs';

dotenv.config({
    path: process.env.ENV_PATH
});
process.env.NODE_ENV = process.env.NODE_ENV || 'development';
const defaultConfig = {
    server: {
        host: process.env.HOST || os.hostname() || 'localhost',
        port: Number(process.env.PORT) || 1337,
        proxy: false,
        cron: {
            enabled: false
        },
        admin: {
            autoOpen: false
        },
        dirs: {
            public: './public'
        },
        transfer: {
            remote: {
                enabled: true
            }
        },
        logger: {
            updates: {
                enabled: true
            },
            startup: {
                enabled: true
            }
        }
    },
    admin: {},
    api: {
        rest: {
            prefix: '/api'
        }
    }
};
const loadConfiguration = (opts)=>{
    const { appDir, distDir, autoReload = false, serveAdminPanel = true } = opts;
    const pkgJSON = require(path.resolve(appDir, 'package.json'));
    const configDir = path.resolve(distDir || process.cwd(), 'config');
    const rootConfig = {
        launchedAt: Date.now(),
        autoReload,
        environment: process.env.NODE_ENV,
        uuid: _.get(pkgJSON, 'strapi.uuid'),
        installId: _.get(pkgJSON, 'strapi.installId'),
        packageJsonStrapi: _.omit(_.get(pkgJSON, 'strapi', {}), 'uuid'),
        info: {
            ...pkgJSON,
            strapi: version
        },
        admin: {
            serveAdminPanel
        }
    };
    // See packages/core/core/src/domain/module/index.ts for plugin config loading
    const baseConfig = omit('plugins', loadConfigDir(configDir)); // plugin config will be loaded later
    const envDir = path.resolve(configDir, 'env', process.env.NODE_ENV);
    const envConfig = loadConfigDir(envDir);
    const config = _.merge(rootConfig, defaultConfig, baseConfig, envConfig);
    const { serverUrl, adminUrl } = getConfigUrls(config);
    const serverAbsoluteUrl = getAbsoluteServerUrl(config);
    const adminAbsoluteUrl = getAbsoluteAdminUrl(config);
    const sameOrigin = new URL(adminAbsoluteUrl).origin === new URL(serverAbsoluteUrl).origin;
    const adminPath = sameOrigin ? adminUrl.replace(strings.getCommonPath(serverUrl, adminUrl), '') : new URL(adminUrl).pathname;
    _.set(config, 'server.url', serverUrl);
    _.set(config, 'server.absoluteUrl', serverAbsoluteUrl);
    _.set(config, 'admin.url', adminUrl);
    _.set(config, 'admin.path', adminPath);
    _.set(config, 'admin.absoluteUrl', adminAbsoluteUrl);
    _.set(config, 'dirs', getDirs(opts, config));
    return config;
};

export { loadConfiguration };
//# sourceMappingURL=index.mjs.map
