'use strict';

var globalAgent = require('global-agent');
var path = require('path');
var _ = require('lodash');
var fp = require('lodash/fp');
var logger = require('@strapi/logger');
var database = require('@strapi/database');
var tsUtils = require('@strapi/typescript-utils');
var index$8 = require('./configuration/index.js');
var factories = require('./factories.js');
var openBrowser = require('./utils/open-browser.js');
var isInitialized = require('./utils/is-initialized.js');
var index = require('./ee/index.js');
require('package-json');
require('configstore');
require('semver');
require('boxen');
require('chalk');
require('@strapi/utils');
var fetch = require('./utils/fetch.js');
var convertCustomFieldType = require('./utils/convert-custom-field-type.js');
var startupLogger = require('./utils/startup-logger.js');
var transformContentTypesToModels = require('./utils/transform-content-types-to-models.js');
var lifecycles = require('./utils/lifecycles.js');
require('node:path');
var container = require('./container.js');
var fs = require('./services/fs.js');
var eventHub = require('./services/event-hub.js');
var index$3 = require('./services/server/index.js');
var reloader = require('./services/reloader.js');
var index$7 = require('./providers/index.js');
var index$5 = require('./services/entity-service/index.js');
var queryParams = require('./services/query-params.js');
var index$4 = require('./services/entity-validator/index.js');
var requestContext = require('./services/request-context.js');
var index$2 = require('./services/auth/index.js');
var customFields = require('./services/custom-fields.js');
var index$1 = require('./services/content-api/index.js');
var dynamicZones = require('./services/utils/dynamic-zones.js');
var conditionalFields = require('./services/utils/conditional-fields.js');
var features = require('./services/features.js');
var index$6 = require('./services/document-service/index.js');
var contentSourceMaps = require('./services/content-source-maps.js');
var coreStore = require('./services/core-store.js');
var config = require('./services/config.js');
var cleanComponentJoinTable = require('./services/document-service/utils/clean-component-join-table.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var globalAgent__namespace = /*#__PURE__*/_interopNamespaceDefault(globalAgent);

class Strapi extends container.Container {
    get admin() {
        return this.get('admin');
    }
    get EE() {
        return index.isEE;
    }
    get ee() {
        return index;
    }
    get dirs() {
        return this.config.get('dirs');
    }
    get reload() {
        return this.get('reload');
    }
    get db() {
        return this.get('db');
    }
    get requestContext() {
        return this.get('requestContext');
    }
    get customFields() {
        return this.get('customFields');
    }
    get entityValidator() {
        return this.get('entityValidator');
    }
    /**
   * @deprecated `strapi.entityService` will be removed in the next major version
   */ get entityService() {
        return this.get('entityService');
    }
    get documents() {
        return this.get('documents');
    }
    get features() {
        return this.get('features');
    }
    get fetch() {
        return this.get('fetch');
    }
    get cron() {
        return this.get('cron');
    }
    get log() {
        return this.get('logger');
    }
    get startupLogger() {
        return this.get('startupLogger');
    }
    get eventHub() {
        return this.get('eventHub');
    }
    get fs() {
        return this.get('fs');
    }
    get server() {
        return this.get('server');
    }
    get telemetry() {
        return this.get('telemetry');
    }
    get sessionManager() {
        return this.get('sessionManager');
    }
    get store() {
        return this.get('coreStore');
    }
    get config() {
        return this.get('config');
    }
    get services() {
        return this.get('services').getAll();
    }
    service(uid) {
        return this.get('services').get(uid);
    }
    get controllers() {
        return this.get('controllers').getAll();
    }
    controller(uid) {
        return this.get('controllers').get(uid);
    }
    get contentTypes() {
        return this.get('content-types').getAll();
    }
    contentType(name) {
        return this.get('content-types').get(name);
    }
    get components() {
        return this.get('components').getAll();
    }
    get policies() {
        return this.get('policies').getAll();
    }
    policy(name) {
        return this.get('policies').get(name);
    }
    get middlewares() {
        return this.get('middlewares').getAll();
    }
    middleware(name) {
        return this.get('middlewares').get(name);
    }
    get plugins() {
        return this.get('plugins').getAll();
    }
    plugin(name) {
        return this.get('plugins').get(name);
    }
    get hooks() {
        return this.get('hooks').getAll();
    }
    hook(name) {
        return this.get('hooks').get(name);
    }
    get apis() {
        return this.get('apis').getAll();
    }
    api(name) {
        return this.get('apis').get(name);
    }
    get auth() {
        return this.get('auth');
    }
    get contentAPI() {
        return this.get('content-api');
    }
    get sanitizers() {
        return this.get('sanitizers');
    }
    get validators() {
        return this.get('validators');
    }
    async start() {
        try {
            if (!this.isLoaded) {
                await this.load();
            }
            await this.listen();
            return this;
        } catch (error) {
            return this.stopWithError(error);
        }
    }
    // TODO: split into more providers
    registerInternalServices() {
        const config$1 = config.createConfigProvider(this.internal_config, this);
        const logger$1 = logger.createLogger({
            level: 'http',
            ...config$1.get('logger'),
            ...config$1.get('server.logger.config')
        });
        // Instantiate the Strapi container
        this.add('config', ()=>config$1).add('query-params', queryParams(this)).add('content-api', index$1(this)).add('auth', index$2()).add('server', ()=>index$3.createServer(this)).add('fs', ()=>fs(this)).add('eventHub', ()=>eventHub()).add('startupLogger', ()=>startupLogger.createStartupLogger(this)).add('logger', ()=>logger$1).add('fetch', ()=>fetch.createStrapiFetch(this)).add('features', ()=>features.createFeaturesService(this)).add('requestContext', requestContext).add('customFields', customFields(this)).add('entityValidator', index$4).add('entityService', ()=>index$5({
                strapi: this,
                db: this.db
            })).add('documents', ()=>index$6.createDocumentService(this)).add('db', ()=>{
            const tsDir = tsUtils.resolveOutDirSync(this.dirs.app.root);
            const tsMigrationsEnabled = this.config.get('database.settings.useTypescriptMigrations') === true && tsDir;
            const projectDir = tsMigrationsEnabled ? tsDir : this.dirs.app.root;
            return new database.Database(_.merge(this.config.get('database'), {
                logger: logger$1,
                settings: {
                    migrations: {
                        dir: path.join(projectDir, 'database/migrations')
                    }
                }
            }));
        }).add('reload', ()=>reloader.createReloader(this)).add('content-source-maps', ()=>contentSourceMaps.createContentSourceMapsService(this));
    }
    sendStartupTelemetry() {
        // Emit started event.
        // do not await to avoid slower startup
        // This event is anonymous
        this.telemetry.send('didStartServer', {
            groupProperties: {
                database: this.config.get('database.connection.client'),
                plugins: Object.keys(this.plugins),
                numberOfAllContentTypes: _.size(this.contentTypes),
                numberOfComponents: _.size(this.components),
                numberOfDynamicZones: dynamicZones(),
                numberOfConditionalFields: conditionalFields(),
                numberOfCustomControllers: Object.values(this.controllers).filter(// TODO: Fix this at the content API loader level to prevent future types issues
                (controller)=>controller !== undefined && factories.isCustomController(controller)).length,
                environment: this.config.environment
            }
        }).catch(this.log.error);
    }
    async openAdmin({ isInitialized }) {
        const shouldOpenAdmin = this.config.get('environment') === 'development' && this.config.get('admin.autoOpen', true) !== false;
        if (shouldOpenAdmin && !isInitialized) {
            try {
                await openBrowser.openBrowser(this.config);
                this.telemetry.send('didOpenTab');
            } catch (e) {
                this.telemetry.send('didNotOpenTab');
            }
        }
    }
    async postListen() {
        const isInitialized$1 = await isInitialized.isInitialized(this);
        this.startupLogger.logStartupMessage({
            isInitialized: isInitialized$1
        });
        this.log.info('Strapi started successfully');
        this.sendStartupTelemetry();
        this.openAdmin({
            isInitialized: isInitialized$1
        });
    }
    /**
   * Add behaviors to the server
   */ async listen() {
        return new Promise((resolve, reject)=>{
            const onListen = async ()=>{
                try {
                    await this.postListen();
                    resolve();
                } catch (error) {
                    reject(error);
                }
            };
            const listenSocket = this.config.get('server.socket');
            if (listenSocket) {
                this.server.listen(listenSocket, onListen);
            } else {
                const { host, port } = this.config.get('server');
                this.server.listen(port, host, onListen);
            }
        });
    }
    stopWithError(err, customMessage) {
        this.log.debug(`⛔️ Server wasn't able to start properly.`);
        if (customMessage) {
            this.log.error(customMessage);
        }
        this.log.error(err);
        return this.stop();
    }
    stop(exitCode = 1) {
        this.destroy();
        if (this.config.get('autoReload')) {
            process.send?.('stop');
        }
        // Kill process
        process.exit(exitCode);
    }
    async load() {
        await this.register();
        await this.bootstrap();
        this.isLoaded = true;
        return this;
    }
    async register() {
        // @ts-expect-error: init is internal
        this.ee.init(this.dirs.app.root, this.log);
        for (const provider of index$7.providers){
            await provider.register?.(this);
        }
        await this.runPluginsLifecycles(lifecycles.LIFECYCLES.REGISTER);
        await this.runUserLifecycles(lifecycles.LIFECYCLES.REGISTER);
        // NOTE: Swap type customField for underlying data type
        convertCustomFieldType.convertCustomFieldType(this);
        return this;
    }
    async bootstrap() {
        this.configureGlobalProxy();
        const models = [
            ...transformContentTypesToModels.transformContentTypesToModels([
                ...Object.values(this.contentTypes),
                ...Object.values(this.components)
            ], this.db.metadata.identifiers),
            ...this.get('models').get()
        ];
        await this.db.init({
            models
        });
        let oldContentTypes;
        if (await this.db.getSchemaConnection().hasTable(coreStore.coreStoreModel.tableName)) {
            oldContentTypes = await this.store.get({
                type: 'strapi',
                name: 'content_types',
                key: 'schema'
            });
        }
        await this.hook('strapi::content-types.beforeSync').call({
            oldContentTypes,
            contentTypes: this.contentTypes
        });
        // NOTE: commenting out repair logic for now as it is causing relationship loss in some cases
        // will revisit soon in the future PR
        const status = await this.db.schema.sync();
        // // if schemas have changed, run repairs
        if (status === 'CHANGED') {
            await this.db.repair.removeOrphanMorphType({
                pivot: 'component_type'
            });
        }
        const alreadyRanComponentRepair = await this.store.get({
            type: 'strapi',
            key: 'unidirectional-join-table-repair-ran'
        });
        if (!alreadyRanComponentRepair) {
            await this.db.repair.processUnidirectionalJoinTables(cleanComponentJoinTable.cleanComponentJoinTable);
            await this.store.set({
                type: 'strapi',
                key: 'unidirectional-join-table-repair-ran',
                value: true
            });
        }
        if (this.EE) {
            await index.checkLicense({
                strapi: this
            });
        }
        await this.hook('strapi::content-types.afterSync').call({
            oldContentTypes,
            contentTypes: this.contentTypes
        });
        await this.store.set({
            type: 'strapi',
            name: 'content_types',
            key: 'schema',
            value: this.contentTypes
        });
        await this.server.initMiddlewares();
        this.server.initRouting();
        await this.contentAPI.permissions.registerActions();
        await this.runPluginsLifecycles(lifecycles.LIFECYCLES.BOOTSTRAP);
        for (const provider of index$7.providers){
            await provider.bootstrap?.(this);
        }
        await this.runUserLifecycles(lifecycles.LIFECYCLES.BOOTSTRAP);
        return this;
    }
    configureGlobalProxy() {
        const globalProxy = this.config.get('server.proxy.global');
        const httpProxy = this.config.get('server.proxy.http') || globalProxy;
        const httpsProxy = this.config.get('server.proxy.https') || globalProxy;
        if (!httpProxy && !httpsProxy) {
            return;
        }
        globalAgent__namespace.bootstrap();
        if (httpProxy) {
            this.log.info(`Using HTTP proxy: ${httpProxy}`);
            global.GLOBAL_AGENT.HTTP_PROXY = httpProxy;
        }
        if (httpsProxy) {
            this.log.info(`Using HTTPS proxy: ${httpsProxy}`);
            global.GLOBAL_AGENT.HTTPS_PROXY = httpsProxy;
        }
    }
    async destroy() {
        this.log.info('Shutting down Strapi');
        await this.runPluginsLifecycles(lifecycles.LIFECYCLES.DESTROY);
        for (const provider of index$7.providers){
            await provider.destroy?.(this);
        }
        await this.runUserLifecycles(lifecycles.LIFECYCLES.DESTROY);
        await this.server.destroy();
        this.eventHub.destroy();
        await this.db?.destroy();
        process.removeAllListeners();
        // @ts-expect-error: Allow clean delete of global.strapi to allow re-instanciation
        delete global.strapi;
        this.log.info('Strapi has been shut down');
    }
    async runPluginsLifecycles(lifecycleName) {
        // plugins
        await this.get('modules')[lifecycleName]();
    }
    async runUserLifecycles(lifecycleName) {
        // user
        const userLifecycleFunction = this.app && this.app[lifecycleName];
        if (fp.isFunction(userLifecycleFunction)) {
            await userLifecycleFunction({
                strapi: this
            });
        }
    }
    getModel(uid) {
        if (uid in this.contentTypes) {
            return this.contentTypes[uid];
        }
        if (uid in this.components) {
            return this.components[uid];
        }
    }
    /**
   * @deprecated Use `strapi.db.query` instead
   */ query(uid) {
        return this.db.query(uid);
    }
    constructor(opts){
        super(), this.isLoaded = false, this.internal_config = {};
        this.internal_config = index$8.loadConfiguration(opts);
        this.registerInternalServices();
        for (const provider of index$7.providers){
            provider.init?.(this);
        }
    }
}

module.exports = Strapi;
//# sourceMappingURL=Strapi.js.map
