'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var groupBy = require('lodash/groupBy');
var isEqual = require('lodash/isEqual');
var mapValues = require('lodash/mapValues');
var reactIntl = require('react-intl');
var reactRedux = require('react-redux');
var reactRouterDom = require('react-router-dom');
var getTrad = require('../../utils/getTrad.js');
var AutoReloadOverlayBlocker = require('../AutoReloadOverlayBlocker.js');
require('../CTBSession/sessionContext.js');
var useCTBSession = require('../CTBSession/useCTBSession.js');
var useCTBTracking = require('../CTBSession/useCTBTracking.js');
var useFormModalNavigation = require('../FormModalNavigation/useFormModalNavigation.js');
var DataManagerContext = require('./DataManagerContext.js');
var reducer = require('./reducer.js');
var useServerRestartWatcher = require('./useServerRestartWatcher.js');
var cleanData = require('./utils/cleanData.js');
var retrieveComponentsThatHaveComponents = require('./utils/retrieveComponentsThatHaveComponents.js');
var retrieveNestedComponents = require('./utils/retrieveNestedComponents.js');
var retrieveSpecificInfoFromComponents = require('./utils/retrieveSpecificInfoFromComponents.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

const selectState = (state)=>state['content-type-builder_dataManagerProvider'] || reducer.initialState;
const DataManagerProvider = ({ children })=>{
    const dispatch = reactRedux.useDispatch();
    const state = reactRedux.useSelector(selectState);
    const dispatchGuidedTour = strapiAdmin.useGuidedTour('DataManagerProvider', (s)=>s.dispatch);
    const location = reactRouterDom.useLocation();
    const { sessionId: ctbSessionId, regenerateSessionId } = useCTBSession.useCTBSession();
    const { components, contentTypes, reservedNames, initialComponents, initialContentTypes, isLoading } = state.current;
    const { toggleNotification } = strapiAdmin.useNotification();
    const { lockAppWithAutoreload, unlockAppWithAutoreload } = AutoReloadOverlayBlocker.useAutoReloadOverlayBlocker();
    const serverRestartWatcher = useServerRestartWatcher.useServerRestartWatcher();
    const getPlugin = strapiAdmin.useStrapiApp('DataManagerProvider', (state)=>state.getPlugin);
    const plugin = getPlugin('content-type-builder');
    const autoReload = strapiAdmin.useAppInfo('DataManagerProvider', (state)=>state.autoReload);
    const { formatMessage } = reactIntl.useIntl();
    const { trackUsage } = useCTBTracking.useCTBTracking();
    const refetchPermissions = strapiAdmin.useAuth('DataManagerProvider', (state)=>state.refetchPermissions);
    const { onCloseModal } = useFormModalNavigation.useFormModalNavigation();
    const [isSaving, setIsSaving] = React__namespace.useState(false);
    const previousLocationRef = React__namespace.useRef(null);
    const isModified = React__namespace.useMemo(()=>{
        return !(isEqual(components, initialComponents) && isEqual(contentTypes, initialContentTypes));
    }, [
        components,
        contentTypes,
        initialComponents,
        initialContentTypes
    ]);
    const fetchClient = strapiAdmin.useFetchClient();
    const isInDevelopmentMode = autoReload;
    const getDataRef = React__namespace.useRef();
    getDataRef.current = async ()=>{
        try {
            const [schemaResponse, reservedNamesResponse] = await Promise.all([
                fetchClient.get(`/content-type-builder/schema`),
                fetchClient.get(`/content-type-builder/reserved-names`)
            ]);
            const { components, contentTypes } = schemaResponse.data.data;
            dispatch(reducer.actions.init({
                components: mapValues(components, (component)=>({
                        ...component,
                        status: 'UNCHANGED'
                    })),
                contentTypes: mapValues(contentTypes, (contentType)=>({
                        ...contentType,
                        status: 'UNCHANGED'
                    })),
                reservedNames: reservedNamesResponse.data
            }));
            dispatch(reducer.actions.clearHistory());
        } catch (err) {
            console.error({
                err
            });
            toggleNotification({
                type: 'danger',
                message: formatMessage({
                    id: 'notification.error',
                    defaultMessage: 'An error occurred'
                })
            });
        }
    };
    React__namespace.useEffect(()=>{
        getDataRef.current();
        previousLocationRef.current = location.pathname;
        return ()=>{
            // Reload the plugin so the cycle is new again
            dispatch(reducer.actions.reloadPlugin());
        };
    // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);
    // Detect navigation away and back to CTB to regenerate session
    React__namespace.useEffect(()=>{
        const currentPath = location.pathname;
        const previousPath = previousLocationRef.current;
        const CTB_PATH = '/plugins/content-type-builder';
        if (previousPath) {
            const isCTBPath = currentPath.includes(CTB_PATH);
            const wasCTBPath = previousPath.includes(CTB_PATH);
            // Regenerate session when returning to CTB after navigating away
            if (!wasCTBPath && isCTBPath) {
                regenerateSessionId();
            }
        }
        previousLocationRef.current = currentPath;
    }, [
        location.pathname,
        regenerateSessionId
    ]);
    React__namespace.useEffect(()=>{
        if (!autoReload) {
            toggleNotification({
                type: 'info',
                message: formatMessage({
                    id: getTrad.getTrad('notification.info.autoreaload-disable')
                })
            });
        }
    }, [
        autoReload,
        toggleNotification,
        formatMessage
    ]);
    const getAllComponentsThatHaveAComponentInTheirAttributes = (components)=>{
        const composWithCompos = retrieveComponentsThatHaveComponents.retrieveComponentsThatHaveComponents(components);
        return composWithCompos;
    };
    const getAllNestedComponents = (components)=>{
        const appNestedCompo = retrieveNestedComponents.retrieveNestedComponents(components);
        return appNestedCompo;
    };
    const updatePermissions = async ()=>{
        await refetchPermissions();
    };
    const saveSchema = async ()=>{
        setIsSaving(true);
        const PluginForms = plugin?.apis?.forms;
        const mutatedCTs = Object.entries(state.current.contentTypes).reduce((acc, [uid, contentType])=>{
            acc[uid] = PluginForms.mutateContentTypeSchema(contentType, initialContentTypes[uid]);
            return acc;
        }, {});
        const { requestData, trackingEventProperties } = cleanData.stateToRequestData({
            components: state.current.components,
            contentTypes: mutatedCTs
        });
        // Track that the save button was clicked (includes session ID via useCTBTracking)
        trackUsage('willUpdateCTBSchema', {
            ...trackingEventProperties,
            ctbSessionId
        });
        const isSendingContentTypes = Object.keys(state.current.contentTypes).length > 0;
        lockAppWithAutoreload();
        try {
            await fetchClient.post(`/content-type-builder/update-schema`, {
                data: requestData
            });
            if (isSendingContentTypes) {
                // Note: didCreateGuidedTourCollectionType doesn't accept properties
                trackUsage('didCreateGuidedTourCollectionType');
            }
            // Make sure the server has restarted
            await serverRestartWatcher();
            // Generate new session ID after server restart
            regenerateSessionId();
            // refetch and update initial state after the data has been saved
            await getDataRef.current();
            // Update the app's permissions
            await updatePermissions();
        } catch (err) {
            console.error({
                err
            });
            toggleNotification({
                type: 'danger',
                message: formatMessage({
                    id: 'notification.error',
                    defaultMessage: 'An error occurred'
                })
            });
            trackUsage('didUpdateCTBSchema', {
                ...trackingEventProperties,
                success: false,
                ctbSessionId
            });
        } finally{
            setIsSaving(false);
            unlockAppWithAutoreload();
            dispatch(strapiAdmin.adminApi.util.invalidateTags([
                'GuidedTourMeta',
                'HomepageKeyStatistics'
            ]));
            dispatchGuidedTour({
                type: 'set_completed_actions',
                payload: [
                    strapiAdmin.GUIDED_TOUR_REQUIRED_ACTIONS.contentTypeBuilder.createSchema
                ]
            });
            trackUsage('didUpdateCTBSchema', {
                ...trackingEventProperties,
                success: true,
                ctbSessionId
            });
        }
    };
    const componentsThatHaveOtherComponentInTheirAttributes = React__namespace.useMemo(()=>{
        return getAllComponentsThatHaveAComponentInTheirAttributes(components);
    }, [
        components
    ]);
    const nestedComponents = React__namespace.useMemo(()=>{
        return getAllNestedComponents(components);
    }, [
        components
    ]);
    const allComponentsCategories = React__namespace.useMemo(()=>{
        return retrieveSpecificInfoFromComponents.retrieveSpecificInfoFromComponents(components, [
            'category'
        ]);
    }, [
        components
    ]);
    const componentsGroupedByCategory = React__namespace.useMemo(()=>{
        return groupBy(components, 'category');
    }, [
        components
    ]);
    const sortedContentTypesList = React__namespace.useMemo(()=>{
        return cleanData.sortContentType(contentTypes);
    }, [
        contentTypes
    ]);
    const context = {
        componentsThatHaveOtherComponentInTheirAttributes,
        nestedComponents,
        saveSchema,
        reservedNames,
        components,
        contentTypes,
        initialComponents,
        initialContentTypes,
        isSaving,
        isModified,
        isInDevelopmentMode,
        allComponentsCategories,
        componentsGroupedByCategory,
        sortedContentTypesList,
        isLoading,
        addAttribute (payload) {
            dispatch(reducer.actions.addAttribute(payload));
        },
        editAttribute (payload) {
            dispatch(reducer.actions.editAttribute(payload));
        },
        addCustomFieldAttribute (payload) {
            dispatch(reducer.actions.addCustomFieldAttribute(payload));
        },
        editCustomFieldAttribute (payload) {
            dispatch(reducer.actions.editCustomFieldAttribute(payload));
        },
        addCreatedComponentToDynamicZone (payload) {
            dispatch(reducer.actions.addCreatedComponentToDynamicZone(payload));
        },
        createSchema (payload) {
            dispatch(reducer.actions.createSchema(payload));
        },
        createComponentSchema ({ data, uid, componentCategory }) {
            dispatch(reducer.actions.createComponentSchema({
                data,
                uid,
                componentCategory
            }));
        },
        changeDynamicZoneComponents ({ forTarget, targetUid, dynamicZoneTarget, newComponents }) {
            dispatch(reducer.actions.changeDynamicZoneComponents({
                forTarget,
                targetUid,
                dynamicZoneTarget,
                newComponents
            }));
        },
        removeAttribute (payload) {
            if (payload.forTarget === 'contentType') {
                // Note: willDeleteFieldOfContentType doesn't accept properties
                trackUsage('willDeleteFieldOfContentType');
            }
            dispatch(reducer.actions.removeField(payload));
        },
        removeComponentFromDynamicZone (payload) {
            dispatch(reducer.actions.removeComponentFromDynamicZone(payload));
        },
        deleteComponent (uid) {
            const userConfirm = window.confirm(formatMessage({
                id: getTrad.getTrad(`popUpWarning.bodyMessage.component.delete`)
            }));
            if (userConfirm) {
                onCloseModal();
                dispatch(reducer.actions.deleteComponent(uid));
            }
        },
        deleteContentType (uid) {
            const userConfirm = window.confirm(formatMessage({
                id: getTrad.getTrad(`popUpWarning.bodyMessage.contentType.delete`)
            }));
            if (userConfirm) {
                onCloseModal();
                dispatch(reducer.actions.deleteContentType(uid));
            }
        },
        updateComponentSchema ({ data, componentUID }) {
            dispatch(reducer.actions.updateComponentSchema({
                data,
                uid: componentUID
            }));
        },
        updateComponentUid ({ componentUID, newComponentUID }) {
            dispatch(reducer.actions.updateComponentUid({
                uid: componentUID,
                newComponentUID
            }));
        },
        updateSchema (args) {
            dispatch(reducer.actions.updateSchema(args));
        },
        moveAttribute (args) {
            dispatch(reducer.actions.moveAttribute(args));
        },
        applyChange (args) {
            dispatch(reducer.actions.applyChange(args));
        },
        history: {
            undo () {
                dispatch(reducer.actions.undo());
            },
            redo () {
                dispatch(reducer.actions.redo());
            },
            discardAllChanges () {
                dispatch(reducer.actions.discardAll());
            },
            canUndo: state.past.length > 0,
            canRedo: state.future.length > 0,
            canDiscardAll: isModified
        }
    };
    return /*#__PURE__*/ jsxRuntime.jsx(DataManagerContext.DataManagerContext.Provider, {
        value: context,
        children: children
    });
};

module.exports = DataManagerProvider;
//# sourceMappingURL=DataManagerProvider.js.map
