import { jsx } from 'react/jsx-runtime';
import { useState, useMemo, useEffect, useContext, createContext } from 'react';
import { useAIAvailability } from '@strapi/admin/strapi-admin/ee';
import '../../CTBSession/sessionContext.mjs';
import { useCTBTracking } from '../../CTBSession/useCTBTracking.mjs';
import { useDataManager } from '../../DataManager/useDataManager.mjs';
import { FeedbackProvider } from '../FeedbackModal.mjs';
import { useAIChat } from '../hooks/useAIFetch.mjs';
import { useChatTitle } from '../hooks/useChatTitle.mjs';
import { useLastSeenSchemas } from '../hooks/useLastSeenSchemas.mjs';
import { transformCTBToChat } from '../lib/transforms/schemas/fromCTB.mjs';
import { UploadProjectToChatProvider } from '../UploadCodeModal.mjs';
import { UploadFigmaToChatProvider } from '../UploadFigmaModal.mjs';
import { SchemaChatProvider } from './SchemaProvider.mjs';

const ChatContext = /*#__PURE__*/ createContext(undefined);
const generateRandomId = ()=>{
    return Math.random().toString(36).substring(2, 15);
};
const BaseChatProvider = ({ children, defaultOpen = false })=>{
    const [chatId, setChatId] = useState(generateRandomId());
    const [isChatOpen, setIsChatOpen] = useState(defaultOpen);
    const [openCount, setOpenCount] = useState(0);
    const [input, setInput] = useState('');
    // Files
    const [attachments, setAttachments] = useState([]);
    const { trackUsage } = useCTBTracking();
    // DataManager
    const { components, contentTypes } = useDataManager();
    // Last user seen schemas
    const { lastSeenSchemas } = useLastSeenSchemas();
    // Schemas to be sent to the chat
    const schemas = useMemo(()=>{
        return [
            ...Object.values(contentTypes).filter((schema)=>schema.status !== 'REMOVED')// Probably we should still keep this one, not sure
            .filter((schema)=>schema.uid !== 'plugin::users-permissions.user').filter((schema)=>schema.visible).map(transformCTBToChat),
            ...Object.values(components).filter((schema)=>schema.status !== 'REMOVED').map(transformCTBToChat)
        ];
    }, [
        contentTypes,
        components
    ]);
    const { id, messages, sendMessage: _sendMessage, status, stop, ...chat } = useAIChat({
        id: chatId?.toString(),
        experimental_throttle: 100
    });
    /* -------------------------------------------------------------------------------------------------
   * AI SDK chat overrides
   * -----------------------------------------------------------------------------------------------*/ // NOTE: body is using state variables, so they can not be passed as a prop in useChat
    const sendMessage = async (message, options)=>{
        if (status === 'streaming' || status === 'submitted') {
            return;
        }
        return _sendMessage(message, {
            ...options,
            body: {
                ...options?.body,
                schemas,
                metadata: {
                    lastSeenSchemas: lastSeenSchemas.map((schema)=>schema.uid)
                }
            }
        });
    };
    const handleSubmit = (e)=>{
        e.preventDefault();
        if (status === 'streaming' || status === 'submitted') {
            return;
        }
        const readyAttachments = attachments.filter((a)=>a.status !== 'loading');
        if (input.trim().length === 0 && attachments.length === 0) {
            return;
        }
        const files = readyAttachments.map((attachment)=>({
                type: 'file',
                filename: attachment.filename,
                mediaType: attachment.mediaType,
                url: attachment.url
            }));
        sendMessage({
            text: input,
            files
        });
        setInput('');
        setAttachments([]);
    };
    /* -------------------------------------------------------------------------------------------------
   * Chat title
   * -----------------------------------------------------------------------------------------------*/ const { title, generateTitle, resetTitle } = useChatTitle({
        chatId: id,
        messages
    });
    // Automatically generate title when we have at least 1 message (user query)
    useEffect(()=>{
        if (messages.length >= 1 && !title) {
            generateTitle();
        }
    }, [
        messages.length,
        title,
        generateTitle
    ]);
    useEffect(()=>{
        if (status === 'error') {
            trackUsage('didAnswerMessage', {
                successful: false
            });
        } else if (status === 'ready' && messages.length > 0 && messages[messages.length - 1]?.role === 'assistant') {
            trackUsage('didAnswerMessage', {
                successful: true
            });
        }
    }, [
        status,
        messages,
        trackUsage
    ]);
    const isChatAvailable = useAIAvailability();
    return /*#__PURE__*/ jsx(ChatContext.Provider, {
        value: {
            isChatEnabled: isChatAvailable,
            id,
            status,
            stop,
            sendMessage,
            ...chat,
            messages,
            handleSubmit,
            input,
            setInput,
            handleInputChange: (e)=>setInput(e.target.value),
            reset: ()=>{
                stop();
                setChatId(generateRandomId());
                trackUsage('didStartNewChat');
                resetTitle();
            },
            schemas,
            // Chat
            title,
            isChatOpen,
            openChat: ()=>{
                setIsChatOpen(true);
                // if this is the first open, it's a new chat
                if (openCount === 0) {
                    trackUsage('didStartNewChat');
                }
                setOpenCount((prev)=>prev + 1);
            },
            closeChat: ()=>setIsChatOpen(false),
            // Attachments
            attachments,
            setAttachments
        },
        children: children
    });
};
const ChatProvider = ({ children, defaultOpen = false })=>{
    return /*#__PURE__*/ jsx(BaseChatProvider, {
        defaultOpen: defaultOpen,
        children: /*#__PURE__*/ jsx(SchemaChatProvider, {
            children: /*#__PURE__*/ jsx(UploadProjectToChatProvider, {
                children: /*#__PURE__*/ jsx(UploadFigmaToChatProvider, {
                    children: /*#__PURE__*/ jsx(FeedbackProvider, {
                        children: children
                    })
                })
            })
        })
    });
};
function useStrapiChat() {
    const context = useContext(ChatContext);
    if (process.env.NODE_ENV === 'development') {
        // In development, provide a fallback to prevent crashes during hot reloading
        // This allows the app to continue working during HMR updates
        if (context === undefined) {
            console.warn('useStrapiChat was called outside of ChatProvider. This is only allowed in development mode.');
            return {};
        }
    } else if (context === undefined) {
        throw new Error('useStrapiChat must be used within a ChatProvider');
    }
    return context;
}

export { BaseChatProvider, ChatProvider, useStrapiChat };
//# sourceMappingURL=ChatProvider.mjs.map
