'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var ee = require('@strapi/admin/strapi-admin/ee');
require('../../CTBSession/sessionContext.js');
var useCTBTracking = require('../../CTBSession/useCTBTracking.js');
var useDataManager = require('../../DataManager/useDataManager.js');
var FeedbackModal = require('../FeedbackModal.js');
var useAIFetch = require('../hooks/useAIFetch.js');
var useChatTitle = require('../hooks/useChatTitle.js');
var useLastSeenSchemas = require('../hooks/useLastSeenSchemas.js');
var fromCTB = require('../lib/transforms/schemas/fromCTB.js');
var UploadCodeModal = require('../UploadCodeModal.js');
var UploadFigmaModal = require('../UploadFigmaModal.js');
var SchemaProvider = require('./SchemaProvider.js');

const ChatContext = /*#__PURE__*/ React.createContext(undefined);
const generateRandomId = ()=>{
    return Math.random().toString(36).substring(2, 15);
};
const BaseChatProvider = ({ children, defaultOpen = false })=>{
    const [chatId, setChatId] = React.useState(generateRandomId());
    const [isChatOpen, setIsChatOpen] = React.useState(defaultOpen);
    const [openCount, setOpenCount] = React.useState(0);
    const [input, setInput] = React.useState('');
    // Files
    const [attachments, setAttachments] = React.useState([]);
    const { trackUsage } = useCTBTracking.useCTBTracking();
    // DataManager
    const { components, contentTypes } = useDataManager.useDataManager();
    // Last user seen schemas
    const { lastSeenSchemas } = useLastSeenSchemas.useLastSeenSchemas();
    // Schemas to be sent to the chat
    const schemas = React.useMemo(()=>{
        return [
            ...Object.values(contentTypes).filter((schema)=>schema.status !== 'REMOVED')// Probably we should still keep this one, not sure
            .filter((schema)=>schema.uid !== 'plugin::users-permissions.user').filter((schema)=>schema.visible).map(fromCTB.transformCTBToChat),
            ...Object.values(components).filter((schema)=>schema.status !== 'REMOVED').map(fromCTB.transformCTBToChat)
        ];
    }, [
        contentTypes,
        components
    ]);
    const { id, messages, sendMessage: _sendMessage, status, stop, ...chat } = useAIFetch.useAIChat({
        id: chatId?.toString(),
        experimental_throttle: 100
    });
    /* -------------------------------------------------------------------------------------------------
   * AI SDK chat overrides
   * -----------------------------------------------------------------------------------------------*/ // NOTE: body is using state variables, so they can not be passed as a prop in useChat
    const sendMessage = async (message, options)=>{
        if (status === 'streaming' || status === 'submitted') {
            return;
        }
        return _sendMessage(message, {
            ...options,
            body: {
                ...options?.body,
                schemas,
                metadata: {
                    lastSeenSchemas: lastSeenSchemas.map((schema)=>schema.uid)
                }
            }
        });
    };
    const handleSubmit = (e)=>{
        e.preventDefault();
        if (status === 'streaming' || status === 'submitted') {
            return;
        }
        const readyAttachments = attachments.filter((a)=>a.status !== 'loading');
        if (input.trim().length === 0 && attachments.length === 0) {
            return;
        }
        const files = readyAttachments.map((attachment)=>({
                type: 'file',
                filename: attachment.filename,
                mediaType: attachment.mediaType,
                url: attachment.url
            }));
        sendMessage({
            text: input,
            files
        });
        setInput('');
        setAttachments([]);
    };
    /* -------------------------------------------------------------------------------------------------
   * Chat title
   * -----------------------------------------------------------------------------------------------*/ const { title, generateTitle, resetTitle } = useChatTitle.useChatTitle({
        chatId: id,
        messages
    });
    // Automatically generate title when we have at least 1 message (user query)
    React.useEffect(()=>{
        if (messages.length >= 1 && !title) {
            generateTitle();
        }
    }, [
        messages.length,
        title,
        generateTitle
    ]);
    React.useEffect(()=>{
        if (status === 'error') {
            trackUsage('didAnswerMessage', {
                successful: false
            });
        } else if (status === 'ready' && messages.length > 0 && messages[messages.length - 1]?.role === 'assistant') {
            trackUsage('didAnswerMessage', {
                successful: true
            });
        }
    }, [
        status,
        messages,
        trackUsage
    ]);
    const isChatAvailable = ee.useAIAvailability();
    return /*#__PURE__*/ jsxRuntime.jsx(ChatContext.Provider, {
        value: {
            isChatEnabled: isChatAvailable,
            id,
            status,
            stop,
            sendMessage,
            ...chat,
            messages,
            handleSubmit,
            input,
            setInput,
            handleInputChange: (e)=>setInput(e.target.value),
            reset: ()=>{
                stop();
                setChatId(generateRandomId());
                trackUsage('didStartNewChat');
                resetTitle();
            },
            schemas,
            // Chat
            title,
            isChatOpen,
            openChat: ()=>{
                setIsChatOpen(true);
                // if this is the first open, it's a new chat
                if (openCount === 0) {
                    trackUsage('didStartNewChat');
                }
                setOpenCount((prev)=>prev + 1);
            },
            closeChat: ()=>setIsChatOpen(false),
            // Attachments
            attachments,
            setAttachments
        },
        children: children
    });
};
const ChatProvider = ({ children, defaultOpen = false })=>{
    return /*#__PURE__*/ jsxRuntime.jsx(BaseChatProvider, {
        defaultOpen: defaultOpen,
        children: /*#__PURE__*/ jsxRuntime.jsx(SchemaProvider.SchemaChatProvider, {
            children: /*#__PURE__*/ jsxRuntime.jsx(UploadCodeModal.UploadProjectToChatProvider, {
                children: /*#__PURE__*/ jsxRuntime.jsx(UploadFigmaModal.UploadFigmaToChatProvider, {
                    children: /*#__PURE__*/ jsxRuntime.jsx(FeedbackModal.FeedbackProvider, {
                        children: children
                    })
                })
            })
        })
    });
};
function useStrapiChat() {
    const context = React.useContext(ChatContext);
    if (process.env.NODE_ENV === 'development') {
        // In development, provide a fallback to prevent crashes during hot reloading
        // This allows the app to continue working during HMR updates
        if (context === undefined) {
            console.warn('useStrapiChat was called outside of ChatProvider. This is only allowed in development mode.');
            return {};
        }
    } else if (context === undefined) {
        throw new Error('useStrapiChat must be used within a ChatProvider');
    }
    return context;
}

exports.BaseChatProvider = BaseChatProvider;
exports.ChatProvider = ChatProvider;
exports.useStrapiChat = useStrapiChat;
//# sourceMappingURL=ChatProvider.js.map
