'use strict';

var jsxRuntime = require('react/jsx-runtime');
var designSystem = require('@strapi/design-system');
var Icons = require('@strapi/icons');
var Markdown = require('react-markdown');
var styledComponents = require('styled-components');
var FeedbackModal = require('../../FeedbackModal.js');
var useFeedback = require('../../hooks/useFeedback.js');
var AnimatedBox = require('../AnimatedBox.js');
var AttachmentPreview = require('../Attachments/AttachmentPreview.js');
var Marker = require('./Marker.js');

const MarkdownStyles = styledComponents.styled(designSystem.Typography)`
  max-width: 65ch;
  margin: 0 auto;
  overflow-wrap: anywhere;
  word-break: break-word;

  h1,
  h2,
  h3,
  h4,
  h5,
  h6 {
    margin-top: 1.25em;
    margin-bottom: 0.5em;
    font-weight: bold;
  }

  p {
    margin-bottom: 1em;
  }

  ul,
  ol {
    padding-left: 1.5em; /* indentation for bullet points */
    margin-bottom: 1em;
  }

  li {
    margin-bottom: 0.5em;
    list-style-type: disc; /* or whatever style you prefer */
  }

  strong {
    font-weight: bold;
  }

  /* code blocks, blockquotes, etc. */
  code {
    background-color: ${({ theme })=>theme.colors.neutral100};
    padding: 0.2em 0.4em;
    border-radius: ${({ theme })=>theme.borderRadius};
    border-color: ${({ theme })=>theme.colors.neutral150};
    border-style: solid;
    font-family: 'SF Mono', SFMono-Regular, ui-monospace, 'DejaVu Sans Mono', Menlo, Consolas,
      monospace;
  }

  /* links */
  a {
    color: ${({ theme })=>theme.colors.primary500};
    background-color: ${({ theme })=>theme.colors.neutral100};
    padding: 0.2em 0.4em;
    border-radius: ${({ theme })=>theme.borderRadius};
    border-color: ${({ theme })=>theme.colors.neutral150};
    border-style: solid;
    text-decoration: none;

    &:hover {
      text-decoration: underline;
    }
  }
`;
const UserMessageTypography = styledComponents.styled(designSystem.Typography)`
  overflow-wrap: anywhere;
  word-break: break-word;
`;
const isSchemaToolPart = (part)=>part && typeof part === 'object' && part.type === 'tool-schemaGenerationTool';
const capitalize = (s)=>s ? s.charAt(0).toUpperCase() + s.slice(1) : '';
const getSchemaLink = (schema)=>{
    const isComponent = (schema.kind ?? schema.modelType) === 'component';
    if (!schema.uid) return undefined;
    return isComponent ? `/plugins/content-type-builder/component-categories/${schema.category ?? ''}/${schema.uid}` : `/plugins/content-type-builder/content-types/${schema.uid}`;
};
const toMarkerFromSchemaTool = (part)=>{
    const outSchemas = part.output?.schemas ?? [];
    const inSchemas = part.input?.schemas ?? [];
    const schemas = outSchemas.length ? outSchemas : inSchemas;
    const numSchemas = schemas.length;
    const state = part.output ? part.output.error ? 'error' : 'success' : 'loading';
    const steps = schemas.map((schema, index)=>({
            id: `${part.toolCallId ?? 'schemaGenerationTool'}-${schema.uid ?? schema.name ?? index}`,
            description: capitalize(schema.name ?? schema.uid ?? 'Schema'),
            status: schema.action === 'create' || schema.action === 'update' || schema.action === 'remove' ? schema.action : 'update',
            link: getSchemaLink(schema)
        }));
    const title = state === 'success' ? `Updated ${numSchemas} schema${numSchemas === 1 ? '' : 's'}` : state === 'error' ? `Failed to update schema${numSchemas === 1 ? '' : 's'}` : 'Updating schemas';
    return {
        type: 'marker',
        title,
        state,
        steps
    };
};
const MessageContent = ({ part })=>{
    if (part.type === 'text') {
        return /*#__PURE__*/ jsxRuntime.jsx(MarkdownStyles, {
            children: /*#__PURE__*/ jsxRuntime.jsx(Markdown, {
                components: {
                    a: ({ node, ...props })=>/*#__PURE__*/ jsxRuntime.jsx("a", {
                            target: "_blank",
                            rel: "noopener noreferrer",
                            ...props
                        })
                },
                children: part.text
            })
        });
    }
    if (isSchemaToolPart(part)) {
        const marker = toMarkerFromSchemaTool(part);
        return /*#__PURE__*/ jsxRuntime.jsx(Marker.Marker, {
            ...marker
        });
    }
    return null;
};
const UserMessage = ({ message })=>{
    const hasText = message.parts.some((content)=>content.type === 'text' && content.text.trim() !== '');
    const attachments = message.parts.filter((content)=>content.type === 'file');
    return /*#__PURE__*/ jsxRuntime.jsxs(AnimatedBox.AnimatedBox, {
        as: designSystem.Flex,
        direction: "column",
        alignItems: "flex-end",
        style: {
            alignSelf: 'flex-end'
        },
        gap: 2,
        maxWidth: "80%",
        children: [
            hasText ? /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                background: "neutral150",
                borderStyle: "none",
                padding: [
                    '10px',
                    '16px'
                ],
                hasRadius: true,
                children: message.parts.map((content, index)=>{
                    if (content.type !== 'text') return null;
                    return /*#__PURE__*/ jsxRuntime.jsx(UserMessageTypography, {
                        children: content.text
                    }, index);
                })
            }) : null,
            attachments.map((attachment, idx)=>/*#__PURE__*/ jsxRuntime.jsx(AttachmentPreview.AttachmentPreview, {
                    // eslint-disable-next-line @typescript-eslint/no-explicit-any
                    attachment: {
                        ...attachment,
                        status: 'ready'
                    }
                }, `${attachment.type === 'file' ? attachment.filename : attachment.type}-${idx}`))
        ]
    });
};
const AssistantMessage = ({ message, isLoading })=>{
    const { upvoteMessage } = useFeedback.useFeedback();
    const { openFeedbackModal } = FeedbackModal.useFeedbackModal();
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Box, {
        style: {
            alignSelf: 'flex-start'
        },
        maxWidth: "90%",
        children: [
            message.parts.map((content, index)=>/*#__PURE__*/ jsxRuntime.jsx(MessageContent, {
                    part: content
                }, index)),
            isLoading ? /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                gap: 1,
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                        label: "Upvote",
                        size: "XS",
                        variant: "ghost",
                        onClick: ()=>upvoteMessage(message.id),
                        children: /*#__PURE__*/ jsxRuntime.jsx(Icons.ThumbUp, {})
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                        label: "Downvote",
                        size: "XS",
                        variant: "ghost",
                        // For downvoting, user must provide specific feedback
                        onClick: ()=>openFeedbackModal(message.id),
                        children: /*#__PURE__*/ jsxRuntime.jsx(Icons.ThumbDown, {})
                    })
                ]
            }) : null
        ]
    });
};
const ChatMessage = ({ message, isLoading })=>{
    /**
   * IMPORTANT: Messages are rendered using react-markdown (heavy compute)
   * Component re-renders on each message update, but AI SDK v5 provides
   * throttling (experimental_throttle: 100ms) which batches updates and reduces
   * re-render frequency during streaming.
   */ if (message.role === 'user') {
        return /*#__PURE__*/ jsxRuntime.jsx(UserMessage, {
            message: message
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsx(AssistantMessage, {
        message: message
    });
};

exports.ChatMessage = ChatMessage;
//# sourceMappingURL=Message.js.map
