'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var designSystem = require('@strapi/design-system');
var ImagePreview = require('./components/ImagePreview.js');
var StepModal = require('./components/StepModal.js');
var useAttachments = require('./hooks/useAttachments.js');
var useFigmaUpload = require('./hooks/useFigmaUpload.js');
var useTranslations = require('./hooks/useTranslations.js');
var ChatProvider = require('./providers/ChatProvider.js');

const UploadFigmaContext = /*#__PURE__*/ React.createContext({
    isFigmaUploadOpen: false,
    submitOnFinish: false,
    openFigmaUpload: ()=>{},
    closeFigmaUpload: ()=>{}
});
const useUploadFigmaToChat = ()=>{
    const context = React.useContext(UploadFigmaContext);
    if (!context) {
        throw new Error('useUploadFigmaToChat must be used within an UploadFigmaToChatProvider');
    }
    return context;
};
const UploadFigmaToChatProvider = ({ children })=>{
    const [isFigmaUploadOpen, setIsFigmaUploadOpen] = React.useState(false); // Default to false
    const [submitOnFinish, setSubmitOnFinish] = React.useState(false);
    const openFigmaUpload = (submitOnFinishParam)=>{
        setIsFigmaUploadOpen(true);
        setSubmitOnFinish(submitOnFinishParam ?? false);
    };
    const closeFigmaUpload = ()=>setIsFigmaUploadOpen(false);
    return /*#__PURE__*/ jsxRuntime.jsxs(UploadFigmaContext.Provider, {
        value: {
            isFigmaUploadOpen,
            submitOnFinish,
            openFigmaUpload,
            closeFigmaUpload
        },
        children: [
            isFigmaUploadOpen && /*#__PURE__*/ jsxRuntime.jsx(UploadFigmaModal, {}),
            children
        ]
    });
};
const FigmaUrlInputStep = ({ figmaUrl, setFigmaUrl })=>{
    const { t } = useTranslations.useTranslations();
    const { isLoading } = StepModal.useStepModal();
    const [showingTokenInput, setShowingTokenInput] = React.useState(!useFigmaUpload.hasFigmaToken());
    const [figmaToken, setFigmaToken] = React.useState(useFigmaUpload.getFigmaToken);
    // Handle saving token and returning to URL input
    const handleSaveToken = ()=>{
        if (figmaToken.trim()) {
            useFigmaUpload.saveFigmaToken(figmaToken);
            setShowingTokenInput(false);
        }
    };
    // If we need to show token step, render the token input
    if (showingTokenInput) {
        return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
            direction: "column",
            gap: 6,
            alignItems: "start",
            children: [
                /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                    direction: "column",
                    gap: 2,
                    alignItems: "start",
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                            variant: "beta",
                            children: t('chat.figma-upload.token-title', 'Enter Figma API Token')
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                            variant: "omega",
                            textColor: "neutral600",
                            children: t('chat.figma-upload.token-description', 'To access your Figma designs, you need to provide a personal access token. This will be stored securely in your browser.')
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Link, {
                            href: "https://help.figma.com/hc/en-us/articles/8085703771159-Manage-personal-access-tokens",
                            isExternal: true,
                            children: t('chat.figma-upload.token-help', 'How to get a Figma API token')
                        })
                    ]
                }),
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                    width: "100%",
                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.TextInput, {
                        name: "figma-token",
                        placeholder: t('chat.figma-upload.token-placeholder', 'Enter Figma API token'),
                        "aria-label": t('chat.figma-upload.token-placeholder', 'Enter Figma API token'),
                        value: figmaToken,
                        onChange: (e)=>setFigmaToken(e.target.value),
                        width: "100%",
                        disabled: isLoading,
                        type: "password"
                    })
                }),
                /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                    gap: 2,
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                            onClick: handleSaveToken,
                            disabled: !figmaToken.trim(),
                            variant: "secondary",
                            children: t('chat.figma-upload.save-token', 'Save token')
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                            onClick: ()=>setShowingTokenInput(false),
                            variant: "tertiary",
                            disabled: !useFigmaUpload.hasFigmaToken(),
                            children: t('chat.figma-upload.cancel', 'Cancel')
                        })
                    ]
                })
            ]
        });
    }
    // Otherwise render the URL input
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
        direction: "column",
        gap: 6,
        alignItems: "start",
        children: [
            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                direction: "column",
                gap: 2,
                alignItems: "start",
                width: "100%",
                children: [
                    /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                        justifyContent: "space-between",
                        alignItems: "center",
                        width: "100%",
                        children: [
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                variant: "beta",
                                children: t('chat.figma-upload.title', 'Import Figma Design')
                            }),
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                onClick: ()=>setShowingTokenInput(true),
                                variant: "tertiary",
                                size: "S",
                                children: t('chat.figma-upload.edit-token', 'Edit API token')
                            })
                        ]
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        variant: "omega",
                        textColor: "neutral600",
                        children: t('chat.figma-upload.description', 'Ask to turn your designs into schemas by attaching a link to one or multiple frames in your Figma files. (Max 15 frames)')
                    })
                ]
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                width: "100%",
                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.TextInput, {
                    name: "figma-url",
                    placeholder: t('chat.figma-upload.url-placeholder', 'Enter Figma URL'),
                    "aria-label": t('chat.figma-upload.url-placeholder', 'Enter Figma URL'),
                    value: figmaUrl,
                    onChange: (e)=>setFigmaUrl(e.target.value),
                    width: "100%",
                    disabled: isLoading,
                    type: "url"
                })
            })
        ]
    });
};
const FigmaImageDisplayStep = ({ images, selectedImages, setSelectedImages })=>{
    const { t } = useTranslations.useTranslations();
    // Handle select/deselect all
    const toggleSelectAll = ()=>{
        if (selectedImages.length === images.length) {
            // Deselect all if all or max allowed are selected
            setSelectedImages([]);
        } else {
            // Select all images up to the max limit
            const allImageIds = images.map((img)=>img.id);
            setSelectedImages(allImageIds);
        }
    };
    if (images.length === 0) {
        return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Flex, {
            direction: "column",
            gap: 4,
            alignItems: "center",
            padding: 4,
            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                variant: "omega",
                children: t('chat.figma-upload.no-images', 'No frames found in the Figma file.')
            })
        });
    }
    // Handle individual frame selection
    const handleFrameSelection = (frameId)=>{
        const newSelection = selectedImages.includes(frameId) ? selectedImages.filter((id)=>id !== frameId) : [
            ...selectedImages,
            frameId
        ];
        setSelectedImages(newSelection);
    };
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
        direction: "column",
        gap: 4,
        alignItems: "start",
        width: "100%",
        height: "min(45vh, 400px)",
        children: [
            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                justifyContent: "space-between",
                width: "100%",
                alignItems: "center",
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        variant: "beta",
                        children: t('chat.figma-upload.select-images', 'Select Frames to Import')
                    }),
                    /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                        gap: 3,
                        alignItems: "center",
                        children: [
                            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Typography, {
                                children: [
                                    selectedImages.length,
                                    " of ",
                                    images.length,
                                    " selected"
                                ]
                            }),
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                onClick: toggleSelectAll,
                                type: "button",
                                variant: "secondary",
                                children: selectedImages.length === images.length ? t('chat.figma-upload.deselect-all', 'Deselect All') : t('chat.figma-upload.select-all', 'Select All')
                            })
                        ]
                    })
                ]
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                paddingRight: 4,
                width: "100%",
                style: {
                    overflowY: 'auto'
                },
                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Grid.Root, {
                    gap: 4,
                    children: images.map((frame, index)=>{
                        const isSelected = selectedImages.includes(frame.id);
                        return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Grid.Item, {
                            col: 6,
                            xs: 12,
                            padding: '1px',
                            children: /*#__PURE__*/ jsxRuntime.jsx(ImagePreview.ImagePreview, {
                                imageUrl: frame.url,
                                imageName: frame.filename || `Frame ${index + 1}`,
                                selected: isSelected,
                                onSelect: ()=>handleFrameSelection(frame.id)
                            })
                        }, frame.id);
                    })
                })
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Modal
 * -----------------------------------------------------------------------------------------------*/ const UploadFigmaModal = ()=>{
    const [figmaUrl, setFigmaUrl] = React.useState('');
    const [figmaImages, setFigmaImages] = React.useState([]);
    const [selectedImages, setSelectedImages] = React.useState([]);
    const { t } = useTranslations.useTranslations();
    const { addAttachments } = useAttachments.useAttachments();
    const { isFigmaUploadOpen, closeFigmaUpload, submitOnFinish } = useUploadFigmaToChat();
    const { input, setInput, setMessages, sendMessage, openChat } = ChatProvider.useStrapiChat();
    const { processFigmaUrl, isLoading, error } = useFigmaUpload.useFigmaUpload({
        onSuccess: (images)=>{
            setFigmaImages(images);
            // Initialize with first 15 images selected
            const initialSelection = images.slice(0, 15).map((img)=>img.id);
            setSelectedImages(initialSelection);
        }
    });
    const handleImportStep = async ()=>{
        await processFigmaUrl(figmaUrl);
        return true;
    };
    // Validate if the URL is a valid Figma URL
    const isValidFigmaUrl = (url)=>{
        if (!url) return false;
        try {
            const urlObj = new URL(url);
            return urlObj.hostname === 'www.figma.com' || urlObj.hostname === 'figma.com';
        } catch (e) {
            return false;
        }
    };
    const handleCancel = ()=>{
        // Reset all state on cancel
        setFigmaUrl('');
        setFigmaImages([]);
        setSelectedImages([]);
        closeFigmaUpload();
    };
    const handleComplete = ()=>{
        // Only attach the selected images
        const selectedFigmaImages = figmaImages.filter((img)=>selectedImages.includes(img.id));
        if (selectedFigmaImages.length === 0) {
            closeFigmaUpload();
            return;
        }
        // Ensure chat is opened
        openChat();
        if (submitOnFinish) {
            // Auto-submit a message to chat with attachments
            sendMessage({
                role: 'user',
                parts: [
                    {
                        type: 'text',
                        text: 'Create schemas from the attached images'
                    },
                    ...selectedFigmaImages
                ]
            });
            closeFigmaUpload();
        } else {
            // If input is empty, set a predefined message
            if (!input) {
                setInput('Create schemas from the attached images');
            }
            addAttachments(selectedFigmaImages);
            closeFigmaUpload();
        }
    };
    return /*#__PURE__*/ jsxRuntime.jsxs(StepModal.StepModal, {
        open: isFigmaUploadOpen,
        onOpenChange: (isOpen)=>{
            if (!isOpen) handleCancel();
        },
        title: t('chat.figma-upload.header', 'Import from Figma'),
        onCancel: handleCancel,
        onComplete: handleComplete,
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(StepModal.StepModal.Step, {
                title: t('chat.figma-upload.step1-title', 'Enter Figma URL'),
                nextLabel: t('chat.figma-upload.import-button', 'Import'),
                cancelLabel: t('form.button.cancel', 'Cancel'),
                disableNext: !figmaUrl || isLoading || !isValidFigmaUrl(figmaUrl),
                onNext: handleImportStep,
                children: /*#__PURE__*/ jsxRuntime.jsx(FigmaUrlInputStep, {
                    figmaUrl: figmaUrl,
                    setFigmaUrl: setFigmaUrl,
                    error: error
                })
            }),
            /*#__PURE__*/ jsxRuntime.jsx(StepModal.StepModal.Step, {
                title: t('chat.figma-upload.step2-title', 'Preview Images'),
                nextLabel: t('form.button.finish', 'Finish'),
                backLabel: t('form.button.back', 'Back'),
                disableNext: selectedImages.length === 0,
                children: /*#__PURE__*/ jsxRuntime.jsx(FigmaImageDisplayStep, {
                    images: figmaImages,
                    selectedImages: selectedImages,
                    setSelectedImages: setSelectedImages
                })
            })
        ]
    });
};

exports.UploadFigmaModal = UploadFigmaModal;
exports.UploadFigmaToChatProvider = UploadFigmaToChatProvider;
exports.useUploadFigmaToChat = useUploadFigmaToChat;
//# sourceMappingURL=UploadFigmaModal.js.map
