import { jsxs, jsx } from 'react/jsx-runtime';
import { useState, createContext, useContext } from 'react';
import { useNotification } from '@strapi/admin/strapi-admin';
import { Flex, Modal, Typography, Grid, Box, Textarea, Button, Checkbox } from '@strapi/design-system';
import { styled } from 'styled-components';
import { useFeedback } from './hooks/useFeedback.mjs';
import { useTranslations } from './hooks/useTranslations.mjs';

const FeedbackModalContext = /*#__PURE__*/ createContext({
    isFeedbackModalOpen: false,
    currentMessageId: null,
    openFeedbackModal: ()=>{},
    closeFeedbackModal: ()=>{}
});
const useFeedbackModal = ()=>useContext(FeedbackModalContext);
const FeedbackProvider = ({ children })=>{
    const [isFeedbackModalOpen, setIsFeedbackModalOpen] = useState(false);
    const [currentMessageId, setCurrentMessageId] = useState(null);
    const openFeedbackModal = (messageId)=>{
        setCurrentMessageId(messageId);
        setIsFeedbackModalOpen(true);
    };
    const closeFeedbackModal = ()=>{
        setIsFeedbackModalOpen(false);
        setCurrentMessageId(null);
    };
    return /*#__PURE__*/ jsxs(FeedbackModalContext.Provider, {
        value: {
            isFeedbackModalOpen,
            currentMessageId,
            openFeedbackModal,
            closeFeedbackModal
        },
        children: [
            isFeedbackModalOpen && currentMessageId && /*#__PURE__*/ jsx(FeedbackModal, {}),
            children
        ]
    });
};
const FeedbackOptionWrapper = styled(Flex)`
  &:hover {
    background-color: ${({ theme })=>theme.colors.neutral100};
  }

  &.selected {
    background-color: ${({ theme })=>theme.colors.primary100};
    border-color: ${({ theme })=>theme.colors.primary200};
  }
`;
const FeedbackOption = ({ id, label, selected, onClick })=>{
    return /*#__PURE__*/ jsxs(FeedbackOptionWrapper, {
        className: selected ? 'selected' : '',
        justifyContent: "space-between",
        hasRadius: true,
        width: "100%",
        cursor: "pointer",
        borderColor: "neutral200",
        padding: 3,
        gap: 3,
        onClick: ()=>onClick(id),
        children: [
            /*#__PURE__*/ jsx(Typography, {
                variant: "omega",
                fontWeight: "bold",
                children: label
            }),
            /*#__PURE__*/ jsx(Checkbox, {
                name: `feedback-${id}`,
                value: id,
                checked: selected,
                onChange: ()=>onClick(id)
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Modal
 * -----------------------------------------------------------------------------------------------*/ const FeedbackModal = ()=>{
    const [feedbackText, setFeedbackText] = useState('');
    const [selectedReasons, setSelectedReasons] = useState([]);
    const { t } = useTranslations();
    const { toggleNotification } = useNotification();
    const { closeFeedbackModal, currentMessageId } = useFeedbackModal();
    const { downvoteMessage, isPending } = useFeedback();
    const feedbackReasons = [
        {
            id: 'invalid_schema',
            label: t('chat.feedback.reason.invalid_schema', 'Invalid schema')
        },
        {
            id: 'bad_recommendation',
            label: t('chat.feedback.reason.bad_recommendation', 'Bad recommendation')
        },
        {
            id: 'slow',
            label: t('chat.feedback.reason.slow', 'Slow')
        },
        {
            id: 'instructions_ignored',
            label: t('chat.feedback.reason.instructions_ignored', 'Instructions ignored')
        },
        {
            id: 'being_lazy',
            label: t('chat.feedback.reason.being_lazy', 'Being lazy')
        },
        {
            id: 'other',
            label: t('chat.feedback.reason.other', 'Other')
        }
    ];
    const handleReasonSelect = (id)=>{
        setSelectedReasons((prevSelected)=>{
            // If already selected, remove it
            if (prevSelected.includes(id)) {
                return prevSelected.filter((reasonId)=>reasonId !== id);
            }
            // Otherwise add it
            return [
                ...prevSelected,
                id
            ];
        });
    };
    const handleSubmitFeedback = async ()=>{
        if (selectedReasons.length === 0) return;
        try {
            await downvoteMessage(currentMessageId, feedbackText, selectedReasons);
            setFeedbackText('');
            setSelectedReasons([]);
            closeFeedbackModal();
        } catch (error) {
            toggleNotification({
                type: 'danger',
                message: t('chat.feedback.error', 'An error occurred while submitting your feedback')
            });
        }
    };
    return /*#__PURE__*/ jsx(Modal.Root, {
        open: true,
        onOpenChange: closeFeedbackModal,
        children: /*#__PURE__*/ jsxs(Modal.Content, {
            children: [
                /*#__PURE__*/ jsx(Modal.Header, {
                    children: /*#__PURE__*/ jsx(Typography, {
                        variant: "omega",
                        fontWeight: "bold",
                        children: t('chat.feedback.title', 'Give feedback')
                    })
                }),
                /*#__PURE__*/ jsx(Modal.Body, {
                    children: /*#__PURE__*/ jsxs(Flex, {
                        direction: "column",
                        alignItems: "start",
                        gap: 6,
                        width: "100%",
                        children: [
                            /*#__PURE__*/ jsxs(Flex, {
                                direction: "column",
                                alignItems: "start",
                                gap: 2,
                                children: [
                                    /*#__PURE__*/ jsx(Typography, {
                                        variant: "beta",
                                        fontWeight: "bold",
                                        children: t('chat.feedback.title', 'Give feedback')
                                    }),
                                    /*#__PURE__*/ jsx(Typography, {
                                        variant: "omega",
                                        children: t('chat.feedback.subtitle', 'Provide additional feedback on this message. Select all that apply.')
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsx(Grid.Root, {
                                width: "100%",
                                gap: 2,
                                children: feedbackReasons.map((reason)=>/*#__PURE__*/ jsx(Grid.Item, {
                                        col: 6,
                                        xs: 12,
                                        children: /*#__PURE__*/ jsx(FeedbackOption, {
                                            id: reason.id,
                                            label: reason.label,
                                            selected: selectedReasons.includes(reason.id),
                                            onClick: handleReasonSelect
                                        }, reason.id)
                                    }, reason.id))
                            }),
                            /*#__PURE__*/ jsxs(Flex, {
                                direction: "column",
                                gap: 2,
                                width: "100%",
                                alignItems: "start",
                                children: [
                                    /*#__PURE__*/ jsx(Typography, {
                                        variant: "omega",
                                        children: t('chat.feedback.comment.label', 'How can we improve? (optional)')
                                    }),
                                    /*#__PURE__*/ jsx(Box, {
                                        width: "100%",
                                        children: /*#__PURE__*/ jsx(Textarea, {
                                            name: "feedback",
                                            placeholder: t('chat.feedback.placeholder', 'Your feedback...'),
                                            onChange: (e)=>setFeedbackText(e.target.value),
                                            value: feedbackText
                                        })
                                    })
                                ]
                            })
                        ]
                    })
                }),
                /*#__PURE__*/ jsxs(Modal.Footer, {
                    children: [
                        /*#__PURE__*/ jsx(Modal.Close, {
                            children: /*#__PURE__*/ jsx(Button, {
                                variant: "tertiary",
                                onClick: closeFeedbackModal,
                                children: t('form.button.cancel', 'Cancel')
                            })
                        }),
                        /*#__PURE__*/ jsx(Button, {
                            onClick: handleSubmitFeedback,
                            loading: isPending,
                            disabled: selectedReasons.length === 0,
                            children: t('form.button.submit', 'Submit')
                        })
                    ]
                })
            ]
        })
    });
};

export { FeedbackModal, FeedbackProvider, useFeedbackModal };
//# sourceMappingURL=FeedbackModal.mjs.map
