'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var designSystem = require('@strapi/design-system');
var styledComponents = require('styled-components');
var useFeedback = require('./hooks/useFeedback.js');
var useTranslations = require('./hooks/useTranslations.js');

const FeedbackModalContext = /*#__PURE__*/ React.createContext({
    isFeedbackModalOpen: false,
    currentMessageId: null,
    openFeedbackModal: ()=>{},
    closeFeedbackModal: ()=>{}
});
const useFeedbackModal = ()=>React.useContext(FeedbackModalContext);
const FeedbackProvider = ({ children })=>{
    const [isFeedbackModalOpen, setIsFeedbackModalOpen] = React.useState(false);
    const [currentMessageId, setCurrentMessageId] = React.useState(null);
    const openFeedbackModal = (messageId)=>{
        setCurrentMessageId(messageId);
        setIsFeedbackModalOpen(true);
    };
    const closeFeedbackModal = ()=>{
        setIsFeedbackModalOpen(false);
        setCurrentMessageId(null);
    };
    return /*#__PURE__*/ jsxRuntime.jsxs(FeedbackModalContext.Provider, {
        value: {
            isFeedbackModalOpen,
            currentMessageId,
            openFeedbackModal,
            closeFeedbackModal
        },
        children: [
            isFeedbackModalOpen && currentMessageId && /*#__PURE__*/ jsxRuntime.jsx(FeedbackModal, {}),
            children
        ]
    });
};
const FeedbackOptionWrapper = styledComponents.styled(designSystem.Flex)`
  &:hover {
    background-color: ${({ theme })=>theme.colors.neutral100};
  }

  &.selected {
    background-color: ${({ theme })=>theme.colors.primary100};
    border-color: ${({ theme })=>theme.colors.primary200};
  }
`;
const FeedbackOption = ({ id, label, selected, onClick })=>{
    return /*#__PURE__*/ jsxRuntime.jsxs(FeedbackOptionWrapper, {
        className: selected ? 'selected' : '',
        justifyContent: "space-between",
        hasRadius: true,
        width: "100%",
        cursor: "pointer",
        borderColor: "neutral200",
        padding: 3,
        gap: 3,
        onClick: ()=>onClick(id),
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                variant: "omega",
                fontWeight: "bold",
                children: label
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Checkbox, {
                name: `feedback-${id}`,
                value: id,
                checked: selected,
                onChange: ()=>onClick(id)
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Modal
 * -----------------------------------------------------------------------------------------------*/ const FeedbackModal = ()=>{
    const [feedbackText, setFeedbackText] = React.useState('');
    const [selectedReasons, setSelectedReasons] = React.useState([]);
    const { t } = useTranslations.useTranslations();
    const { toggleNotification } = strapiAdmin.useNotification();
    const { closeFeedbackModal, currentMessageId } = useFeedbackModal();
    const { downvoteMessage, isPending } = useFeedback.useFeedback();
    const feedbackReasons = [
        {
            id: 'invalid_schema',
            label: t('chat.feedback.reason.invalid_schema', 'Invalid schema')
        },
        {
            id: 'bad_recommendation',
            label: t('chat.feedback.reason.bad_recommendation', 'Bad recommendation')
        },
        {
            id: 'slow',
            label: t('chat.feedback.reason.slow', 'Slow')
        },
        {
            id: 'instructions_ignored',
            label: t('chat.feedback.reason.instructions_ignored', 'Instructions ignored')
        },
        {
            id: 'being_lazy',
            label: t('chat.feedback.reason.being_lazy', 'Being lazy')
        },
        {
            id: 'other',
            label: t('chat.feedback.reason.other', 'Other')
        }
    ];
    const handleReasonSelect = (id)=>{
        setSelectedReasons((prevSelected)=>{
            // If already selected, remove it
            if (prevSelected.includes(id)) {
                return prevSelected.filter((reasonId)=>reasonId !== id);
            }
            // Otherwise add it
            return [
                ...prevSelected,
                id
            ];
        });
    };
    const handleSubmitFeedback = async ()=>{
        if (selectedReasons.length === 0) return;
        try {
            await downvoteMessage(currentMessageId, feedbackText, selectedReasons);
            setFeedbackText('');
            setSelectedReasons([]);
            closeFeedbackModal();
        } catch (error) {
            toggleNotification({
                type: 'danger',
                message: t('chat.feedback.error', 'An error occurred while submitting your feedback')
            });
        }
    };
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Root, {
        open: true,
        onOpenChange: closeFeedbackModal,
        children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Modal.Content, {
            children: [
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Header, {
                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        variant: "omega",
                        fontWeight: "bold",
                        children: t('chat.feedback.title', 'Give feedback')
                    })
                }),
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Body, {
                    children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                        direction: "column",
                        alignItems: "start",
                        gap: 6,
                        width: "100%",
                        children: [
                            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                                direction: "column",
                                alignItems: "start",
                                gap: 2,
                                children: [
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                        variant: "beta",
                                        fontWeight: "bold",
                                        children: t('chat.feedback.title', 'Give feedback')
                                    }),
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                        variant: "omega",
                                        children: t('chat.feedback.subtitle', 'Provide additional feedback on this message. Select all that apply.')
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Grid.Root, {
                                width: "100%",
                                gap: 2,
                                children: feedbackReasons.map((reason)=>/*#__PURE__*/ jsxRuntime.jsx(designSystem.Grid.Item, {
                                        col: 6,
                                        xs: 12,
                                        children: /*#__PURE__*/ jsxRuntime.jsx(FeedbackOption, {
                                            id: reason.id,
                                            label: reason.label,
                                            selected: selectedReasons.includes(reason.id),
                                            onClick: handleReasonSelect
                                        }, reason.id)
                                    }, reason.id))
                            }),
                            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                                direction: "column",
                                gap: 2,
                                width: "100%",
                                alignItems: "start",
                                children: [
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                        variant: "omega",
                                        children: t('chat.feedback.comment.label', 'How can we improve? (optional)')
                                    }),
                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                        width: "100%",
                                        children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Textarea, {
                                            name: "feedback",
                                            placeholder: t('chat.feedback.placeholder', 'Your feedback...'),
                                            onChange: (e)=>setFeedbackText(e.target.value),
                                            value: feedbackText
                                        })
                                    })
                                ]
                            })
                        ]
                    })
                }),
                /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Modal.Footer, {
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Close, {
                            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                variant: "tertiary",
                                onClick: closeFeedbackModal,
                                children: t('form.button.cancel', 'Cancel')
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                            onClick: handleSubmitFeedback,
                            loading: isPending,
                            disabled: selectedReasons.length === 0,
                            children: t('form.button.submit', 'Submit')
                        })
                    ]
                })
            ]
        })
    });
};

exports.FeedbackModal = FeedbackModal;
exports.FeedbackProvider = FeedbackProvider;
exports.useFeedbackModal = useFeedbackModal;
//# sourceMappingURL=FeedbackModal.js.map
