import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import { useEffect, useRef, useState } from 'react';
import { useGuidedTour, useAppInfo, tours } from '@strapi/admin/strapi-admin';
import { Flex, IconButton, Typography, Box, Button } from '@strapi/design-system';
import { Sparkle, Plus, ArrowUp, Paperclip, Code, Upload } from '@strapi/icons';
import { styled } from 'styled-components';
import '../CTBSession/sessionContext.mjs';
import { useCTBTracking } from '../CTBSession/useCTBTracking.mjs';
import { Alert } from './components/Alert.mjs';
import { AnimatedBox } from './components/AnimatedBox.mjs';
import { AttachmentPreview } from './components/Attachments/AttachmentPreview.mjs';
import { Dropzone } from './components/Dropzone.mjs';
import { Panel } from './components/FloatingPanel.mjs';
import { FigmaIcon } from './components/icons/FigmaIcon.mjs';
import { StopIcon } from './components/icons/StopIcon.mjs';
import { Input } from './components/Input.mjs';
import { ChatMessage } from './components/Messages/Message.mjs';
import { ResizableTextArea } from './components/ResizableTextArea.mjs';
import { SimpleMenu, MenuItem } from './components/SimpleMenu.mjs';
import { TOO_MANY_REQUESTS_ERROR, LICENSE_LIMIT_REACHED_ERROR, LICENSE_LIMIT_EXCEEDED_ERROR, CHAT_TOO_LONG_ERROR } from './hooks/useAIFetch.mjs';
import { useAttachments } from './hooks/useAttachments.mjs';
import { useTranslations } from './hooks/useTranslations.mjs';
import { useStrapiChat } from './providers/ChatProvider.mjs';
import { useUploadProjectToChat } from './UploadCodeModal.mjs';
import { useUploadFigmaToChat } from './UploadFigmaModal.mjs';

/* -------------------------------------------------------------------------------------------------
 * Chat Message Suggestions
 * -----------------------------------------------------------------------------------------------*/ const ResponsiveFlex = styled(Flex)`
  @media (max-height: 500px) {
    display: none;
  }
`;
const ChatSuggestions = ()=>{
    const { sendMessage } = useStrapiChat();
    const { t } = useTranslations();
    const { trackUsage } = useCTBTracking();
    const SUGGESTIONS = [
        t('chat.input.defaults.generate', 'Generate a product schema'),
        t('chat.input.defaults.ctb', 'Tell me about the Content-Type Builder'),
        t('chat.input.defaults.strapi', 'Tell me about Strapi')
    ];
    const SUGGESTION_TO_PROMPT_TYPE = {
        [t('chat.input.defaults.generate', 'Generate a product schema')]: 'generate-product-schema',
        [t('chat.input.defaults.ctb', 'Tell me about the Content-Type Builder')]: 'tell-me-about-the-content-type-builder',
        [t('chat.input.defaults.strapi', 'Tell me about Strapi')]: 'tell-me-about-strapi'
    };
    const suggestionsTitle = t('chat.input.defaults.title', 'How can I help you?');
    return /*#__PURE__*/ jsx(AnimatedBox, {
        children: /*#__PURE__*/ jsxs(ResponsiveFlex, {
            direction: "column",
            alignItems: 'flex-start',
            gap: 3,
            children: [
                /*#__PURE__*/ jsx(Typography, {
                    variant: "omega",
                    fontWeight: "bold",
                    children: suggestionsTitle
                }),
                /*#__PURE__*/ jsx(Flex, {
                    direction: "column",
                    alignItems: 'flex-start',
                    gap: 2,
                    children: SUGGESTIONS.map((suggestion)=>/*#__PURE__*/ jsx(Button, {
                            startIcon: /*#__PURE__*/ jsx(Sparkle, {
                                fill: "neutral500"
                            }),
                            size: "M",
                            variant: "tertiary",
                            onClick: ()=>{
                                trackUsage('didUsePresetPrompt', {
                                    promptType: SUGGESTION_TO_PROMPT_TYPE[suggestion]
                                });
                                sendMessage({
                                    text: suggestion
                                });
                            },
                            children: /*#__PURE__*/ jsx(Typography, {
                                fontWeight: "regular",
                                children: suggestion
                            })
                        }, suggestion))
                })
            ]
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * Chat Errors
 * -----------------------------------------------------------------------------------------------*/ const getErrorMessage = (err)=>{
    try {
        const error = JSON.parse(err.message);
        return error.error;
    } catch (e) {
        return err;
    }
};
const ChatError = ()=>{
    const { error } = useStrapiChat();
    const { t } = useTranslations();
    const defaultErrorMessage = t('chat.messages.error', 'Something went wrong.');
    const tooManyRequestsMessage = t('chat.messages.too-many-requests', 'Too many requests, please try again later.');
    const licenseLimitReachedMessage = t('chat.messages.license-limit-reached', 'License limit reached, please try again tomorrow.');
    const licenseLimitExceededMessage = t('chat.messages.license-limit-exceeded', 'AI credit limit exceeded.');
    const chatTooLongError = t('chat.messages.too-long-error', 'This conversation reached its maximum length. Start a new conversation');
    if (!error) return null;
    const errorMessage = getErrorMessage(error);
    if (!errorMessage || typeof errorMessage !== 'string') {
        return null;
    }
    if (errorMessage === TOO_MANY_REQUESTS_ERROR) {
        return /*#__PURE__*/ jsx(Alert, {
            title: tooManyRequestsMessage,
            variant: "warning"
        });
    }
    if (errorMessage === LICENSE_LIMIT_REACHED_ERROR) {
        return /*#__PURE__*/ jsx(Alert, {
            title: licenseLimitReachedMessage,
            variant: "warning"
        });
    }
    if (errorMessage.split('.')[0] === LICENSE_LIMIT_EXCEEDED_ERROR) {
        return /*#__PURE__*/ jsx(Alert, {
            title: licenseLimitExceededMessage,
            variant: "danger"
        });
    }
    if (errorMessage === CHAT_TOO_LONG_ERROR) {
        return /*#__PURE__*/ jsx(Alert, {
            title: chatTooLongError,
            variant: "danger"
        });
    }
    return /*#__PURE__*/ jsx(Alert, {
        title: defaultErrorMessage,
        variant: "danger"
    });
};
/* -------------------------------------------------------------------------------------------------
 * Chat Messages
 * -----------------------------------------------------------------------------------------------*/ const ChatContent = ({ messages })=>{
    const messageEndRef = useRef(null);
    const { status } = useStrapiChat();
    // Scroll to the bottom of the chat when new messages arrive
    useEffect(()=>{
        // Add a small delay to ensure all message content is fully rendered
        // including conditionally rendered elements like feedback buttons
        const scrollTimeout = setTimeout(()=>{
            messageEndRef.current?.scrollIntoView({
                behavior: 'smooth'
            });
        }, 20);
        return ()=>clearTimeout(scrollTimeout);
    }, [
        messages,
        status
    ]);
    return /*#__PURE__*/ jsxs(Fragment, {
        children: [
            /*#__PURE__*/ jsxs(Flex, {
                direction: "column",
                gap: 5,
                children: [
                    messages.map((message, idx)=>/*#__PURE__*/ jsx(ChatMessage, {
                            message: message,
                            // Chat loading and message is the last one
                            isLoading: (status === 'streaming' || status === 'submitted') && idx === messages.length - 1
                        }, message.id)),
                    status === 'error' && /*#__PURE__*/ jsx(ChatError, {})
                ]
            }),
            /*#__PURE__*/ jsx("div", {
                ref: messageEndRef
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Attachments
 * -----------------------------------------------------------------------------------------------*/ const ChatAttachmentList = ()=>{
    const { attachments, removeAttachmentByIndex } = useAttachments();
    if (!attachments.length) return null;
    return /*#__PURE__*/ jsx(Input.Attachments, {
        children: attachments.map((attachment, index)=>/*#__PURE__*/ jsx(Box, {
                maxWidth: '250px',
                children: /*#__PURE__*/ jsx(AttachmentPreview, {
                    attachment: attachment,
                    onRemove: ()=>removeAttachmentByIndex(index)
                })
            }, attachment.filename))
    });
};
const ChatAttachmentMenu = ()=>{
    const { attachFiles } = useAttachments();
    const { openCodeUpload } = useUploadProjectToChat();
    const { openFigmaUpload } = useUploadFigmaToChat();
    const { t } = useTranslations();
    const fileInputRef = useRef(null);
    const uploadLabel = t('chat.tooltips.upload-attachments', 'Upload attachments');
    const importCodeText = t('chat.attachments.menu.import-code', 'Import code');
    const attachImageText = t('chat.attachments.menu.attach-image', 'Attach image');
    const importFigmaText = t('chat.attachments.menu.import-figma', 'Import from Figma');
    return /*#__PURE__*/ jsxs(SimpleMenu, {
        label: uploadLabel,
        icon: /*#__PURE__*/ jsx(Paperclip, {}),
        tag: IconButton,
        popoverPlacement: "top-end",
        children: [
            /*#__PURE__*/ jsx(MenuItem, {
                startIcon: /*#__PURE__*/ jsx(Code, {}),
                onSelect: ()=>openCodeUpload(),
                children: importCodeText
            }),
            /*#__PURE__*/ jsx(MenuItem, {
                startIcon: /*#__PURE__*/ jsx(FigmaIcon, {}),
                onSelect: ()=>openFigmaUpload(),
                children: importFigmaText
            }),
            /*#__PURE__*/ jsx("input", {
                type: "file",
                accept: "image/*",
                multiple: true,
                hidden: true,
                ref: fileInputRef
            }),
            /*#__PURE__*/ jsx(MenuItem, {
                startIcon: /*#__PURE__*/ jsx(Upload, {}),
                onSelect: ()=>{
                    const fileInput = fileInputRef.current;
                    if (!fileInput) return;
                    fileInput.onchange = (e)=>{
                        const target = e.target;
                        if (target.files && target.files.length > 0) {
                            attachFiles(Array.from(target.files));
                        }
                    };
                    fileInput.click();
                },
                children: attachImageText
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Input
 * -----------------------------------------------------------------------------------------------*/ const LoadingDots = ({ children })=>{
    const [dots, setDots] = useState('');
    useEffect(()=>{
        const dotStates = [
            '',
            '.',
            '..',
            '...'
        ];
        let currentIndex = 0;
        const interval = setInterval(()=>{
            currentIndex = (currentIndex + 1) % dotStates.length;
            setDots(dotStates[currentIndex]);
        }, 400); // Change every 400ms
        return ()=>clearInterval(interval);
    }, []);
    return /*#__PURE__*/ jsxs(Typography, {
        variant: "pi",
        textColor: "neutral600",
        children: [
            children,
            dots
        ]
    });
};
const ChatInput = (props)=>{
    const { input, handleSubmit, handleInputChange, messages, status, stop, attachments } = useStrapiChat();
    const { t } = useTranslations();
    const isLoading = status === 'streaming' || status === 'submitted';
    return /*#__PURE__*/ jsx(Fragment, {
        children: /*#__PURE__*/ jsx("form", {
            onSubmit: handleSubmit,
            ...props,
            children: /*#__PURE__*/ jsxs(Input.Root, {
                isLoading: isLoading,
                gap: 4,
                children: [
                    /*#__PURE__*/ jsxs(Input.Header, {
                        minHeight: 0,
                        children: [
                            messages.length === 0 && /*#__PURE__*/ jsx(Input.HeaderItem, {
                                children: /*#__PURE__*/ jsx(ChatSuggestions, {})
                            }),
                            isLoading && /*#__PURE__*/ jsx(Input.HeaderItem, {
                                children: /*#__PURE__*/ jsx(LoadingDots, {
                                    children: t('chat.input.thinking', 'Strapi AI is thinking')
                                })
                            })
                        ]
                    }),
                    /*#__PURE__*/ jsxs(Input.Content, {
                        children: [
                            /*#__PURE__*/ jsx(Dropzone.Area, {}),
                            /*#__PURE__*/ jsx(ChatAttachmentList, {}),
                            /*#__PURE__*/ jsx(Box, {
                                paddingBottom: 3,
                                children: /*#__PURE__*/ jsx(ResizableTextArea, {
                                    value: input,
                                    onChange: handleInputChange,
                                    onSubmit: handleSubmit,
                                    placeholder: t('chat.input.placeholder', 'Ask Strapi AI...')
                                })
                            }),
                            /*#__PURE__*/ jsxs(Input.Actions, {
                                children: [
                                    /*#__PURE__*/ jsx(ChatAttachmentMenu, {}),
                                    isLoading ? /*#__PURE__*/ jsx(IconButton, {
                                        label: t('chat.tooltips.stop-generation', 'Stop'),
                                        type: "button",
                                        variant: "default",
                                        onClick: (e)=>{
                                            e.preventDefault();
                                            stop();
                                        },
                                        children: /*#__PURE__*/ jsx(StopIcon, {})
                                    }) : /*#__PURE__*/ jsx(IconButton, {
                                        label: t('chat.tooltips.send-message', 'Send'),
                                        variant: "default",
                                        type: "submit",
                                        // allow sending an empty message if there are attachments
                                        disabled: input.trim().length === 0 && attachments.length === 0,
                                        children: /*#__PURE__*/ jsx(ArrowUp, {})
                                    })
                                ]
                            })
                        ]
                    })
                ]
            })
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * Chat
 * -----------------------------------------------------------------------------------------------*/ const Chat = ()=>{
    const { title, messages, isChatOpen, isChatEnabled, openChat, closeChat, reset } = useStrapiChat();
    const { attachFiles } = useAttachments();
    const { t } = useTranslations();
    const state = useGuidedTour('Chat', (s)=>s.state);
    const currentEnvironment = useAppInfo('Chat', (state)=>state.currentEnvironment);
    // Disable AI Chat in production mode
    const isProduction = currentEnvironment === 'production';
    // Auto-open chat when AIChat guided tour step is active
    useEffect(()=>{
        const isAIChatStepActive = state.enabled && state.tours.contentTypeBuilder?.currentStep === 1 && // AIChat is step 1 in contentTypeBuilderStepsAI
        !state.tours.contentTypeBuilder?.isCompleted;
        if (isAIChatStepActive && !isChatOpen && openChat) {
            openChat();
        }
    }, [
        state.tours.contentTypeBuilder?.currentStep,
        state.tours.contentTypeBuilder?.isCompleted,
        state.enabled,
        isChatOpen,
        openChat
    ]);
    // Labels
    const openChatLabel = t('chat.tooltips.open-chat', 'Open chat');
    const closeChatLabel = t('chat.tooltips.close-chat', 'Close chat');
    const createChatLabel = t('chat.tooltips.create-chat', 'New conversation');
    const defaultTitle = t('chat.header.default-title', 'New conversation');
    const mistakesDisclaimer = t('chat.input.strapi-ai-can-make-errors', 'Strapi AI can make mistakes.');
    // Don't render the chat at all in production mode or if chat is not enabled
    if (!isChatEnabled || isProduction) {
        return null;
    }
    return /*#__PURE__*/ jsx(Dropzone.Root, {
        onAddFiles: (files)=>{
            attachFiles(files);
        },
        children: /*#__PURE__*/ jsxs(Panel.Root, {
            size: "md",
            position: "bottom-right",
            isOpen: isChatOpen,
            onToggle: isChatOpen ? closeChat : openChat,
            toggleIcon: /*#__PURE__*/ jsx(IconButton, {
                onClick: isChatOpen ? closeChat : openChat,
                label: isChatOpen ? closeChatLabel : openChatLabel,
                variant: "default",
                size: "L",
                children: /*#__PURE__*/ jsx(Sparkle, {})
            }),
            children: [
                /*#__PURE__*/ jsxs(Panel.Header, {
                    children: [
                        /*#__PURE__*/ jsx(Typography, {
                            tag: "h3",
                            variant: "omega",
                            fontWeight: "bold",
                            children: title || defaultTitle
                        }),
                        /*#__PURE__*/ jsxs(Flex, {
                            gap: 0,
                            children: [
                                /*#__PURE__*/ jsx(IconButton, {
                                    label: createChatLabel,
                                    variant: "ghost",
                                    onClick: reset,
                                    children: /*#__PURE__*/ jsx(Plus, {})
                                }),
                                /*#__PURE__*/ jsx(Panel.Close, {
                                    label: closeChatLabel
                                })
                            ]
                        })
                    ]
                }),
                /*#__PURE__*/ jsx(Panel.Body, {
                    children: /*#__PURE__*/ jsxs(tours.contentTypeBuilder.AIChat, {
                        children: [
                            !messages.length ? /*#__PURE__*/ jsx(Typography, {
                                variant: "pi",
                                textColor: "neutral600",
                                children: mistakesDisclaimer
                            }) : null,
                            /*#__PURE__*/ jsx(ChatContent, {
                                messages: messages
                            })
                        ]
                    })
                }),
                /*#__PURE__*/ jsx(Panel.Footer, {
                    children: /*#__PURE__*/ jsx(ChatInput, {})
                })
            ]
        })
    });
};

export { Chat };
//# sourceMappingURL=Chat.mjs.map
