'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var designSystem = require('@strapi/design-system');
var Icons = require('@strapi/icons');
var styledComponents = require('styled-components');
require('../CTBSession/sessionContext.js');
var useCTBTracking = require('../CTBSession/useCTBTracking.js');
var Alert = require('./components/Alert.js');
var AnimatedBox = require('./components/AnimatedBox.js');
var AttachmentPreview = require('./components/Attachments/AttachmentPreview.js');
var Dropzone = require('./components/Dropzone.js');
var FloatingPanel = require('./components/FloatingPanel.js');
var FigmaIcon = require('./components/icons/FigmaIcon.js');
var StopIcon = require('./components/icons/StopIcon.js');
var Input = require('./components/Input.js');
var Message = require('./components/Messages/Message.js');
var ResizableTextArea = require('./components/ResizableTextArea.js');
var SimpleMenu = require('./components/SimpleMenu.js');
var useAIFetch = require('./hooks/useAIFetch.js');
var useAttachments = require('./hooks/useAttachments.js');
var useTranslations = require('./hooks/useTranslations.js');
var ChatProvider = require('./providers/ChatProvider.js');
var UploadCodeModal = require('./UploadCodeModal.js');
var UploadFigmaModal = require('./UploadFigmaModal.js');

/* -------------------------------------------------------------------------------------------------
 * Chat Message Suggestions
 * -----------------------------------------------------------------------------------------------*/ const ResponsiveFlex = styledComponents.styled(designSystem.Flex)`
  @media (max-height: 500px) {
    display: none;
  }
`;
const ChatSuggestions = ()=>{
    const { sendMessage } = ChatProvider.useStrapiChat();
    const { t } = useTranslations.useTranslations();
    const { trackUsage } = useCTBTracking.useCTBTracking();
    const SUGGESTIONS = [
        t('chat.input.defaults.generate', 'Generate a product schema'),
        t('chat.input.defaults.ctb', 'Tell me about the Content-Type Builder'),
        t('chat.input.defaults.strapi', 'Tell me about Strapi')
    ];
    const SUGGESTION_TO_PROMPT_TYPE = {
        [t('chat.input.defaults.generate', 'Generate a product schema')]: 'generate-product-schema',
        [t('chat.input.defaults.ctb', 'Tell me about the Content-Type Builder')]: 'tell-me-about-the-content-type-builder',
        [t('chat.input.defaults.strapi', 'Tell me about Strapi')]: 'tell-me-about-strapi'
    };
    const suggestionsTitle = t('chat.input.defaults.title', 'How can I help you?');
    return /*#__PURE__*/ jsxRuntime.jsx(AnimatedBox.AnimatedBox, {
        children: /*#__PURE__*/ jsxRuntime.jsxs(ResponsiveFlex, {
            direction: "column",
            alignItems: 'flex-start',
            gap: 3,
            children: [
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                    variant: "omega",
                    fontWeight: "bold",
                    children: suggestionsTitle
                }),
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Flex, {
                    direction: "column",
                    alignItems: 'flex-start',
                    gap: 2,
                    children: SUGGESTIONS.map((suggestion)=>/*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                            startIcon: /*#__PURE__*/ jsxRuntime.jsx(Icons.Sparkle, {
                                fill: "neutral500"
                            }),
                            size: "M",
                            variant: "tertiary",
                            onClick: ()=>{
                                trackUsage('didUsePresetPrompt', {
                                    promptType: SUGGESTION_TO_PROMPT_TYPE[suggestion]
                                });
                                sendMessage({
                                    text: suggestion
                                });
                            },
                            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                fontWeight: "regular",
                                children: suggestion
                            })
                        }, suggestion))
                })
            ]
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * Chat Errors
 * -----------------------------------------------------------------------------------------------*/ const getErrorMessage = (err)=>{
    try {
        const error = JSON.parse(err.message);
        return error.error;
    } catch (e) {
        return err;
    }
};
const ChatError = ()=>{
    const { error } = ChatProvider.useStrapiChat();
    const { t } = useTranslations.useTranslations();
    const defaultErrorMessage = t('chat.messages.error', 'Something went wrong.');
    const tooManyRequestsMessage = t('chat.messages.too-many-requests', 'Too many requests, please try again later.');
    const licenseLimitReachedMessage = t('chat.messages.license-limit-reached', 'License limit reached, please try again tomorrow.');
    const licenseLimitExceededMessage = t('chat.messages.license-limit-exceeded', 'AI credit limit exceeded.');
    const chatTooLongError = t('chat.messages.too-long-error', 'This conversation reached its maximum length. Start a new conversation');
    if (!error) return null;
    const errorMessage = getErrorMessage(error);
    if (!errorMessage || typeof errorMessage !== 'string') {
        return null;
    }
    if (errorMessage === useAIFetch.TOO_MANY_REQUESTS_ERROR) {
        return /*#__PURE__*/ jsxRuntime.jsx(Alert.Alert, {
            title: tooManyRequestsMessage,
            variant: "warning"
        });
    }
    if (errorMessage === useAIFetch.LICENSE_LIMIT_REACHED_ERROR) {
        return /*#__PURE__*/ jsxRuntime.jsx(Alert.Alert, {
            title: licenseLimitReachedMessage,
            variant: "warning"
        });
    }
    if (errorMessage.split('.')[0] === useAIFetch.LICENSE_LIMIT_EXCEEDED_ERROR) {
        return /*#__PURE__*/ jsxRuntime.jsx(Alert.Alert, {
            title: licenseLimitExceededMessage,
            variant: "danger"
        });
    }
    if (errorMessage === useAIFetch.CHAT_TOO_LONG_ERROR) {
        return /*#__PURE__*/ jsxRuntime.jsx(Alert.Alert, {
            title: chatTooLongError,
            variant: "danger"
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsx(Alert.Alert, {
        title: defaultErrorMessage,
        variant: "danger"
    });
};
/* -------------------------------------------------------------------------------------------------
 * Chat Messages
 * -----------------------------------------------------------------------------------------------*/ const ChatContent = ({ messages })=>{
    const messageEndRef = React.useRef(null);
    const { status } = ChatProvider.useStrapiChat();
    // Scroll to the bottom of the chat when new messages arrive
    React.useEffect(()=>{
        // Add a small delay to ensure all message content is fully rendered
        // including conditionally rendered elements like feedback buttons
        const scrollTimeout = setTimeout(()=>{
            messageEndRef.current?.scrollIntoView({
                behavior: 'smooth'
            });
        }, 20);
        return ()=>clearTimeout(scrollTimeout);
    }, [
        messages,
        status
    ]);
    return /*#__PURE__*/ jsxRuntime.jsxs(jsxRuntime.Fragment, {
        children: [
            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                direction: "column",
                gap: 5,
                children: [
                    messages.map((message, idx)=>/*#__PURE__*/ jsxRuntime.jsx(Message.ChatMessage, {
                            message: message,
                            // Chat loading and message is the last one
                            isLoading: (status === 'streaming' || status === 'submitted') && idx === messages.length - 1
                        }, message.id)),
                    status === 'error' && /*#__PURE__*/ jsxRuntime.jsx(ChatError, {})
                ]
            }),
            /*#__PURE__*/ jsxRuntime.jsx("div", {
                ref: messageEndRef
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Attachments
 * -----------------------------------------------------------------------------------------------*/ const ChatAttachmentList = ()=>{
    const { attachments, removeAttachmentByIndex } = useAttachments.useAttachments();
    if (!attachments.length) return null;
    return /*#__PURE__*/ jsxRuntime.jsx(Input.Input.Attachments, {
        children: attachments.map((attachment, index)=>/*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                maxWidth: '250px',
                children: /*#__PURE__*/ jsxRuntime.jsx(AttachmentPreview.AttachmentPreview, {
                    attachment: attachment,
                    onRemove: ()=>removeAttachmentByIndex(index)
                })
            }, attachment.filename))
    });
};
const ChatAttachmentMenu = ()=>{
    const { attachFiles } = useAttachments.useAttachments();
    const { openCodeUpload } = UploadCodeModal.useUploadProjectToChat();
    const { openFigmaUpload } = UploadFigmaModal.useUploadFigmaToChat();
    const { t } = useTranslations.useTranslations();
    const fileInputRef = React.useRef(null);
    const uploadLabel = t('chat.tooltips.upload-attachments', 'Upload attachments');
    const importCodeText = t('chat.attachments.menu.import-code', 'Import code');
    const attachImageText = t('chat.attachments.menu.attach-image', 'Attach image');
    const importFigmaText = t('chat.attachments.menu.import-figma', 'Import from Figma');
    return /*#__PURE__*/ jsxRuntime.jsxs(SimpleMenu.SimpleMenu, {
        label: uploadLabel,
        icon: /*#__PURE__*/ jsxRuntime.jsx(Icons.Paperclip, {}),
        tag: designSystem.IconButton,
        popoverPlacement: "top-end",
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(SimpleMenu.MenuItem, {
                startIcon: /*#__PURE__*/ jsxRuntime.jsx(Icons.Code, {}),
                onSelect: ()=>openCodeUpload(),
                children: importCodeText
            }),
            /*#__PURE__*/ jsxRuntime.jsx(SimpleMenu.MenuItem, {
                startIcon: /*#__PURE__*/ jsxRuntime.jsx(FigmaIcon.FigmaIcon, {}),
                onSelect: ()=>openFigmaUpload(),
                children: importFigmaText
            }),
            /*#__PURE__*/ jsxRuntime.jsx("input", {
                type: "file",
                accept: "image/*",
                multiple: true,
                hidden: true,
                ref: fileInputRef
            }),
            /*#__PURE__*/ jsxRuntime.jsx(SimpleMenu.MenuItem, {
                startIcon: /*#__PURE__*/ jsxRuntime.jsx(Icons.Upload, {}),
                onSelect: ()=>{
                    const fileInput = fileInputRef.current;
                    if (!fileInput) return;
                    fileInput.onchange = (e)=>{
                        const target = e.target;
                        if (target.files && target.files.length > 0) {
                            attachFiles(Array.from(target.files));
                        }
                    };
                    fileInput.click();
                },
                children: attachImageText
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * Input
 * -----------------------------------------------------------------------------------------------*/ const LoadingDots = ({ children })=>{
    const [dots, setDots] = React.useState('');
    React.useEffect(()=>{
        const dotStates = [
            '',
            '.',
            '..',
            '...'
        ];
        let currentIndex = 0;
        const interval = setInterval(()=>{
            currentIndex = (currentIndex + 1) % dotStates.length;
            setDots(dotStates[currentIndex]);
        }, 400); // Change every 400ms
        return ()=>clearInterval(interval);
    }, []);
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Typography, {
        variant: "pi",
        textColor: "neutral600",
        children: [
            children,
            dots
        ]
    });
};
const ChatInput = (props)=>{
    const { input, handleSubmit, handleInputChange, messages, status, stop, attachments } = ChatProvider.useStrapiChat();
    const { t } = useTranslations.useTranslations();
    const isLoading = status === 'streaming' || status === 'submitted';
    return /*#__PURE__*/ jsxRuntime.jsx(jsxRuntime.Fragment, {
        children: /*#__PURE__*/ jsxRuntime.jsx("form", {
            onSubmit: handleSubmit,
            ...props,
            children: /*#__PURE__*/ jsxRuntime.jsxs(Input.Input.Root, {
                isLoading: isLoading,
                gap: 4,
                children: [
                    /*#__PURE__*/ jsxRuntime.jsxs(Input.Input.Header, {
                        minHeight: 0,
                        children: [
                            messages.length === 0 && /*#__PURE__*/ jsxRuntime.jsx(Input.Input.HeaderItem, {
                                children: /*#__PURE__*/ jsxRuntime.jsx(ChatSuggestions, {})
                            }),
                            isLoading && /*#__PURE__*/ jsxRuntime.jsx(Input.Input.HeaderItem, {
                                children: /*#__PURE__*/ jsxRuntime.jsx(LoadingDots, {
                                    children: t('chat.input.thinking', 'Strapi AI is thinking')
                                })
                            })
                        ]
                    }),
                    /*#__PURE__*/ jsxRuntime.jsxs(Input.Input.Content, {
                        children: [
                            /*#__PURE__*/ jsxRuntime.jsx(Dropzone.Dropzone.Area, {}),
                            /*#__PURE__*/ jsxRuntime.jsx(ChatAttachmentList, {}),
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                paddingBottom: 3,
                                children: /*#__PURE__*/ jsxRuntime.jsx(ResizableTextArea.ResizableTextArea, {
                                    value: input,
                                    onChange: handleInputChange,
                                    onSubmit: handleSubmit,
                                    placeholder: t('chat.input.placeholder', 'Ask Strapi AI...')
                                })
                            }),
                            /*#__PURE__*/ jsxRuntime.jsxs(Input.Input.Actions, {
                                children: [
                                    /*#__PURE__*/ jsxRuntime.jsx(ChatAttachmentMenu, {}),
                                    isLoading ? /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                                        label: t('chat.tooltips.stop-generation', 'Stop'),
                                        type: "button",
                                        variant: "default",
                                        onClick: (e)=>{
                                            e.preventDefault();
                                            stop();
                                        },
                                        children: /*#__PURE__*/ jsxRuntime.jsx(StopIcon.StopIcon, {})
                                    }) : /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                                        label: t('chat.tooltips.send-message', 'Send'),
                                        variant: "default",
                                        type: "submit",
                                        // allow sending an empty message if there are attachments
                                        disabled: input.trim().length === 0 && attachments.length === 0,
                                        children: /*#__PURE__*/ jsxRuntime.jsx(Icons.ArrowUp, {})
                                    })
                                ]
                            })
                        ]
                    })
                ]
            })
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * Chat
 * -----------------------------------------------------------------------------------------------*/ const Chat = ()=>{
    const { title, messages, isChatOpen, isChatEnabled, openChat, closeChat, reset } = ChatProvider.useStrapiChat();
    const { attachFiles } = useAttachments.useAttachments();
    const { t } = useTranslations.useTranslations();
    const state = strapiAdmin.useGuidedTour('Chat', (s)=>s.state);
    const currentEnvironment = strapiAdmin.useAppInfo('Chat', (state)=>state.currentEnvironment);
    // Disable AI Chat in production mode
    const isProduction = currentEnvironment === 'production';
    // Auto-open chat when AIChat guided tour step is active
    React.useEffect(()=>{
        const isAIChatStepActive = state.enabled && state.tours.contentTypeBuilder?.currentStep === 1 && // AIChat is step 1 in contentTypeBuilderStepsAI
        !state.tours.contentTypeBuilder?.isCompleted;
        if (isAIChatStepActive && !isChatOpen && openChat) {
            openChat();
        }
    }, [
        state.tours.contentTypeBuilder?.currentStep,
        state.tours.contentTypeBuilder?.isCompleted,
        state.enabled,
        isChatOpen,
        openChat
    ]);
    // Labels
    const openChatLabel = t('chat.tooltips.open-chat', 'Open chat');
    const closeChatLabel = t('chat.tooltips.close-chat', 'Close chat');
    const createChatLabel = t('chat.tooltips.create-chat', 'New conversation');
    const defaultTitle = t('chat.header.default-title', 'New conversation');
    const mistakesDisclaimer = t('chat.input.strapi-ai-can-make-errors', 'Strapi AI can make mistakes.');
    // Don't render the chat at all in production mode or if chat is not enabled
    if (!isChatEnabled || isProduction) {
        return null;
    }
    return /*#__PURE__*/ jsxRuntime.jsx(Dropzone.Dropzone.Root, {
        onAddFiles: (files)=>{
            attachFiles(files);
        },
        children: /*#__PURE__*/ jsxRuntime.jsxs(FloatingPanel.Panel.Root, {
            size: "md",
            position: "bottom-right",
            isOpen: isChatOpen,
            onToggle: isChatOpen ? closeChat : openChat,
            toggleIcon: /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                onClick: isChatOpen ? closeChat : openChat,
                label: isChatOpen ? closeChatLabel : openChatLabel,
                variant: "default",
                size: "L",
                children: /*#__PURE__*/ jsxRuntime.jsx(Icons.Sparkle, {})
            }),
            children: [
                /*#__PURE__*/ jsxRuntime.jsxs(FloatingPanel.Panel.Header, {
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                            tag: "h3",
                            variant: "omega",
                            fontWeight: "bold",
                            children: title || defaultTitle
                        }),
                        /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                            gap: 0,
                            children: [
                                /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                                    label: createChatLabel,
                                    variant: "ghost",
                                    onClick: reset,
                                    children: /*#__PURE__*/ jsxRuntime.jsx(Icons.Plus, {})
                                }),
                                /*#__PURE__*/ jsxRuntime.jsx(FloatingPanel.Panel.Close, {
                                    label: closeChatLabel
                                })
                            ]
                        })
                    ]
                }),
                /*#__PURE__*/ jsxRuntime.jsx(FloatingPanel.Panel.Body, {
                    children: /*#__PURE__*/ jsxRuntime.jsxs(strapiAdmin.tours.contentTypeBuilder.AIChat, {
                        children: [
                            !messages.length ? /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                variant: "pi",
                                textColor: "neutral600",
                                children: mistakesDisclaimer
                            }) : null,
                            /*#__PURE__*/ jsxRuntime.jsx(ChatContent, {
                                messages: messages
                            })
                        ]
                    })
                }),
                /*#__PURE__*/ jsxRuntime.jsx(FloatingPanel.Panel.Footer, {
                    children: /*#__PURE__*/ jsxRuntime.jsx(ChatInput, {})
                })
            ]
        })
    });
};

exports.Chat = Chat;
//# sourceMappingURL=Chat.js.map
