'use strict';

var utils = require('@strapi/utils');
var isEqual = require('lodash/isEqual');
var lodash = require('lodash');
var constants = require('../constants.js');
var index = require('../utils/index.js');

async function deleteActionsOnDisableDraftAndPublish({ oldContentTypes, contentTypes }) {
    if (!oldContentTypes) {
        return;
    }
    for(const uid in contentTypes){
        if (!oldContentTypes[uid]) {
            continue;
        }
        const oldContentType = oldContentTypes[uid];
        const contentType = contentTypes[uid];
        if (utils.contentTypes.hasDraftAndPublish(oldContentType) && !utils.contentTypes.hasDraftAndPublish(contentType)) {
            await strapi.db?.queryBuilder(constants.RELEASE_ACTION_MODEL_UID).delete().where({
                contentType: uid
            }).execute();
        }
    }
}
async function deleteActionsOnDeleteContentType({ oldContentTypes, contentTypes }) {
    const deletedContentTypes = lodash.difference(lodash.keys(oldContentTypes), lodash.keys(contentTypes)) ?? [];
    if (deletedContentTypes.length) {
        await utils.async.map(deletedContentTypes, async (deletedContentTypeUID)=>{
            return strapi.db?.queryBuilder(constants.RELEASE_ACTION_MODEL_UID).delete().where({
                contentType: deletedContentTypeUID
            }).execute();
        });
    }
}
async function migrateIsValidAndStatusReleases() {
    const releasesWithoutStatus = await strapi.db.query(constants.RELEASE_MODEL_UID).findMany({
        where: {
            status: null,
            releasedAt: null
        },
        populate: {
            actions: {
                populate: {
                    entry: true
                }
            }
        }
    });
    utils.async.map(releasesWithoutStatus, async (release)=>{
        const actions = release.actions;
        const notValidatedActions = actions.filter((action)=>action.isEntryValid === null);
        for (const action of notValidatedActions){
            // We need to check the Action is related to a valid entry because we can't assume this is gonna be always the case
            // example: users could make changes directly to their database, or data could be lost
            if (action.entry) {
                const isEntryValid = index.getDraftEntryValidStatus({
                    contentType: action.contentType,
                    documentId: action.entryDocumentId,
                    locale: action.locale
                }, {
                    strapi
                });
                await strapi.db.query(constants.RELEASE_ACTION_MODEL_UID).update({
                    where: {
                        id: action.id
                    },
                    data: {
                        isEntryValid
                    }
                });
            }
        }
        return index.getService('release', {
            strapi
        }).updateReleaseStatus(release.id);
    });
    const publishedReleases = await strapi.db.query(constants.RELEASE_MODEL_UID).findMany({
        where: {
            status: null,
            releasedAt: {
                $notNull: true
            }
        }
    });
    utils.async.map(publishedReleases, async (release)=>{
        return strapi.db.query(constants.RELEASE_MODEL_UID).update({
            where: {
                id: release.id
            },
            data: {
                status: 'done'
            }
        });
    });
}
async function revalidateChangedContentTypes({ oldContentTypes, contentTypes }) {
    if (oldContentTypes !== undefined && contentTypes !== undefined) {
        const contentTypesWithDraftAndPublish = Object.keys(oldContentTypes).filter((uid)=>oldContentTypes[uid]?.options?.draftAndPublish);
        const releasesAffected = new Set();
        utils.async.map(contentTypesWithDraftAndPublish, async (contentTypeUID)=>{
            const oldContentType = oldContentTypes[contentTypeUID];
            const contentType = contentTypes[contentTypeUID];
            // If attributes have changed, we need to revalidate actions because maybe validations rules are different
            if (!isEqual(oldContentType?.attributes, contentType?.attributes)) {
                const actions = await strapi.db.query(constants.RELEASE_ACTION_MODEL_UID).findMany({
                    where: {
                        contentType: contentTypeUID
                    },
                    populate: {
                        entry: true,
                        release: true
                    }
                });
                await utils.async.map(actions, async (action)=>{
                    if (action.entry && action.release && action.type === 'publish') {
                        const isEntryValid = await index.getDraftEntryValidStatus({
                            contentType: contentTypeUID,
                            documentId: action.entryDocumentId,
                            locale: action.locale
                        }, {
                            strapi
                        });
                        releasesAffected.add(action.release.id);
                        await strapi.db.query(constants.RELEASE_ACTION_MODEL_UID).update({
                            where: {
                                id: action.id
                            },
                            data: {
                                isEntryValid
                            }
                        });
                    }
                });
            }
        }).then(()=>{
            // We need to update the status of the releases affected
            utils.async.map(releasesAffected, async (releaseId)=>{
                return index.getService('release', {
                    strapi
                }).updateReleaseStatus(releaseId);
            });
        });
    }
}
async function disableContentTypeLocalized({ oldContentTypes, contentTypes }) {
    if (!oldContentTypes) {
        return;
    }
    const i18nPlugin = strapi.plugin('i18n');
    if (!i18nPlugin) {
        return;
    }
    for(const uid in contentTypes){
        if (!oldContentTypes[uid]) {
            continue;
        }
        const oldContentType = oldContentTypes[uid];
        const contentType = contentTypes[uid];
        const { isLocalizedContentType } = i18nPlugin.service('content-types');
        // if i18N is disabled remove non default locales before sync
        if (isLocalizedContentType(oldContentType) && !isLocalizedContentType(contentType)) {
            await strapi.db.queryBuilder(constants.RELEASE_ACTION_MODEL_UID).update({
                locale: null
            }).where({
                contentType: uid
            }).execute();
        }
    }
}
async function enableContentTypeLocalized({ oldContentTypes, contentTypes }) {
    if (!oldContentTypes) {
        return;
    }
    const i18nPlugin = strapi.plugin('i18n');
    if (!i18nPlugin) {
        return;
    }
    for(const uid in contentTypes){
        if (!oldContentTypes[uid]) {
            continue;
        }
        const oldContentType = oldContentTypes[uid];
        const contentType = contentTypes[uid];
        const { isLocalizedContentType } = i18nPlugin.service('content-types');
        const { getDefaultLocale } = i18nPlugin.service('locales');
        // if i18N is enabled remove non default locales before sync
        if (!isLocalizedContentType(oldContentType) && isLocalizedContentType(contentType)) {
            const defaultLocale = await getDefaultLocale();
            await strapi.db.queryBuilder(constants.RELEASE_ACTION_MODEL_UID).update({
                locale: defaultLocale
            }).where({
                contentType: uid
            }).execute();
        }
    }
}

exports.deleteActionsOnDeleteContentType = deleteActionsOnDeleteContentType;
exports.deleteActionsOnDisableDraftAndPublish = deleteActionsOnDisableDraftAndPublish;
exports.disableContentTypeLocalized = disableContentTypeLocalized;
exports.enableContentTypeLocalized = enableContentTypeLocalized;
exports.migrateIsValidAndStatusReleases = migrateIsValidAndStatusReleases;
exports.revalidateChangedContentTypes = revalidateChangedContentTypes;
//# sourceMappingURL=index.js.map
