import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import { Page, useAPIErrorHandler, useNotification, useRBAC, Form, Layouts, GradientBadge, useField, isFetchError } from '@strapi/admin/strapi-admin';
import { Button, Flex, Typography, Grid, Field, Combobox, ComboboxOption } from '@strapi/design-system';
import { Check } from '@strapi/icons';
import { useIntl } from 'react-intl';
import { useTypedSelector } from '../modules/hooks.mjs';
import { useGetReleaseSettingsQuery, useUpdateReleaseSettingsMutation } from '../services/release.mjs';
import { getTimezones } from '../utils/time.mjs';
import { SETTINGS_SCHEMA } from '../validation/schemas.mjs';

const ReleasesSettingsPage = ()=>{
    const { formatMessage } = useIntl();
    const { formatAPIError } = useAPIErrorHandler();
    const { toggleNotification } = useNotification();
    const { data, isLoading: isLoadingSettings } = useGetReleaseSettingsQuery();
    const [updateReleaseSettings, { isLoading: isSubmittingForm }] = useUpdateReleaseSettingsMutation();
    const permissions = useTypedSelector((state)=>state.admin_app.permissions['settings']?.['releases']);
    const { allowedActions: { canUpdate } } = useRBAC(permissions);
    const { timezoneList } = getTimezones(new Date());
    const handleSubmit = async (body, { setErrors })=>{
        const { defaultTimezone } = body;
        const formattedDefaultTimezone = defaultTimezone;
        const isBodyTimezoneValid = timezoneList.some((timezone)=>timezone.value === formattedDefaultTimezone);
        if (!isBodyTimezoneValid && defaultTimezone) {
            const errorMessage = formatMessage({
                id: 'components.Input.error.validation.combobox.invalid',
                defaultMessage: 'The value provided is not valid'
            });
            setErrors({
                defaultTimezone: errorMessage
            });
            toggleNotification({
                type: 'danger',
                message: errorMessage
            });
            return;
        }
        const newBody = !defaultTimezone || !isBodyTimezoneValid ? {
            defaultTimezone: null
        } : {
            defaultTimezone: formattedDefaultTimezone
        };
        try {
            const response = await updateReleaseSettings(newBody);
            if ('data' in response) {
                toggleNotification({
                    type: 'success',
                    message: formatMessage({
                        id: 'content-releases.pages.Settings.releases.setting.default-timezone-notification-success',
                        defaultMessage: 'Default timezone updated.'
                    })
                });
            } else if (isFetchError(response.error)) {
                toggleNotification({
                    type: 'danger',
                    message: formatAPIError(response.error)
                });
            } else {
                toggleNotification({
                    type: 'danger',
                    message: formatMessage({
                        id: 'notification.error',
                        defaultMessage: 'An error occurred'
                    })
                });
            }
        } catch (error) {
            toggleNotification({
                type: 'danger',
                message: formatMessage({
                    id: 'notification.error',
                    defaultMessage: 'An error occurred'
                })
            });
        }
    };
    if (isLoadingSettings) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    const releasePageTitle = formatMessage({
        id: 'content-releases.pages.Releases.title',
        defaultMessage: 'Releases'
    });
    return /*#__PURE__*/ jsxs(Fragment, {
        children: [
            /*#__PURE__*/ jsx(Page.Title, {
                children: formatMessage({
                    id: 'Settings.PageTitle',
                    defaultMessage: 'Settings - {name}'
                }, {
                    name: releasePageTitle
                })
            }),
            /*#__PURE__*/ jsx(Page.Main, {
                "aria-busy": isLoadingSettings,
                tabIndex: -1,
                children: /*#__PURE__*/ jsx(Form, {
                    method: "PUT",
                    initialValues: {
                        defaultTimezone: data?.data.defaultTimezone
                    },
                    onSubmit: handleSubmit,
                    validationSchema: SETTINGS_SCHEMA,
                    children: ({ modified, isSubmitting })=>{
                        return /*#__PURE__*/ jsxs(Fragment, {
                            children: [
                                /*#__PURE__*/ jsx(Layouts.Header, {
                                    primaryAction: canUpdate ? /*#__PURE__*/ jsx(Button, {
                                        disabled: !modified || isSubmittingForm,
                                        loading: isSubmitting,
                                        startIcon: /*#__PURE__*/ jsx(Check, {}),
                                        type: "submit",
                                        children: formatMessage({
                                            id: 'global.save',
                                            defaultMessage: 'Save'
                                        })
                                    }) : null,
                                    secondaryAction: /*#__PURE__*/ jsx(GradientBadge, {
                                        label: formatMessage({
                                            id: 'components.premiumFeature.title',
                                            defaultMessage: 'Premium feature'
                                        })
                                    }),
                                    title: releasePageTitle,
                                    subtitle: formatMessage({
                                        id: 'content-releases.pages.Settings.releases.description',
                                        defaultMessage: 'Create and manage content updates'
                                    })
                                }),
                                /*#__PURE__*/ jsx(Layouts.Content, {
                                    children: /*#__PURE__*/ jsxs(Flex, {
                                        direction: "column",
                                        background: "neutral0",
                                        alignItems: "stretch",
                                        padding: 6,
                                        gap: 6,
                                        shadow: "filterShadow",
                                        hasRadius: true,
                                        children: [
                                            /*#__PURE__*/ jsx(Typography, {
                                                variant: "delta",
                                                tag: "h2",
                                                children: formatMessage({
                                                    id: 'content-releases.pages.Settings.releases.preferences.title',
                                                    defaultMessage: 'Preferences'
                                                })
                                            }),
                                            /*#__PURE__*/ jsx(Grid.Root, {
                                                children: /*#__PURE__*/ jsx(Grid.Item, {
                                                    col: 6,
                                                    xs: 12,
                                                    direction: "column",
                                                    alignItems: "stretch",
                                                    children: /*#__PURE__*/ jsx(TimezoneDropdown, {})
                                                })
                                            })
                                        ]
                                    })
                                })
                            ]
                        });
                    }
                })
            })
        ]
    });
};
const TimezoneDropdown = ()=>{
    const permissions = useTypedSelector((state)=>state.admin_app.permissions['settings']?.['releases']);
    const { allowedActions: { canUpdate } } = useRBAC(permissions);
    const { formatMessage } = useIntl();
    const { timezoneList } = getTimezones(new Date());
    const field = useField('defaultTimezone');
    return /*#__PURE__*/ jsxs(Field.Root, {
        name: "defaultTimezone",
        hint: formatMessage({
            id: 'content-releases.pages.Settings.releases.timezone.hint',
            defaultMessage: 'The timezone of every release can still be changed individually.'
        }),
        error: field.error,
        children: [
            /*#__PURE__*/ jsx(Field.Label, {
                children: formatMessage({
                    id: 'content-releases.pages.Settings.releases.timezone.label',
                    defaultMessage: 'Default timezone'
                })
            }),
            /*#__PURE__*/ jsx(Combobox, {
                autocomplete: {
                    type: 'list',
                    filter: 'contains'
                },
                onTextValueChange: (value)=>field.onChange('defaultTimezone', value),
                onChange: (value)=>{
                    if (field.value && value || !field.value) {
                        field.onChange('defaultTimezone', value);
                    }
                },
                onClear: ()=>field.onChange('defaultTimezone', ''),
                value: field.value,
                disabled: !canUpdate,
                children: timezoneList.map((timezone)=>/*#__PURE__*/ jsx(ComboboxOption, {
                        value: timezone.value,
                        children: timezone.value.replace(/&/, ' ')
                    }, timezone.value))
            }),
            /*#__PURE__*/ jsx(Field.Hint, {}),
            /*#__PURE__*/ jsx(Field.Error, {})
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * ProtectedSettingsPage
 * -----------------------------------------------------------------------------------------------*/ const ProtectedReleasesSettingsPage = ()=>{
    const permissions = useTypedSelector((state)=>state.admin_app.permissions['settings']?.['releases']?.read);
    return /*#__PURE__*/ jsx(Page.Protect, {
        permissions: permissions,
        children: /*#__PURE__*/ jsx(ReleasesSettingsPage, {})
    });
};

export { ProtectedReleasesSettingsPage };
//# sourceMappingURL=ReleasesSettingsPage.mjs.map
