'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var designSystem = require('@strapi/design-system');
var dateFns = require('date-fns');
var dateFnsTz = require('date-fns-tz');
var formik = require('formik');
var reactIntl = require('react-intl');
var reactRouterDom = require('react-router-dom');
var pluginId = require('../pluginId.js');
var time = require('../utils/time.js');
var schemas = require('../validation/schemas.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

const ReleaseModal = ({ handleClose, open, handleSubmit, initialValues, isLoading = false })=>{
    const { formatMessage } = reactIntl.useIntl();
    const { pathname } = reactRouterDom.useLocation();
    const isCreatingRelease = pathname === `/plugins/${pluginId.pluginId}`;
    // Set default first timezone from the list if no system timezone detected
    const { timezoneList, systemTimezone = {
        value: 'UTC+00:00-Africa/Abidjan '
    } } = time.getTimezones(initialValues.scheduledAt ? new Date(initialValues.scheduledAt) : new Date());
    /**
   * Generate scheduled time using selected date, time and timezone
   */ const getScheduledTimestamp = (values)=>{
        const { date, time, timezone } = values;
        if (!date || !time || !timezone) return null;
        const timezoneWithoutOffset = timezone.split('&')[1];
        return dateFnsTz.zonedTimeToUtc(`${date} ${time}`, timezoneWithoutOffset);
    };
    /**
   * Get timezone with offset to show the selected value in the dropdown
   */ const getTimezoneWithOffset = ()=>{
        const currentTimezone = timezoneList.find((timezone)=>timezone.value.split('&')[1] === initialValues.timezone);
        return currentTimezone?.value || systemTimezone.value;
    };
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Root, {
        open: open,
        onOpenChange: handleClose,
        children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Modal.Content, {
            children: [
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Header, {
                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Title, {
                        children: formatMessage({
                            id: 'content-releases.modal.title',
                            defaultMessage: '{isCreatingRelease, select, true {New release} other {Edit release}}'
                        }, {
                            isCreatingRelease: isCreatingRelease
                        })
                    })
                }),
                /*#__PURE__*/ jsxRuntime.jsx(formik.Formik, {
                    onSubmit: (values)=>{
                        handleSubmit({
                            ...values,
                            timezone: values.timezone ? values.timezone.split('&')[1] : null,
                            scheduledAt: values.isScheduled ? getScheduledTimestamp(values) : null
                        });
                    },
                    initialValues: {
                        ...initialValues,
                        timezone: initialValues.timezone ? getTimezoneWithOffset() : systemTimezone.value
                    },
                    validationSchema: schemas.RELEASE_SCHEMA,
                    validateOnChange: false,
                    children: ({ values, errors, handleChange, setFieldValue })=>{
                        return /*#__PURE__*/ jsxRuntime.jsxs(formik.Form, {
                            children: [
                                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Body, {
                                    children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                                        direction: "column",
                                        alignItems: "stretch",
                                        gap: 6,
                                        children: [
                                            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Field.Root, {
                                                name: "name",
                                                error: errors.name && formatMessage({
                                                    id: errors.name,
                                                    defaultMessage: errors.name
                                                }),
                                                required: true,
                                                children: [
                                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Label, {
                                                        children: formatMessage({
                                                            id: 'content-releases.modal.form.input.label.release-name',
                                                            defaultMessage: 'Name'
                                                        })
                                                    }),
                                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.TextInput, {
                                                        type: "text",
                                                        value: values.name,
                                                        onChange: handleChange
                                                    }),
                                                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Error, {})
                                                ]
                                            }),
                                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                                width: "max-content",
                                                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Checkbox, {
                                                    name: "isScheduled",
                                                    checked: values.isScheduled,
                                                    onCheckedChange: (checked)=>{
                                                        setFieldValue('isScheduled', checked);
                                                        if (!checked) {
                                                            // Clear scheduling info from a release on unchecking schedule release, which reset scheduling info in DB
                                                            setFieldValue('date', null);
                                                            setFieldValue('time', '');
                                                            setFieldValue('timezone', null);
                                                        } else {
                                                            // On ticking back schedule release date, time and timezone should be restored to the initial state
                                                            setFieldValue('date', initialValues.date);
                                                            setFieldValue('time', initialValues.time);
                                                            setFieldValue('timezone', initialValues.timezone ?? systemTimezone?.value);
                                                        }
                                                    },
                                                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                                        textColor: values.isScheduled ? 'primary600' : 'neutral800',
                                                        fontWeight: values.isScheduled ? 'semiBold' : 'regular',
                                                        children: formatMessage({
                                                            id: 'modal.form.input.label.schedule-release',
                                                            defaultMessage: 'Schedule release'
                                                        })
                                                    })
                                                })
                                            }),
                                            values.isScheduled && /*#__PURE__*/ jsxRuntime.jsxs(jsxRuntime.Fragment, {
                                                children: [
                                                    /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                                                        gap: 4,
                                                        alignItems: "start",
                                                        direction: {
                                                            initial: 'column',
                                                            medium: 'row'
                                                        },
                                                        wrap: "wrap",
                                                        children: [
                                                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                                                width: "100%",
                                                                flex: 1,
                                                                children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Field.Root, {
                                                                    name: "date",
                                                                    error: errors.date && formatMessage({
                                                                        id: errors.date,
                                                                        defaultMessage: errors.date
                                                                    }),
                                                                    required: true,
                                                                    children: [
                                                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Label, {
                                                                            children: formatMessage({
                                                                                id: 'content-releases.modal.form.input.label.date',
                                                                                defaultMessage: 'Date'
                                                                            })
                                                                        }),
                                                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.DatePicker, {
                                                                            onChange: (date)=>{
                                                                                const isoFormatDate = date ? dateFns.formatISO(date, {
                                                                                    representation: 'date'
                                                                                }) : null;
                                                                                setFieldValue('date', isoFormatDate);
                                                                            },
                                                                            clearLabel: formatMessage({
                                                                                id: 'content-releases.modal.form.input.clearLabel',
                                                                                defaultMessage: 'Clear'
                                                                            }),
                                                                            onClear: ()=>{
                                                                                setFieldValue('date', null);
                                                                            },
                                                                            value: values.date ? new Date(values.date) : new Date(),
                                                                            minDate: dateFnsTz.utcToZonedTime(new Date(), values.timezone.split('&')[1])
                                                                        }),
                                                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Error, {})
                                                                    ]
                                                                })
                                                            }),
                                                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                                                width: "100%",
                                                                flex: 1,
                                                                children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Field.Root, {
                                                                    name: "time",
                                                                    error: errors.time && formatMessage({
                                                                        id: errors.time,
                                                                        defaultMessage: errors.time
                                                                    }),
                                                                    required: true,
                                                                    children: [
                                                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Label, {
                                                                            children: formatMessage({
                                                                                id: 'content-releases.modal.form.input.label.time',
                                                                                defaultMessage: 'Time'
                                                                            })
                                                                        }),
                                                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.TimePicker, {
                                                                            onChange: (time)=>{
                                                                                setFieldValue('time', time);
                                                                            },
                                                                            clearLabel: formatMessage({
                                                                                id: 'content-releases.modal.form.input.clearLabel',
                                                                                defaultMessage: 'Clear'
                                                                            }),
                                                                            onClear: ()=>{
                                                                                setFieldValue('time', '');
                                                                            },
                                                                            value: values.time || undefined
                                                                        }),
                                                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Error, {})
                                                                    ]
                                                                })
                                                            })
                                                        ]
                                                    }),
                                                    /*#__PURE__*/ jsxRuntime.jsx(TimezoneComponent, {
                                                        timezoneOptions: timezoneList
                                                    })
                                                ]
                                            })
                                        ]
                                    })
                                }),
                                /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Modal.Footer, {
                                    children: [
                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Modal.Close, {
                                            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                                variant: "tertiary",
                                                name: "cancel",
                                                children: formatMessage({
                                                    id: 'cancel',
                                                    defaultMessage: 'Cancel'
                                                })
                                            })
                                        }),
                                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Button, {
                                            name: "submit",
                                            loading: isLoading,
                                            type: "submit",
                                            children: formatMessage({
                                                id: 'content-releases.modal.form.button.submit',
                                                defaultMessage: '{isCreatingRelease, select, true {Continue} other {Save}}'
                                            }, {
                                                isCreatingRelease: isCreatingRelease
                                            })
                                        })
                                    ]
                                })
                            ]
                        });
                    }
                })
            ]
        })
    });
};
const TimezoneComponent = ({ timezoneOptions })=>{
    const { values, errors, setFieldValue } = formik.useFormikContext();
    const { formatMessage } = reactIntl.useIntl();
    const [timezoneList, setTimezoneList] = React__namespace.useState(timezoneOptions);
    React__namespace.useEffect(()=>{
        if (values.date) {
            // Update the timezone offset which varies with DST based on the date selected
            const { timezoneList } = time.getTimezones(new Date(values.date));
            setTimezoneList(timezoneList);
            const updatedTimezone = values.timezone && timezoneList.find((tz)=>tz.value.split('&')[1] === values.timezone.split('&')[1]);
            if (updatedTimezone) {
                setFieldValue('timezone', updatedTimezone.value);
            }
        }
    }, [
        setFieldValue,
        values.date,
        values.timezone
    ]);
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Field.Root, {
        name: "timezone",
        error: errors.timezone && formatMessage({
            id: errors.timezone,
            defaultMessage: errors.timezone
        }),
        required: true,
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Label, {
                children: formatMessage({
                    id: 'content-releases.modal.form.input.label.timezone',
                    defaultMessage: 'Timezone'
                })
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Combobox, {
                autocomplete: {
                    type: 'list',
                    filter: 'contains'
                },
                value: values.timezone || undefined,
                textValue: values.timezone ? values.timezone.replace(/&/, ' ') : undefined,
                onChange: (timezone)=>{
                    setFieldValue('timezone', timezone);
                },
                onTextValueChange: (timezone)=>{
                    setFieldValue('timezone', timezone);
                },
                onClear: ()=>{
                    setFieldValue('timezone', '');
                },
                children: timezoneList.map((timezone)=>/*#__PURE__*/ jsxRuntime.jsx(designSystem.ComboboxOption, {
                        value: timezone.value,
                        children: timezone.value.replace(/&/, ' ')
                    }, timezone.value))
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Field.Error, {})
        ]
    });
};

exports.ReleaseModal = ReleaseModal;
//# sourceMappingURL=ReleaseModal.js.map
