import { propEq, merge, isEmpty, set } from 'lodash/fp';
import strapiUtils from '@strapi/utils';
import { getService } from '../../utils/index.mjs';

const { isVisibleAttribute, isScalarAttribute, getDoesAttributeRequireValidation } = strapiUtils.contentTypes;
const { isAnyToMany } = strapiUtils.relations;
const { PUBLISHED_AT_ATTRIBUTE } = strapiUtils.contentTypes.constants;
const isMorphToRelation = (attribute)=>isRelation(attribute) && attribute.relation.includes('morphTo');
const isMedia = propEq('type', 'media');
const isRelation = propEq('type', 'relation');
const isComponent = propEq('type', 'component');
const isDynamicZone = propEq('type', 'dynamiczone');
/**
 * Populate the model for relation
 * @param attribute - Attribute containing a relation
 * @param attribute.relation - type of relation
 * @param model - Model of the populated entity
 * @param attributeName
 * @param options - Options to apply while populating
 */ function getPopulateForRelation(attribute, model, attributeName, { countMany, countOne, initialPopulate }) {
    const isManyRelation = isAnyToMany(attribute);
    if (initialPopulate) {
        return initialPopulate;
    }
    // If populating localizations attribute, also include validatable fields
    // Mainly needed for bulk locale publishing, so the Client has all the information necessary to perform validations
    if (attributeName === 'localizations') {
        const validationPopulate = getPopulateForValidation(model.uid);
        return {
            populate: validationPopulate.populate
        };
    }
    // always populate createdBy, updatedBy, localizations etc.
    if (!isVisibleAttribute(model, attributeName)) {
        return true;
    }
    if (isManyRelation && countMany || !isManyRelation && countOne) {
        return {
            count: true
        };
    }
    return true;
}
/**
 * Populate the model for Dynamic Zone components
 * @param attribute - Attribute containing the components
 * @param attribute.components - IDs of components
 * @param options - Options to apply while populating
 */ function getPopulateForDZ(attribute, options, level) {
    // Use fragments to populate the dynamic zone components
    const populatedComponents = (attribute.components || []).reduce((acc, componentUID)=>({
            ...acc,
            [componentUID]: {
                populate: getDeepPopulate(componentUID, options, level + 1)
            }
        }), {});
    return {
        on: populatedComponents
    };
}
/**
 * Get the populated value based on the type of the attribute
 * @param attributeName - Name of the attribute
 * @param model - Model of the populated entity
 * @param model.attributes
 * @param options - Options to apply while populating
 * @param options.countMany
 * @param options.countOne
 * @param options.maxLevel
 * @param level
 */ function getPopulateFor(attributeName, model, options, level) {
    const attribute = model.attributes[attributeName];
    switch(attribute.type){
        case 'relation':
            // @ts-expect-error - TODO: support populate count typing
            return {
                [attributeName]: getPopulateForRelation(attribute, model, attributeName, options)
            };
        case 'component':
            return {
                [attributeName]: {
                    populate: getDeepPopulate(attribute.component, options, level + 1)
                }
            };
        case 'media':
            return {
                [attributeName]: {
                    populate: {
                        folder: true
                    }
                }
            };
        case 'dynamiczone':
            return {
                [attributeName]: getPopulateForDZ(attribute, options, level)
            };
        default:
            return {};
    }
}
/**
 * Deeply populate a model based on UID
 * @param uid - Unique identifier of the model
 * @param options - Options to apply while populating
 * @param level - Current level of nested call
 */ const getDeepPopulate = (uid, { initialPopulate = {}, countMany = false, countOne = false, maxLevel = Infinity } = {}, level = 1)=>{
    if (level > maxLevel) {
        return {};
    }
    const model = strapi.getModel(uid);
    if (!model) {
        return {};
    }
    return Object.keys(model.attributes).reduce((populateAcc, attributeName)=>merge(populateAcc, getPopulateFor(attributeName, model, {
            // @ts-expect-error - improve types
            initialPopulate: initialPopulate?.[attributeName],
            countMany,
            countOne,
            maxLevel
        }, level)), {});
};
/**
 * Deeply populate a model based on UID. Only populating fields that require validation.
 * @param uid - Unique identifier of the model
 * @param options - Options to apply while populating
 * @param level - Current level of nested call
 */ const getPopulateForValidation = (uid)=>{
    const model = strapi.getModel(uid);
    if (!model) {
        return {};
    }
    return Object.entries(model.attributes).reduce((populateAcc, [attributeName, attribute])=>{
        if (isScalarAttribute(attribute)) {
            // If the scalar attribute requires validation, add it to the fields array
            if (getDoesAttributeRequireValidation(attribute)) {
                populateAcc.fields = populateAcc.fields || [];
                populateAcc.fields.push(attributeName);
            }
            return populateAcc;
        }
        if (isMedia(attribute)) {
            if (getDoesAttributeRequireValidation(attribute)) {
                populateAcc.populate = populateAcc.populate || {};
                populateAcc.populate[attributeName] = {
                    populate: {
                        folder: true
                    }
                };
                return populateAcc;
            }
        }
        if (isComponent(attribute)) {
            // @ts-expect-error - should be a component
            const component = attribute.component;
            // Get the validation result for this component
            const componentResult = getPopulateForValidation(component);
            if (Object.keys(componentResult).length > 0) {
                populateAcc.populate = populateAcc.populate || {};
                populateAcc.populate[attributeName] = componentResult;
            }
            return populateAcc;
        }
        if (isDynamicZone(attribute)) {
            const components = attribute.components;
            // Handle dynamic zone components
            const componentsResult = (components || []).reduce((acc, componentUID)=>{
                // Get validation populate for this component
                const componentResult = getPopulateForValidation(componentUID);
                // Only include component if it has fields requiring validation
                if (Object.keys(componentResult).length > 0) {
                    acc[componentUID] = componentResult;
                }
                return acc;
            }, {});
            // Only add to populate if we have components requiring validation
            if (Object.keys(componentsResult).length > 0) {
                populateAcc.populate = populateAcc.populate || {};
                populateAcc.populate[attributeName] = {
                    on: componentsResult
                };
            }
        }
        return populateAcc;
    }, {});
};
/**
 * getDeepPopulateDraftCount works recursively on the attributes of a model
 * creating a populated object to count all the unpublished relations within the model
 * These relations can be direct to this content type or contained within components/dynamic zones
 * @param  uid of the model
 * @returns result
 * @returns result.populate
 * @returns result.hasRelations
 */ const getDeepPopulateDraftCount = (uid)=>{
    const model = strapi.getModel(uid);
    let hasRelations = false;
    const populate = Object.keys(model.attributes).reduce((populateAcc, attributeName)=>{
        const attribute = model.attributes[attributeName];
        switch(attribute.type){
            case 'relation':
                {
                    // TODO: Support polymorphic relations
                    const isMorphRelation = attribute.relation.toLowerCase().startsWith('morph');
                    if (isMorphRelation) {
                        break;
                    }
                    if (isVisibleAttribute(model, attributeName)) {
                        populateAcc[attributeName] = {
                            count: true,
                            filters: {
                                [PUBLISHED_AT_ATTRIBUTE]: {
                                    $null: true
                                }
                            }
                        };
                        hasRelations = true;
                    }
                    break;
                }
            case 'component':
                {
                    const { populate, hasRelations: childHasRelations } = getDeepPopulateDraftCount(attribute.component);
                    if (childHasRelations) {
                        populateAcc[attributeName] = {
                            populate
                        };
                        hasRelations = true;
                    }
                    break;
                }
            case 'dynamiczone':
                {
                    const dzPopulateFragment = attribute.components?.reduce((acc, componentUID)=>{
                        const { populate: componentPopulate, hasRelations: componentHasRelations } = getDeepPopulateDraftCount(componentUID);
                        if (componentHasRelations) {
                            hasRelations = true;
                            return {
                                ...acc,
                                [componentUID]: {
                                    populate: componentPopulate
                                }
                            };
                        }
                        return acc;
                    }, {});
                    if (!isEmpty(dzPopulateFragment)) {
                        populateAcc[attributeName] = {
                            on: dzPopulateFragment
                        };
                    }
                    break;
                }
        }
        return populateAcc;
    }, {});
    return {
        populate,
        hasRelations
    };
};
/**
 *  Create a Strapi populate object which populates all attribute fields of a Strapi query.
 */ const getQueryPopulate = async (uid, query)=>{
    let populateQuery = {};
    await strapiUtils.traverse.traverseQueryFilters(/**
     *
     * @param {Object} param0
     * @param {string} param0.key - Attribute name
     * @param {Object} param0.attribute - Attribute definition
     * @param {string} param0.path - Content Type path to the attribute
     * @returns
     */ ({ attribute, path })=>{
        // TODO: handle dynamic zones and morph relations
        if (!attribute || isDynamicZone(attribute) || isMorphToRelation(attribute)) {
            return;
        }
        // Populate all relations, components and media
        if (isRelation(attribute) || isMedia(attribute) || isComponent(attribute)) {
            const populatePath = path.attribute.replace(/\./g, '.populate.');
            // @ts-expect-error - lodash doesn't resolve the Populate type correctly
            populateQuery = set(populatePath, {}, populateQuery);
        }
    }, {
        schema: strapi.getModel(uid),
        getModel: strapi.getModel.bind(strapi)
    }, query);
    return populateQuery;
};
const buildDeepPopulate = (uid)=>{
    return getService('populate-builder')(uid).populateDeep(Infinity).countRelations().build();
};

export { buildDeepPopulate, getDeepPopulate, getDeepPopulateDraftCount, getPopulateForValidation, getQueryPopulate };
//# sourceMappingURL=populate.mjs.map
