import { stringify } from 'qs';
import { SINGLE_TYPES } from '../constants/collections.mjs';
import { contentManagerApi } from './api.mjs';

const documentApi = contentManagerApi.injectEndpoints({
    overrideExisting: true,
    endpoints: (builder)=>({
            autoCloneDocument: builder.mutation({
                query: ({ model, sourceId, params })=>({
                        url: `/content-manager/collection-types/${model}/auto-clone/${sourceId}`,
                        method: 'POST',
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, error, { model })=>{
                    if (error) {
                        return [];
                    }
                    return [
                        {
                            type: 'Document',
                            id: `${model}_LIST`
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ];
                }
            }),
            cloneDocument: builder.mutation({
                query: ({ model, sourceId, data, params })=>({
                        url: `/content-manager/collection-types/${model}/clone/${sourceId}`,
                        method: 'POST',
                        data,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, _error, { model })=>[
                        {
                            type: 'Document',
                            id: `${model}_LIST`
                        },
                        {
                            type: 'UidAvailability',
                            id: model
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ]
            }),
            /**
     * Creates a new collection-type document. This should ONLY be used for collection-types.
     * single-types should always be using `updateDocument` since they always exist.
     */ createDocument: builder.mutation({
                query: ({ model, data, params })=>({
                        url: `/content-manager/collection-types/${model}`,
                        method: 'POST',
                        data,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (result, _error, { model })=>[
                        {
                            type: 'Document',
                            id: `${model}_LIST`
                        },
                        'Relations',
                        {
                            type: 'UidAvailability',
                            id: model
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ],
                transformResponse: (response, meta, arg)=>{
                    /**
         * TODO v6
         * Adapt plugin:users-permissions.user to return the same response
         * shape as all other requests. The error is returned as expected.
         */ if (!('data' in response) && arg.model === 'plugin::users-permissions.user') {
                        return {
                            data: response,
                            meta: {
                                availableStatus: [],
                                availableLocales: []
                            }
                        };
                    }
                    return response;
                }
            }),
            deleteDocument: builder.mutation({
                query: ({ collectionType, model, documentId, params })=>({
                        url: `/content-manager/${collectionType}/${model}${collectionType !== SINGLE_TYPES && documentId ? `/${documentId}` : ''}`,
                        method: 'DELETE',
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, _error, { collectionType, model, documentId })=>[
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_LIST` : model
                        },
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        },
                        {
                            type: 'Document',
                            id: `${model}_ALL_ITEMS`
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ]
            }),
            deleteManyDocuments: builder.mutation({
                query: ({ model, params, ...body })=>({
                        url: `/content-manager/collection-types/${model}/actions/bulkDelete`,
                        method: 'POST',
                        data: body,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_res, _error, { model })=>[
                        {
                            type: 'Document',
                            id: `${model}_LIST`
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ]
            }),
            discardDocument: builder.mutation({
                query: ({ collectionType, model, documentId, params })=>({
                        url: documentId ? `/content-manager/${collectionType}/${model}/${documentId}/actions/discard` : `/content-manager/${collectionType}/${model}/actions/discard`,
                        method: 'POST',
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, _error, { collectionType, model, documentId })=>{
                    return [
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        },
                        {
                            type: 'Document',
                            id: `${model}_LIST`
                        },
                        'Relations',
                        {
                            type: 'UidAvailability',
                            id: model
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ];
                }
            }),
            /**
     * Gets all documents of a collection type or single type.
     * By passing different params you can get different results e.g. only published documents or 'es' documents.
     */ getAllDocuments: builder.query({
                query: ({ model, params })=>({
                        url: `/content-manager/collection-types/${model}`,
                        method: 'GET',
                        config: {
                            params: stringify(params, {
                                encode: true
                            })
                        }
                    }),
                providesTags: (result, _error, arg)=>{
                    return [
                        {
                            type: 'Document',
                            id: `ALL_LIST`
                        },
                        {
                            type: 'Document',
                            id: `${arg.model}_LIST`
                        },
                        ...result?.results.map(({ documentId })=>({
                                type: 'Document',
                                id: `${arg.model}_${documentId}`
                            })) ?? []
                    ];
                }
            }),
            getDraftRelationCount: builder.query({
                query: ({ collectionType, model, documentId, params })=>({
                        url: documentId ? `/content-manager/${collectionType}/${model}/${documentId}/actions/countDraftRelations` : `/content-manager/${collectionType}/${model}/actions/countDraftRelations`,
                        method: 'GET',
                        config: {
                            params
                        }
                    })
            }),
            getDocument: builder.query({
                // @ts-expect-error – TODO: fix ts error where data unknown doesn't work with response via an assertion?
                queryFn: async ({ collectionType, model, documentId, params }, _api, _extraOpts, baseQuery)=>{
                    const res = await baseQuery({
                        url: `/content-manager/${collectionType}/${model}${documentId ? `/${documentId}` : ''}`,
                        method: 'GET',
                        config: {
                            params
                        }
                    });
                    /**
         * To stop the query from locking itself in multiple retries, we intercept the error here and manage correctly.
         * This is because single-types don't have a list view and fetching them with the route `/single-types/:model`
         * never returns a list, just a single document but this won't exist if you've not made one before.
         */ if (res.error && res.error.name === 'NotFoundError' && collectionType === SINGLE_TYPES) {
                        return {
                            data: {
                                document: undefined
                            },
                            error: undefined
                        };
                    }
                    return res;
                },
                providesTags: (result, _error, { collectionType, model, documentId })=>{
                    return [
                        // we prefer the result's id because we don't fetch single-types with an ID.
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${result && 'documentId' in result ? result.documentId : documentId}` : model
                        },
                        // Make it easy to invalidate all individual documents queries for a model
                        {
                            type: 'Document',
                            id: `${model}_ALL_ITEMS`
                        }
                    ];
                }
            }),
            getManyDraftRelationCount: builder.query({
                query: ({ model, ...params })=>({
                        url: `/content-manager/collection-types/${model}/actions/countManyEntriesDraftRelations`,
                        method: 'GET',
                        config: {
                            params
                        }
                    }),
                transformResponse: (response)=>response.data
            }),
            /**
     * This endpoint will either create or update documents at the same time as publishing.
     */ publishDocument: builder.mutation({
                query: ({ collectionType, model, documentId, params, data })=>({
                        url: documentId ? `/content-manager/${collectionType}/${model}/${documentId}/actions/publish` : `/content-manager/${collectionType}/${model}/actions/publish`,
                        method: 'POST',
                        data,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, _error, { collectionType, model, documentId })=>{
                    return [
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        },
                        {
                            type: 'Document',
                            id: `${model}_LIST`
                        },
                        'Relations',
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList',
                        {
                            type: 'AILocalizationJobs',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        }
                    ];
                }
            }),
            publishManyDocuments: builder.mutation({
                query: ({ model, params, ...body })=>({
                        url: `/content-manager/collection-types/${model}/actions/bulkPublish`,
                        method: 'POST',
                        data: body,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_res, _error, { model, documentIds })=>{
                    return [
                        ...documentIds.map((id)=>({
                                type: 'Document',
                                id: `${model}_${id}`
                            })),
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ];
                }
            }),
            updateDocument: builder.mutation({
                query: ({ collectionType, model, documentId, data, params })=>({
                        url: `/content-manager/${collectionType}/${model}${documentId ? `/${documentId}` : ''}`,
                        method: 'PUT',
                        data,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, _error, { collectionType, model, documentId })=>{
                    return [
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        },
                        'Relations',
                        {
                            type: 'UidAvailability',
                            id: model
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList',
                        {
                            type: 'AILocalizationJobs',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        }
                    ];
                },
                async onQueryStarted ({ data, ...patch }, { dispatch, queryFulfilled }) {
                    // Optimistically update the cache with the new data
                    const patchResult = dispatch(documentApi.util.updateQueryData('getDocument', patch, (draft)=>{
                        Object.assign(draft.data, data);
                    }));
                    try {
                        await queryFulfilled;
                    } catch  {
                        // Rollback the optimistic update if there's an error
                        patchResult.undo();
                    }
                },
                transformResponse: (response, meta, arg)=>{
                    /**
         * TODO v6
         * Adapt plugin:users-permissions.user to return the same response
         * shape as all other requests. The error is returned as expected.
         */ if (!('data' in response) && arg.model === 'plugin::users-permissions.user') {
                        return {
                            data: response,
                            meta: {
                                availableStatus: [],
                                availableLocales: []
                            }
                        };
                    }
                    return response;
                }
            }),
            unpublishDocument: builder.mutation({
                query: ({ collectionType, model, documentId, params, data })=>({
                        url: documentId ? `/content-manager/${collectionType}/${model}/${documentId}/actions/unpublish` : `/content-manager/${collectionType}/${model}/actions/unpublish`,
                        method: 'POST',
                        data,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_result, _error, { collectionType, model, documentId })=>{
                    return [
                        {
                            type: 'Document',
                            id: collectionType !== SINGLE_TYPES ? `${model}_${documentId}` : model
                        },
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ];
                }
            }),
            unpublishManyDocuments: builder.mutation({
                query: ({ model, params, ...body })=>({
                        url: `/content-manager/collection-types/${model}/actions/bulkUnpublish`,
                        method: 'POST',
                        data: body,
                        config: {
                            params
                        }
                    }),
                invalidatesTags: (_res, _error, { model, documentIds })=>[
                        ...documentIds.map((id)=>({
                                type: 'Document',
                                id: `${model}_${id}`
                            })),
                        'RecentDocumentList',
                        'CountDocuments',
                        'UpcomingReleasesList'
                    ]
            })
        })
});
const { useAutoCloneDocumentMutation, useCloneDocumentMutation, useCreateDocumentMutation, useDeleteDocumentMutation, useDeleteManyDocumentsMutation, useDiscardDocumentMutation, useGetAllDocumentsQuery, useLazyGetDocumentQuery, useGetDocumentQuery, useLazyGetDraftRelationCountQuery, useGetManyDraftRelationCountQuery, usePublishDocumentMutation, usePublishManyDocumentsMutation, useUpdateDocumentMutation, useUnpublishDocumentMutation, useUnpublishManyDocumentsMutation } = documentApi;

export { useAutoCloneDocumentMutation, useCloneDocumentMutation, useCreateDocumentMutation, useDeleteDocumentMutation, useDeleteManyDocumentsMutation, useDiscardDocumentMutation, useGetAllDocumentsQuery, useGetDocumentQuery, useLazyGetDraftRelationCountQuery as useGetDraftRelationCountQuery, useGetManyDraftRelationCountQuery, useLazyGetDocumentQuery, usePublishDocumentMutation, usePublishManyDocumentsMutation, useUnpublishDocumentMutation, useUnpublishManyDocumentsMutation, useUpdateDocumentMutation };
//# sourceMappingURL=documents.mjs.map
