'use strict';

// NOTE: This override is for the properties on _user's site_, it's not about Strapi Admin.
/**
 * previewScript will be injected into the preview iframe after being stringified.
 * Therefore it CANNOT use any imports, or refer to any variables outside of its own scope.
 * It's why many functions are defined within previewScript, it's the only way to avoid going full spaghetti.
 * To get a better overview of everything previewScript does, go to the orchestration part at its end.
 */ const previewScript = (shouldRun = true)=>{
    /* -----------------------------------------------------------------------------------------------
   * Params
   * ---------------------------------------------------------------------------------------------*/ const HIGHLIGHT_PADDING = 2; // in pixels
    const HIGHLIGHT_HOVER_COLOR = window.STRAPI_HIGHLIGHT_HOVER_COLOR ?? '#4945ff'; // dark primary500
    const HIGHLIGHT_ACTIVE_COLOR = window.STRAPI_HIGHLIGHT_ACTIVE_COLOR ?? '#7b79ff'; // dark primary600
    const HIGHLIGHT_STYLES_ID = 'strapi-preview-highlight-styles';
    const DOUBLE_CLICK_TIMEOUT = 300; // milliseconds to wait for potential double-click
    const DISABLE_STEGA_DECODING = window.STRAPI_DISABLE_STEGA_DECODING ?? false;
    const SOURCE_ATTRIBUTE = 'data-strapi-source';
    const OVERLAY_ID = 'strapi-preview-overlay';
    const INTERNAL_EVENTS = {
        STRAPI_FIELD_FOCUS: 'strapiFieldFocus',
        STRAPI_FIELD_BLUR: 'strapiFieldBlur',
        STRAPI_FIELD_CHANGE: 'strapiFieldChange',
        STRAPI_FIELD_FOCUS_INTENT: 'strapiFieldFocusIntent',
        STRAPI_FIELD_SINGLE_CLICK_HINT: 'strapiFieldSingleClickHint'
    };
    /**
   * Calling the function in no-run mode lets us retrieve the constants from other files and keep
   * a single source of truth for them. It's the only way to do this because this script can't
   * refer to any variables outside of its own scope, because it's stringified before it's run.
   */ if (!shouldRun) {
        return {
            INTERNAL_EVENTS
        };
    }
    /* -----------------------------------------------------------------------------------------------
   * Utils
   * ---------------------------------------------------------------------------------------------*/ const sendMessage = (type, payload)=>{
        window.parent.postMessage({
            type,
            payload
        }, '*');
    };
    const getElementsByPath = (path)=>{
        return document.querySelectorAll(`[${SOURCE_ATTRIBUTE}*="path=${path}"]`);
    };
    /* -----------------------------------------------------------------------------------------------
   * Functionality pieces
   * ---------------------------------------------------------------------------------------------*/ const setupStegaDOMObserver = async ()=>{
        if (DISABLE_STEGA_DECODING) {
            return;
        }
        const { vercelStegaDecode: stegaDecode, vercelStegaClean: stegaClean } = await import(// @ts-expect-error it's not a local dependency
        // eslint-disable-next-line import/no-unresolved
        'https://cdn.jsdelivr.net/npm/@vercel/stega@0.1.2/+esm');
        const applyStegaToElement = (element)=>{
            const directTextNodes = Array.from(element.childNodes).filter((node)=>node.nodeType === Node.TEXT_NODE);
            const directTextContent = directTextNodes.map((node)=>node.textContent || '').join('');
            if (directTextContent) {
                try {
                    // TODO: check if we can call split instead of decode+clean
                    const result = stegaDecode(directTextContent);
                    if (result && 'strapiSource' in result) {
                        element.setAttribute(SOURCE_ATTRIBUTE, result.strapiSource);
                        // Remove encoded part from DOM text content (to avoid breaking links for example)
                        directTextNodes.forEach((node)=>{
                            if (node.textContent) {
                                const cleanedText = stegaClean(node.textContent);
                                if (cleanedText !== node.textContent) {
                                    node.textContent = cleanedText;
                                }
                            }
                        });
                    }
                } catch (error) {}
            }
        };
        // Process all existing elements
        const allElements = document.querySelectorAll('*');
        Array.from(allElements).forEach(applyStegaToElement);
        // Create observer for new elements and text changes
        const observer = new MutationObserver((mutations)=>{
            mutations.forEach((mutation)=>{
                // Handle added nodes
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach((node)=>{
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            const element = node;
                            // Process the added element
                            applyStegaToElement(element);
                            // Process all child elements
                            const childElements = element.querySelectorAll('*');
                            Array.from(childElements).forEach(applyStegaToElement);
                        }
                    });
                }
                // Handle text content changes
                if (mutation.type === 'characterData' && mutation.target.parentElement) {
                    applyStegaToElement(mutation.target.parentElement);
                }
            });
        });
        observer.observe(document, {
            childList: true,
            subtree: true,
            characterData: true
        });
        return observer;
    };
    const createHighlightStyles = ()=>{
        const existingStyles = document.getElementById(HIGHLIGHT_STYLES_ID);
        // Remove existing styles to avoid duplicates
        if (existingStyles) {
            existingStyles.remove();
        }
        const styleElement = document.createElement('style');
        styleElement.id = HIGHLIGHT_STYLES_ID;
        styleElement.textContent = `
      .strapi-highlight {
        position: absolute;
        outline: 2px solid transparent;
        pointer-events: auto;
        border-radius: 2px;
        background-color: transparent;
        will-change: transform;
        transition: outline-color 0.1s ease-in-out;
      }

      .strapi-highlight:hover {
        outline-color: ${HIGHLIGHT_HOVER_COLOR} !important;
      }

      .strapi-highlight.strapi-highlight-focused {
        outline-color: ${HIGHLIGHT_ACTIVE_COLOR} !important;
        outline-width: 3px !important;
      }
    `;
        document.head.appendChild(styleElement);
        return styleElement;
    };
    const createOverlaySystem = ()=>{
        // Clean up before creating a new overlay so we can safely call previewScript multiple times
        window.__strapi_previewCleanup?.();
        document.getElementById(OVERLAY_ID)?.remove();
        const overlay = document.createElement('div');
        overlay.id = OVERLAY_ID;
        overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      pointer-events: none;
      z-index: 9999;
    `;
        window.document.body.appendChild(overlay);
        return overlay;
    };
    const createHighlightManager = (overlay)=>{
        const elementsToHighlight = new Map();
        const eventListeners = [];
        const focusedHighlights = [];
        const pendingClicks = new Map(); // number is timeout id
        let focusedField = null;
        const drawHighlight = (target, highlight)=>{
            if (!highlight) return;
            const rect = target.getBoundingClientRect();
            highlight.style.width = `${rect.width + HIGHLIGHT_PADDING * 2}px`;
            highlight.style.height = `${rect.height + HIGHLIGHT_PADDING * 2}px`;
            highlight.style.transform = `translate(${rect.left - HIGHLIGHT_PADDING}px, ${rect.top - HIGHLIGHT_PADDING}px)`;
        };
        const updateAllHighlights = ()=>{
            elementsToHighlight.forEach((highlight, element)=>{
                drawHighlight(element, highlight);
            });
        };
        const createHighlightForElement = (element)=>{
            if (elementsToHighlight.has(element)) {
                // Already has a highlight
                return;
            }
            const highlight = document.createElement('div');
            highlight.className = 'strapi-highlight';
            const clickHandler = (event)=>{
                // Skip if this is a re-dispatched event from our delayed handler to avoid infinite loops
                if (event.__strapi_redispatched) {
                    return;
                }
                // Prevent the immediate action for interactive elements
                event.preventDefault();
                event.stopPropagation();
                // Clear any existing timeout for this element
                const existingTimeout = pendingClicks.get(element);
                if (existingTimeout) {
                    window.clearTimeout(existingTimeout);
                    pendingClicks.delete(element);
                }
                // Set up a delayed single-click handler
                const timeout = window.setTimeout(()=>{
                    pendingClicks.delete(element);
                    // Send single-click hint notification
                    sendMessage(INTERNAL_EVENTS.STRAPI_FIELD_SINGLE_CLICK_HINT, null);
                    // Re-trigger the click on the underlying element after the double-click timeout
                    // Create a new event to dispatch with a marker to prevent re-handling
                    const newEvent = new MouseEvent('click', {
                        bubbles: true,
                        cancelable: true,
                        view: window,
                        detail: 1,
                        button: event.button,
                        buttons: event.buttons,
                        clientX: event.clientX,
                        clientY: event.clientY,
                        ctrlKey: event.ctrlKey,
                        altKey: event.altKey,
                        shiftKey: event.shiftKey,
                        metaKey: event.metaKey
                    });
                    newEvent.__strapi_redispatched = true;
                    element.dispatchEvent(newEvent);
                }, DOUBLE_CLICK_TIMEOUT);
                pendingClicks.set(element, timeout);
            };
            const doubleClickHandler = (event)=>{
                // Prevent the default behavior on double-click
                event.preventDefault();
                event.stopPropagation();
                // Clear any pending single-click action
                const existingTimeout = pendingClicks.get(element);
                if (existingTimeout) {
                    clearTimeout(existingTimeout);
                    pendingClicks.delete(element);
                }
                const sourceAttribute = element.getAttribute(SOURCE_ATTRIBUTE);
                if (sourceAttribute) {
                    const rect = element.getBoundingClientRect();
                    sendMessage(INTERNAL_EVENTS.STRAPI_FIELD_FOCUS_INTENT, {
                        path: sourceAttribute,
                        position: {
                            top: rect.top,
                            left: rect.left,
                            right: rect.right,
                            bottom: rect.bottom,
                            width: rect.width,
                            height: rect.height
                        }
                    });
                }
            };
            const mouseDownHandler = (event)=>{
                // Prevent default multi click to select behavior
                if (event.detail >= 2) {
                    event.preventDefault();
                }
            };
            highlight.addEventListener('click', clickHandler);
            highlight.addEventListener('dblclick', doubleClickHandler);
            highlight.addEventListener('mousedown', mouseDownHandler);
            // Store event listeners for cleanup
            eventListeners.push({
                element: highlight,
                type: 'click',
                handler: clickHandler
            }, {
                element: highlight,
                type: 'dblclick',
                handler: doubleClickHandler
            }, {
                element: highlight,
                type: 'mousedown',
                handler: mouseDownHandler
            });
            elementsToHighlight.set(element, highlight);
            overlay.appendChild(highlight);
            drawHighlight(element, highlight);
        };
        const removeHighlightForElement = (element)=>{
            const highlight = elementsToHighlight.get(element);
            if (!highlight) return;
            // Clear any pending click timeout for this element
            const pendingTimeout = pendingClicks.get(element);
            if (pendingTimeout) {
                window.clearTimeout(pendingTimeout);
                pendingClicks.delete(element);
            }
            highlight.remove();
            elementsToHighlight.delete(element);
            // Remove event listeners for this highlight
            const listenersToRemove = eventListeners.filter((listener)=>listener.element === highlight);
            listenersToRemove.forEach(({ element, type, handler })=>{
                element.removeEventListener(type, handler);
            });
            // Mutate eventListeners to remove listeners for this highlight
            eventListeners.splice(0, eventListeners.length, ...eventListeners.filter((listener)=>listener.element !== highlight));
        };
        // Process all existing elements with source attributes
        const initialElements = window.document.querySelectorAll(`[${SOURCE_ATTRIBUTE}]`);
        Array.from(initialElements).forEach((element)=>{
            if (element instanceof HTMLElement) {
                createHighlightForElement(element);
            }
        });
        return {
            get elements () {
                return Array.from(elementsToHighlight.keys());
            },
            get highlights () {
                return Array.from(elementsToHighlight.values());
            },
            updateAllHighlights,
            eventListeners,
            focusedHighlights,
            createHighlightForElement,
            removeHighlightForElement,
            setFocusedField: (field)=>{
                focusedField = field;
            },
            getFocusedField: ()=>focusedField,
            clearAllPendingClicks: ()=>{
                pendingClicks.forEach((timeout)=>clearTimeout(timeout));
                pendingClicks.clear();
            }
        };
    };
    /**
   * We need to track scroll in all the element parents in order to keep the highlight position
   * in sync with the element position. Listening to window scroll is not enough because the
   * element can be inside one or more scrollable containers.
   */ const setupScrollManagement = (highlightManager)=>{
        const updateOnScroll = ()=>{
            highlightManager.updateAllHighlights();
        };
        const scrollableElements = new Set();
        scrollableElements.add(window);
        // Find all scrollable ancestors for all tracked elements and set up scroll listeners
        highlightManager.elements.forEach((element)=>{
            let parent = element.parentElement;
            while(parent){
                const computedStyle = window.getComputedStyle(parent);
                const overflow = computedStyle.overflow + computedStyle.overflowX + computedStyle.overflowY;
                if (overflow.includes('scroll') || overflow.includes('auto')) {
                    scrollableElements.add(parent);
                }
                parent = parent.parentElement;
            }
        });
        // Add scroll listeners to all scrollable elements
        scrollableElements.forEach((element)=>{
            if (element === window) {
                window.addEventListener('scroll', updateOnScroll);
                window.addEventListener('resize', updateOnScroll);
            } else {
                element.addEventListener('scroll', updateOnScroll);
            }
        });
        const cleanup = ()=>{
            scrollableElements.forEach((element)=>{
                if (element === window) {
                    window.removeEventListener('scroll', updateOnScroll);
                    window.removeEventListener('resize', updateOnScroll);
                } else {
                    element.removeEventListener('scroll', updateOnScroll);
                }
            });
        };
        return {
            cleanup
        };
    };
    const setupObservers = (highlightManager, stegaObserver)=>{
        const resizeObserver = new ResizeObserver(()=>{
            highlightManager.updateAllHighlights();
        });
        const observeElementForResize = (element)=>{
            resizeObserver.observe(element);
        };
        // Observe existing elements
        highlightManager.elements.forEach(observeElementForResize);
        resizeObserver.observe(document.documentElement);
        // Create highlight observer to watch for new elements with source attributes
        const highlightObserver = new MutationObserver((mutations)=>{
            mutations.forEach((mutation)=>{
                if (mutation.type === 'attributes' && mutation.attributeName === SOURCE_ATTRIBUTE) {
                    const target = mutation.target;
                    if (target.hasAttribute(SOURCE_ATTRIBUTE)) {
                        highlightManager.createHighlightForElement(target);
                        observeElementForResize(target);
                    } else {
                        highlightManager.removeHighlightForElement(target);
                    }
                }
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach((node)=>{
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            const element = node;
                            // Check if the added element has source attribute
                            if (element.hasAttribute(SOURCE_ATTRIBUTE) && element instanceof HTMLElement) {
                                highlightManager.createHighlightForElement(element);
                                observeElementForResize(element);
                            }
                            // Check all child elements for source attributes
                            const elementsWithSource = element.querySelectorAll(`[${SOURCE_ATTRIBUTE}]`);
                            Array.from(elementsWithSource).forEach((childElement)=>{
                                if (childElement instanceof HTMLElement) {
                                    highlightManager.createHighlightForElement(childElement);
                                    observeElementForResize(childElement);
                                }
                            });
                        }
                    });
                    mutation.removedNodes.forEach((node)=>{
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            const element = node;
                            highlightManager.removeHighlightForElement(element);
                        }
                    });
                }
            });
        });
        highlightObserver.observe(document, {
            childList: true,
            subtree: true,
            attributes: true,
            attributeFilter: [
                SOURCE_ATTRIBUTE
            ]
        });
        return {
            resizeObserver,
            highlightObserver,
            stegaObserver
        };
    };
    const setupEventHandlers = (highlightManager)=>{
        const handleMessage = (event)=>{
            if (!event.data?.type) return;
            // The user typed in an input, reflect the change in the preview
            if (event.data.type === INTERNAL_EVENTS.STRAPI_FIELD_CHANGE) {
                const { field, value } = event.data.payload;
                if (!field) return;
                getElementsByPath(field).forEach((element)=>{
                    if (element instanceof HTMLElement) {
                        element.textContent = value || '';
                    }
                });
                // Update highlight dimensions since the new text content may affect them
                highlightManager.updateAllHighlights();
                return;
            }
            // The user focused a new input, update the highlights in the preview
            if (event.data.type === INTERNAL_EVENTS.STRAPI_FIELD_FOCUS) {
                const { field } = event.data.payload;
                if (!field) return;
                // Clear existing focused highlights
                highlightManager.focusedHighlights.forEach((highlight)=>{
                    highlight.classList.remove('strapi-highlight-focused');
                });
                highlightManager.focusedHighlights.length = 0;
                // Set new focused field and highlight matching elements
                highlightManager.setFocusedField(field);
                getElementsByPath(field).forEach((element, index)=>{
                    if (index === 0) {
                        element.scrollIntoView({
                            behavior: 'smooth',
                            block: 'center'
                        });
                    }
                    const highlight = highlightManager.highlights[Array.from(highlightManager.elements).indexOf(element)];
                    if (highlight) {
                        highlight.classList.add('strapi-highlight-focused');
                        highlightManager.focusedHighlights.push(highlight);
                    }
                });
                return;
            }
            // The user is no longer focusing an input, remove the highlights
            if (event.data.type === INTERNAL_EVENTS.STRAPI_FIELD_BLUR) {
                const { field } = event.data.payload;
                if (field !== highlightManager.getFocusedField()) return;
                highlightManager.focusedHighlights.forEach((highlight)=>{
                    highlight.classList.remove('strapi-highlight-focused');
                });
                highlightManager.focusedHighlights.length = 0;
                highlightManager.setFocusedField(null);
            }
        };
        window.addEventListener('message', handleMessage);
        // Add the message handler to the cleanup list
        const messageEventListener = {
            element: window,
            type: 'message',
            handler: handleMessage
        };
        return [
            ...highlightManager.eventListeners,
            messageEventListener
        ];
    };
    const createCleanupSystem = (overlay, observers, scrollManager, eventHandlers, highlightManager)=>{
        window.__strapi_previewCleanup = ()=>{
            observers.resizeObserver.disconnect();
            observers.highlightObserver.disconnect();
            observers.stegaObserver?.disconnect();
            // Clean up scroll listeners
            scrollManager.cleanup();
            // Clear all pending click timeouts
            highlightManager.clearAllPendingClicks();
            // Remove highlight event listeners
            eventHandlers.forEach(({ element, type, handler })=>{
                element.removeEventListener(type, handler);
            });
            // Clean up CSS styles
            const existingStyles = document.getElementById(HIGHLIGHT_STYLES_ID);
            if (existingStyles) {
                existingStyles.remove();
            }
            overlay.remove();
        };
    };
    /* -----------------------------------------------------------------------------------------------
   * Orchestration
   * ---------------------------------------------------------------------------------------------*/ setupStegaDOMObserver().then((stegaObserver)=>{
        createHighlightStyles();
        const overlay = createOverlaySystem();
        const highlightManager = createHighlightManager(overlay);
        const observers = setupObservers(highlightManager, stegaObserver);
        const scrollManager = setupScrollManagement(highlightManager);
        const eventHandlers = setupEventHandlers(highlightManager);
        createCleanupSystem(overlay, observers, scrollManager, eventHandlers, highlightManager);
    });
};

exports.previewScript = previewScript;
//# sourceMappingURL=previewScript.js.map
