import { jsxs, jsx, Fragment } from 'react/jsx-runtime';
import 'react';
import { useStrapiApp, useQueryParams, useNotification, useClipboard, DescriptionComponentRenderer, useHistory } from '@strapi/admin/strapi-admin';
import { Typography, Tabs, Flex, IconButton } from '@strapi/design-system';
import { Link, Cross } from '@strapi/icons';
import { stringify } from 'qs';
import { useIntl } from 'react-intl';
import { Link as Link$1 } from 'react-router-dom';
import { styled } from 'styled-components';
import { InjectionZone } from '../../components/InjectionZone.mjs';
import { DocumentActionButton } from '../../pages/EditView/components/DocumentActions.mjs';
import { DocumentStatus } from '../../pages/EditView/components/DocumentStatus.mjs';
import { getDocumentStatus } from '../../pages/EditView/EditViewPage.mjs';
import { usePreviewContext } from '../pages/Preview.mjs';

/* -------------------------------------------------------------------------------------------------
 * ClosePreviewButton
 * -----------------------------------------------------------------------------------------------*/ const ClosePreviewButton = ()=>{
    const [{ query }] = useQueryParams();
    const { formatMessage } = useIntl();
    const canGoBack = useHistory('BackButton', (state)=>state.canGoBack);
    const goBack = useHistory('BackButton', (state)=>state.goBack);
    const history = useHistory('BackButton', (state)=>state.history);
    const locationIndex = useHistory('BackButton', (state)=>state.currentLocationIndex);
    /**
   * Get the link destination from the history.
   * Rely on a fallback (the parent edit view page) if there's no page to go back .
   */ const historyTo = canGoBack ? history.at(locationIndex - 2) : undefined;
    const fallback = {
        pathname: '..',
        search: stringify(query, {
            encode: false
        })
    };
    const toWithFallback = historyTo ?? fallback;
    const handleClick = (e)=>{
        if (canGoBack) {
            // Prevent normal link behavior, go back in the history stack instead
            e.preventDefault();
            goBack();
            return;
        }
    // Otherwise rely on native link behavior to go back to the edit view. We don't use navigate()
    // here in order to get the relative="path" functionality from the Link component.
    };
    return /*#__PURE__*/ jsx(IconButton, {
        variant: "ghost",
        tag: Link$1,
        relative: "path",
        to: toWithFallback,
        onClick: handleClick,
        label: formatMessage({
            id: 'content-manager.preview.header.close',
            defaultMessage: 'Close preview'
        }),
        children: /*#__PURE__*/ jsx(Cross, {})
    });
};
/* -------------------------------------------------------------------------------------------------
 * Status
 * -----------------------------------------------------------------------------------------------*/ const Status = ()=>{
    // Get status
    const document = usePreviewContext('PreviewHeader', (state)=>state.document);
    const schema = usePreviewContext('PreviewHeader', (state)=>state.schema);
    const meta = usePreviewContext('PreviewHeader', (state)=>state.meta);
    const hasDraftAndPublished = schema?.options?.draftAndPublish ?? false;
    if (!hasDraftAndPublished) {
        return null;
    }
    const status = getDocumentStatus(document, meta);
    return /*#__PURE__*/ jsx(DocumentStatus, {
        status: status,
        size: "XS"
    });
};
const PreviewTabs = ()=>{
    const { formatMessage } = useIntl();
    // URL query params
    const [{ query }, setQuery] = useQueryParams();
    // Get status
    const document = usePreviewContext('PreviewHeader', (state)=>state.document);
    const schema = usePreviewContext('PreviewHeader', (state)=>state.schema);
    const meta = usePreviewContext('PreviewHeader', (state)=>state.meta);
    const hasDraftAndPublish = schema?.options?.draftAndPublish ?? false;
    const documentStatus = getDocumentStatus(document, meta);
    const handleTabChange = (status)=>{
        if (status === 'published' || status === 'draft') {
            setQuery({
                status
            }, 'push', true);
        }
    };
    if (!hasDraftAndPublish) {
        return null;
    }
    return /*#__PURE__*/ jsx(Tabs.Root, {
        variant: "simple",
        value: query.status || 'draft',
        onValueChange: handleTabChange,
        children: /*#__PURE__*/ jsxs(Tabs.List, {
            "aria-label": formatMessage({
                id: 'preview.tabs.label',
                defaultMessage: 'Document status'
            }),
            children: [
                /*#__PURE__*/ jsx(StatusTab, {
                    value: "draft",
                    children: formatMessage({
                        id: 'content-manager.containers.List.draft',
                        defaultMessage: 'draft'
                    })
                }),
                /*#__PURE__*/ jsx(StatusTab, {
                    value: "published",
                    disabled: documentStatus === 'draft',
                    children: formatMessage({
                        id: 'content-manager.containers.List.published',
                        defaultMessage: 'published'
                    })
                })
            ]
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * PreviewHeader
 * -----------------------------------------------------------------------------------------------*/ const PreviewHeader = ()=>{
    const title = usePreviewContext('PreviewHeader', (state)=>state.title);
    const document = usePreviewContext('PreviewHeader', (state)=>state.document);
    const schema = usePreviewContext('PreviewHeader', (state)=>state.schema);
    const meta = usePreviewContext('PreviewHeader', (state)=>state.meta);
    const plugins = useStrapiApp('PreviewHeader', (state)=>state.plugins);
    const [{ query }] = useQueryParams();
    const { formatMessage } = useIntl();
    const { toggleNotification } = useNotification();
    const { copy } = useClipboard();
    const handleCopyLink = ()=>{
        copy(window.location.href);
        toggleNotification({
            message: formatMessage({
                id: 'content-manager.preview.copy.success',
                defaultMessage: 'Copied preview link'
            }),
            type: 'success'
        });
    };
    const hasDraftAndPublish = schema.options?.draftAndPublish ?? false;
    const documentActionProps = {
        activeTab: query.status ?? null,
        collectionType: schema.kind === 'collectionType' ? 'collection-types' : 'single-types',
        model: schema.uid,
        documentId: schema.kind === 'collectionType' ? document.documentId : undefined,
        document,
        meta
    };
    return /*#__PURE__*/ jsxs(Flex, {
        height: "48px",
        gap: 4,
        background: "neutral0",
        borderColor: "neutral150",
        tag: "header",
        children: [
            /*#__PURE__*/ jsxs(TitleContainer, {
                height: "100%",
                paddingLeft: 2,
                paddingRight: 4,
                children: [
                    /*#__PURE__*/ jsx(ClosePreviewButton, {}),
                    /*#__PURE__*/ jsx(PreviewTitle, {
                        tag: "h1",
                        title: title,
                        maxWidth: "200px",
                        fontSize: 2,
                        paddingLeft: 2,
                        paddingRight: 3,
                        fontWeight: 600,
                        children: title
                    }),
                    /*#__PURE__*/ jsx(Status, {})
                ]
            }),
            /*#__PURE__*/ jsxs(Flex, {
                flex: 1,
                paddingRight: 2,
                gap: 2,
                justifyContent: hasDraftAndPublish ? 'space-between' : 'flex-end',
                children: [
                    /*#__PURE__*/ jsx(Flex, {
                        flex: "1 1 70%",
                        children: /*#__PURE__*/ jsx(PreviewTabs, {})
                    }),
                    /*#__PURE__*/ jsxs(Flex, {
                        gap: 2,
                        children: [
                            /*#__PURE__*/ jsx(IconButton, {
                                label: formatMessage({
                                    id: 'preview.copy.label',
                                    defaultMessage: 'Copy preview link'
                                }),
                                onClick: handleCopyLink,
                                children: /*#__PURE__*/ jsx(Link, {})
                            }),
                            /*#__PURE__*/ jsx(InjectionZone, {
                                area: "preview.actions"
                            }),
                            /*#__PURE__*/ jsx(DescriptionComponentRenderer, {
                                props: documentActionProps,
                                descriptions: plugins['content-manager'].apis.getDocumentActions('preview'),
                                children: (actions)=>{
                                    const filteredActions = actions.filter((action)=>[
                                            action.position
                                        ].flat().includes('preview'));
                                    const [primaryAction, secondaryAction] = filteredActions;
                                    if (!primaryAction && !secondaryAction) return null;
                                    // Both actions are available when draft and publish enabled
                                    if (primaryAction && secondaryAction) {
                                        return /*#__PURE__*/ jsxs(Fragment, {
                                            children: [
                                                /*#__PURE__*/ jsx(DocumentActionButton, {
                                                    ...secondaryAction,
                                                    variant: secondaryAction.variant || 'secondary'
                                                }),
                                                /*#__PURE__*/ jsx(DocumentActionButton, {
                                                    ...primaryAction,
                                                    variant: primaryAction.variant || 'default'
                                                })
                                            ]
                                        });
                                    }
                                    // Otherwise we just have the save action
                                    return /*#__PURE__*/ jsx(DocumentActionButton, {
                                        ...primaryAction,
                                        variant: primaryAction.variant || 'secondary'
                                    });
                                }
                            })
                        ]
                    })
                ]
            })
        ]
    });
};
const PreviewTitle = styled(Typography)`
  overflow: hidden;
  text-overflow: ellipsis;
  white-space: nowrap;
`;
const StatusTab = styled(Tabs.Trigger)`
  text-transform: uppercase;
`;
const TitleContainer = styled(Flex)`
  border-right: 1px solid ${({ theme })=>theme.colors.neutral150};
`;

export { PreviewHeader };
//# sourceMappingURL=PreviewHeader.mjs.map
