import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { useTable, Table, useQueryParams, getYupValidationErrors } from '@strapi/admin/strapi-admin';
import { Typography, RawTable, Modal, Box, Button, Flex, Loader, IconButton, Tbody, Tr, Td, Tooltip } from '@strapi/design-system';
import { Pencil, CheckCircle, ArrowsCounterClockwise, CrossCircle } from '@strapi/icons';
import { useIntl } from 'react-intl';
import { useLocation, Link } from 'react-router-dom';
import { styled } from 'styled-components';
import { ValidationError } from 'yup';
import { useDocumentRBAC } from '../../../../features/DocumentRBAC.mjs';
import { useContentTypeSchema } from '../../../../hooks/useContentTypeSchema.mjs';
import { useDocumentActions } from '../../../../hooks/useDocumentActions.mjs';
import { useDocLayout } from '../../../../hooks/useDocumentLayout.mjs';
import { contentManagerApi } from '../../../../services/api.mjs';
import { useGetAllDocumentsQuery, usePublishManyDocumentsMutation } from '../../../../services/documents.mjs';
import { buildValidParams } from '../../../../utils/api.mjs';
import { getTranslation } from '../../../../utils/translations.mjs';
import { createYupSchema } from '../../../../utils/validation.mjs';
import { DocumentStatus } from '../../../EditView/components/DocumentStatus.mjs';
import { ConfirmDialogPublishAll } from './ConfirmBulkActionDialog.mjs';

const TypographyMaxWidth = styled(Typography)`
  max-width: 300px;
`;
const TableComponent = styled(RawTable)`
  width: 100%;
  table-layout: fixed;
  td:first-child {
    border-right: 1px solid ${({ theme })=>theme.colors.neutral150};
  }
  td:first-of-type {
    padding: ${({ theme })=>theme.spaces[4]};
  }
`;
/* -------------------------------------------------------------------------------------------------
 * EntryValidationText
 * -----------------------------------------------------------------------------------------------*/ const formatErrorMessages = (errors, parentKey, formatMessage)=>{
    const messages = [];
    Object.entries(errors).forEach(([key, value])=>{
        const currentKey = parentKey ? `${parentKey}.${key}` : key;
        if (typeof value === 'object' && value !== null && !Array.isArray(value)) {
            if ('id' in value && 'defaultMessage' in value) {
                messages.push(formatMessage({
                    id: `${value.id}.withField`,
                    defaultMessage: value.defaultMessage
                }, {
                    field: currentKey
                }));
            } else {
                messages.push(...formatErrorMessages(// @ts-expect-error TODO: check why value is not compatible with FormErrors
                value, currentKey, formatMessage));
            }
        } else {
            messages.push(formatMessage({
                id: `${value}.withField`,
                defaultMessage: value
            }, {
                field: currentKey
            }));
        }
    });
    return messages;
};
const EntryValidationText = ({ validationErrors, status })=>{
    const { formatMessage } = useIntl();
    if (validationErrors) {
        const validationErrorsMessages = formatErrorMessages(validationErrors, '', formatMessage).join(' ');
        return /*#__PURE__*/ jsxs(Flex, {
            gap: 2,
            children: [
                /*#__PURE__*/ jsx(CrossCircle, {
                    fill: "danger600"
                }),
                /*#__PURE__*/ jsx(Tooltip, {
                    label: validationErrorsMessages,
                    children: /*#__PURE__*/ jsx(TypographyMaxWidth, {
                        textColor: "danger600",
                        variant: "omega",
                        fontWeight: "bold",
                        ellipsis: true,
                        children: validationErrorsMessages
                    })
                })
            ]
        });
    }
    if (status === 'published') {
        return /*#__PURE__*/ jsxs(Flex, {
            gap: 2,
            children: [
                /*#__PURE__*/ jsx(CheckCircle, {
                    fill: "success600"
                }),
                /*#__PURE__*/ jsx(Typography, {
                    textColor: "success600",
                    fontWeight: "bold",
                    children: formatMessage({
                        id: 'content-manager.bulk-publish.already-published',
                        defaultMessage: 'Already Published'
                    })
                })
            ]
        });
    }
    if (status === 'modified') {
        return /*#__PURE__*/ jsxs(Flex, {
            gap: 2,
            children: [
                /*#__PURE__*/ jsx(ArrowsCounterClockwise, {
                    fill: "alternative600"
                }),
                /*#__PURE__*/ jsx(Typography, {
                    textColor: "alternative600",
                    fontWeight: "bold",
                    children: formatMessage({
                        id: 'content-manager.bulk-publish.modified',
                        defaultMessage: 'Ready to publish changes'
                    })
                })
            ]
        });
    }
    return /*#__PURE__*/ jsxs(Flex, {
        gap: 2,
        children: [
            /*#__PURE__*/ jsx(CheckCircle, {
                fill: "success600"
            }),
            /*#__PURE__*/ jsx(Typography, {
                textColor: "success600",
                fontWeight: "bold",
                children: formatMessage({
                    id: 'app.utils.ready-to-publish',
                    defaultMessage: 'Ready to publish'
                })
            })
        ]
    });
};
const TABLE_HEADERS = [
    {
        name: 'id',
        label: 'id'
    },
    {
        name: 'name',
        label: 'name'
    },
    {
        name: 'status',
        label: 'status'
    },
    {
        name: 'publicationStatus',
        label: 'Publication status'
    }
];
const SelectedEntriesTableContent = ({ isPublishing, rowsToDisplay = [], entriesToPublish = [], validationErrors = {} })=>{
    const { pathname } = useLocation();
    const { formatMessage } = useIntl();
    const { list: { settings: { mainField } } } = useDocLayout();
    const shouldDisplayMainField = mainField != null && mainField !== 'id';
    return /*#__PURE__*/ jsxs(Table.Content, {
        children: [
            /*#__PURE__*/ jsxs(Table.Head, {
                children: [
                    /*#__PURE__*/ jsx(Table.HeaderCheckboxCell, {}),
                    TABLE_HEADERS.filter((head)=>head.name !== 'name' || shouldDisplayMainField).map((head)=>/*#__PURE__*/ jsx(Table.HeaderCell, {
                            ...head
                        }, head.name))
                ]
            }),
            /*#__PURE__*/ jsx(Table.Loading, {}),
            /*#__PURE__*/ jsx(Table.Body, {
                children: rowsToDisplay.map((row)=>/*#__PURE__*/ jsxs(Table.Row, {
                        children: [
                            /*#__PURE__*/ jsx(Table.CheckboxCell, {
                                id: row.id
                            }),
                            /*#__PURE__*/ jsx(Table.Cell, {
                                children: /*#__PURE__*/ jsx(Typography, {
                                    children: row.id
                                })
                            }),
                            shouldDisplayMainField && /*#__PURE__*/ jsx(Table.Cell, {
                                children: /*#__PURE__*/ jsx(Typography, {
                                    children: row[mainField]
                                })
                            }),
                            /*#__PURE__*/ jsx(Table.Cell, {
                                children: /*#__PURE__*/ jsx(DocumentStatus, {
                                    status: row.status,
                                    maxWidth: 'min-content'
                                })
                            }),
                            /*#__PURE__*/ jsx(Table.Cell, {
                                children: isPublishing && entriesToPublish.includes(row.documentId) ? /*#__PURE__*/ jsxs(Flex, {
                                    gap: 2,
                                    children: [
                                        /*#__PURE__*/ jsx(Typography, {
                                            children: formatMessage({
                                                id: 'content-manager.success.record.publishing',
                                                defaultMessage: 'Publishing...'
                                            })
                                        }),
                                        /*#__PURE__*/ jsx(Loader, {
                                            small: true
                                        })
                                    ]
                                }) : /*#__PURE__*/ jsx(EntryValidationText, {
                                    validationErrors: validationErrors[row.documentId],
                                    status: row.status
                                })
                            }),
                            /*#__PURE__*/ jsx(Table.Cell, {
                                children: /*#__PURE__*/ jsx(Flex, {
                                    children: /*#__PURE__*/ jsx(IconButton, {
                                        tag: Link,
                                        to: {
                                            pathname: `${pathname}/${row.documentId}`,
                                            search: row.locale && `?plugins[i18n][locale]=${row.locale}`
                                        },
                                        state: {
                                            from: pathname
                                        },
                                        label: formatMessage({
                                            id: 'content-manager.bulk-publish.edit',
                                            defaultMessage: 'Edit'
                                        }),
                                        target: "_blank",
                                        marginLeft: "auto",
                                        variant: "ghost",
                                        children: /*#__PURE__*/ jsx(Pencil, {
                                            width: '1.6rem',
                                            height: '1.6rem'
                                        })
                                    })
                                })
                            })
                        ]
                    }, row.id))
            })
        ]
    });
};
const PublicationStatusSummary = ({ count, icon, message, textColor })=>{
    return /*#__PURE__*/ jsxs(Flex, {
        justifyContent: "space-between",
        flex: 1,
        gap: 3,
        children: [
            /*#__PURE__*/ jsxs(Flex, {
                gap: 2,
                children: [
                    icon,
                    /*#__PURE__*/ jsx(Typography, {
                        textColor: textColor,
                        fontWeight: "bold",
                        children: message
                    })
                ]
            }),
            /*#__PURE__*/ jsx(Typography, {
                fontWeight: "bold",
                children: count
            })
        ]
    });
};
const PublicationStatusGrid = ({ entriesReadyToPublishCount, entriesPublishedCount, entriesModifiedCount, entriesWithErrorsCount })=>{
    const { formatMessage } = useIntl();
    return /*#__PURE__*/ jsx(Box, {
        hasRadius: true,
        borderColor: "neutral150",
        children: /*#__PURE__*/ jsx(TableComponent, {
            colCount: 2,
            rowCount: 2,
            children: /*#__PURE__*/ jsxs(Tbody, {
                children: [
                    /*#__PURE__*/ jsxs(Tr, {
                        children: [
                            /*#__PURE__*/ jsx(Td, {
                                children: /*#__PURE__*/ jsx(PublicationStatusSummary, {
                                    textColor: "success600",
                                    count: entriesReadyToPublishCount,
                                    icon: /*#__PURE__*/ jsx(CheckCircle, {
                                        fill: "success600"
                                    }),
                                    message: formatMessage({
                                        id: 'app.utils.ready-to-publish',
                                        defaultMessage: 'Ready to publish'
                                    })
                                })
                            }),
                            /*#__PURE__*/ jsx(Td, {
                                children: /*#__PURE__*/ jsx(PublicationStatusSummary, {
                                    textColor: "success600",
                                    count: entriesPublishedCount,
                                    icon: /*#__PURE__*/ jsx(CheckCircle, {
                                        fill: "success600"
                                    }),
                                    message: formatMessage({
                                        id: 'app.utils.already-published',
                                        defaultMessage: 'Already published'
                                    })
                                })
                            })
                        ]
                    }),
                    /*#__PURE__*/ jsxs(Tr, {
                        children: [
                            /*#__PURE__*/ jsx(Td, {
                                children: /*#__PURE__*/ jsx(PublicationStatusSummary, {
                                    textColor: "alternative600",
                                    count: entriesModifiedCount,
                                    icon: /*#__PURE__*/ jsx(ArrowsCounterClockwise, {
                                        fill: "alternative600"
                                    }),
                                    message: formatMessage({
                                        id: 'content-manager.bulk-publish.modified',
                                        defaultMessage: 'Ready to publish changes'
                                    })
                                })
                            }),
                            /*#__PURE__*/ jsx(Td, {
                                children: /*#__PURE__*/ jsx(PublicationStatusSummary, {
                                    textColor: "danger600",
                                    count: entriesWithErrorsCount,
                                    icon: /*#__PURE__*/ jsx(CrossCircle, {
                                        fill: "danger600"
                                    }),
                                    message: formatMessage({
                                        id: 'content-manager.bulk-publish.waiting-for-action',
                                        defaultMessage: 'Waiting for action'
                                    })
                                })
                            })
                        ]
                    })
                ]
            })
        })
    });
};
const SelectedEntriesModalContent = ({ listViewSelectedEntries, toggleModal, setListViewSelectedDocuments, model })=>{
    const { formatMessage } = useIntl();
    const { schema, components } = useContentTypeSchema(model);
    const documentIds = listViewSelectedEntries.map(({ documentId })=>documentId);
    // We want to keep the selected entries order same as the list view
    const [{ query }] = useQueryParams();
    const params = React.useMemo(()=>buildValidParams(query), [
        query
    ]);
    // Fetch the documents based on the selected entries and update the modal table
    const { data, isLoading, isFetching, refetch } = useGetAllDocumentsQuery({
        model,
        params: {
            page: '1',
            pageSize: documentIds.length.toString(),
            sort: query.sort,
            filters: {
                documentId: {
                    $in: documentIds
                }
            },
            locale: query.plugins?.i18n?.locale
        }
    }, {
        selectFromResult: ({ data, ...restRes })=>({
                data: data?.results ?? [],
                ...restRes
            })
    });
    // Validate the entries based on the schema to show errors if any
    const { rows, validationErrors } = React.useMemo(()=>{
        if (data.length > 0 && schema) {
            const validate = createYupSchema(schema.attributes, components, // Since this is the "Publish" action, the validation
            // schema must enforce the rules for published entities
            {
                status: 'published'
            });
            const validationErrors = {};
            const rows = data.map((entry)=>{
                try {
                    validate.validateSync(entry, {
                        abortEarly: false
                    });
                    return entry;
                } catch (e) {
                    if (e instanceof ValidationError) {
                        validationErrors[entry.documentId] = getYupValidationErrors(e);
                    }
                    return entry;
                }
            });
            return {
                rows,
                validationErrors
            };
        }
        return {
            rows: [],
            validationErrors: {}
        };
    }, [
        components,
        data,
        schema
    ]);
    const [isDialogOpen, setIsDialogOpen] = React.useState(false);
    const { publishMany: bulkPublishAction, isLoading: isPublishing } = useDocumentActions();
    const [, { isLoading: isSubmittingForm }] = usePublishManyDocumentsMutation();
    const selectedRows = useTable('publishAction', (state)=>state.selectedRows);
    // Filter selected entries from the updated modal table rows
    const selectedEntries = rows.filter((entry)=>selectedRows.some((selectedEntry)=>selectedEntry.documentId === entry.documentId));
    const entriesToPublish = selectedEntries.filter((entry)=>!validationErrors[entry.documentId]).map((entry)=>entry.documentId);
    const selectedEntriesWithErrorsCount = selectedEntries.filter(({ documentId })=>validationErrors[documentId]).length;
    const selectedEntriesPublishedCount = selectedEntries.filter(({ status })=>status === 'published').length;
    const selectedEntriesModifiedCount = selectedEntries.filter(({ status, documentId })=>status === 'modified' && !validationErrors[documentId]).length;
    const selectedEntriesWithNoErrorsCount = selectedEntries.length - selectedEntriesWithErrorsCount - selectedEntriesPublishedCount;
    const toggleDialog = ()=>setIsDialogOpen((prev)=>!prev);
    const handleConfirmBulkPublish = async ()=>{
        toggleDialog();
        const res = await bulkPublishAction({
            model: model,
            documentIds: entriesToPublish,
            params
        });
        if (!('error' in res)) {
            const unpublishedEntries = rows.filter((row)=>{
                return !entriesToPublish.includes(row.documentId);
            });
            // Keep selection of the entries in list view that were not published
            setListViewSelectedDocuments(unpublishedEntries);
        }
    };
    return /*#__PURE__*/ jsxs(Fragment, {
        children: [
            /*#__PURE__*/ jsxs(Modal.Body, {
                children: [
                    /*#__PURE__*/ jsx(PublicationStatusGrid, {
                        entriesReadyToPublishCount: selectedEntriesWithNoErrorsCount - selectedEntriesModifiedCount,
                        entriesPublishedCount: selectedEntriesPublishedCount,
                        entriesModifiedCount: selectedEntriesModifiedCount,
                        entriesWithErrorsCount: selectedEntriesWithErrorsCount
                    }),
                    /*#__PURE__*/ jsx(Box, {
                        marginTop: 7,
                        children: /*#__PURE__*/ jsx(SelectedEntriesTableContent, {
                            isPublishing: isSubmittingForm,
                            rowsToDisplay: rows,
                            entriesToPublish: entriesToPublish,
                            validationErrors: validationErrors
                        })
                    })
                ]
            }),
            /*#__PURE__*/ jsxs(Modal.Footer, {
                children: [
                    /*#__PURE__*/ jsx(Button, {
                        onClick: toggleModal,
                        variant: "tertiary",
                        children: formatMessage({
                            id: 'app.components.Button.cancel',
                            defaultMessage: 'Cancel'
                        })
                    }),
                    /*#__PURE__*/ jsxs(Flex, {
                        gap: 2,
                        children: [
                            /*#__PURE__*/ jsx(Button, {
                                onClick: refetch,
                                variant: "tertiary",
                                loading: isFetching,
                                children: formatMessage({
                                    id: 'app.utils.refresh',
                                    defaultMessage: 'Refresh'
                                })
                            }),
                            /*#__PURE__*/ jsx(Button, {
                                onClick: toggleDialog,
                                disabled: selectedEntries.length === 0 || selectedEntries.length === selectedEntriesWithErrorsCount || selectedEntriesPublishedCount === selectedEntries.length || isLoading,
                                loading: isPublishing || isSubmittingForm,
                                children: formatMessage({
                                    id: 'app.utils.publish',
                                    defaultMessage: 'Publish'
                                })
                            })
                        ]
                    })
                ]
            }),
            /*#__PURE__*/ jsx(ConfirmDialogPublishAll, {
                isOpen: isDialogOpen,
                onToggleDialog: toggleDialog,
                isConfirmButtonLoading: isPublishing || isSubmittingForm,
                onConfirm: handleConfirmBulkPublish
            })
        ]
    });
};
/* -------------------------------------------------------------------------------------------------
 * PublishAction
 * -----------------------------------------------------------------------------------------------*/ const PublishAction = ({ documents, model })=>{
    const { formatMessage } = useIntl();
    // Publish button visibility
    const hasPublishPermission = useDocumentRBAC('unpublishAction', (state)=>state.canPublish);
    const showPublishButton = hasPublishPermission && documents.some(({ status })=>status !== 'published');
    const setListViewSelectedDocuments = useTable('publishAction', (state)=>state.selectRow);
    const refetchList = ()=>{
        contentManagerApi.util.invalidateTags([
            {
                type: 'Document',
                id: `${model}_LIST`
            }
        ]);
    };
    if (!showPublishButton) return null;
    return {
        actionType: 'publish',
        variant: 'tertiary',
        label: formatMessage({
            id: 'app.utils.publish',
            defaultMessage: 'Publish'
        }),
        dialog: {
            type: 'modal',
            title: formatMessage({
                id: getTranslation('containers.ListPage.selectedEntriesModal.title'),
                defaultMessage: 'Publish entries'
            }),
            content: ({ onClose })=>{
                return /*#__PURE__*/ jsx(Table.Root, {
                    rows: documents,
                    defaultSelectedRows: documents,
                    headers: TABLE_HEADERS,
                    children: /*#__PURE__*/ jsx(SelectedEntriesModalContent, {
                        listViewSelectedEntries: documents,
                        toggleModal: ()=>{
                            onClose();
                            refetchList();
                        },
                        setListViewSelectedDocuments: setListViewSelectedDocuments,
                        model: model
                    })
                });
            },
            onClose: ()=>{
                refetchList();
            }
        }
    };
};

export { PublishAction, SelectedEntriesModalContent };
//# sourceMappingURL=PublishAction.mjs.map
