import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import * as React from 'react';
import { Layouts, useTracking, useNotification, useAPIErrorHandler, useQueryParams, useStrapiApp, Page, tours, BackButton, SearchInput, Table, Pagination, useTable, useRBAC } from '@strapi/admin/strapi-admin';
import { Box, EmptyStateLayout, Flex, Typography, Button } from '@strapi/design-system';
import { Plus } from '@strapi/icons';
import { EmptyDocuments } from '@strapi/icons/symbols';
import isEqual from 'lodash/isEqual';
import { stringify } from 'qs';
import { useIntl } from 'react-intl';
import { useNavigate, Link, useParams } from 'react-router-dom';
import { styled } from 'styled-components';
import { InjectionZone } from '../../components/InjectionZone.mjs';
import { HOOKS } from '../../constants/hooks.mjs';
import { PERMISSIONS } from '../../constants/plugin.mjs';
import { useDocumentRBAC, DocumentRBAC } from '../../features/DocumentRBAC.mjs';
import { useDoc } from '../../hooks/useDocument.mjs';
import { useDocumentLayout, convertListLayoutToFieldLayouts } from '../../hooks/useDocumentLayout.mjs';
import { usePrev } from '../../hooks/usePrev.mjs';
import { useGetAllDocumentsQuery } from '../../services/documents.mjs';
import { buildValidParams } from '../../utils/api.mjs';
import { getTranslation } from '../../utils/translations.mjs';
import { getDisplayName } from '../../utils/users.mjs';
import { DocumentStatus } from '../EditView/components/DocumentStatus.mjs';
import { BulkActionsRenderer } from './components/BulkActions/Actions.mjs';
import { Filters as FiltersImpl } from './components/Filters.mjs';
import { TableActions } from './components/TableActions.mjs';
import { CellContent } from './components/TableCells/CellContent.mjs';
import { ViewSettingsMenu } from './components/ViewSettingsMenu.mjs';

const { INJECT_COLUMN_IN_TABLE } = HOOKS;
/* -------------------------------------------------------------------------------------------------
 * ListViewPage
 * -----------------------------------------------------------------------------------------------*/ const LayoutsHeaderCustom = styled(Layouts.Header)`
  overflow-wrap: anywhere;
`;
const ListViewPage = ()=>{
    const { trackUsage } = useTracking();
    const navigate = useNavigate();
    const { formatMessage } = useIntl();
    const { toggleNotification } = useNotification();
    const { _unstableFormatAPIError: formatAPIError } = useAPIErrorHandler(getTranslation);
    const { collectionType, model, schema } = useDoc();
    const { list } = useDocumentLayout(model);
    const [displayedHeaders, setDisplayedHeaders] = React.useState([]);
    const listLayout = usePrev(list.layout);
    React.useEffect(()=>{
        /**
     * ONLY update the displayedHeaders if the document
     * layout has actually changed in value.
     */ if (!isEqual(listLayout, list.layout)) {
            setDisplayedHeaders(list.layout);
        }
    }, [
        list.layout,
        listLayout
    ]);
    const handleSetHeaders = (headers)=>{
        setDisplayedHeaders(convertListLayoutToFieldLayouts(headers, schema.attributes, list.metadatas));
    };
    const [{ query }] = useQueryParams({
        page: '1',
        pageSize: list.settings.pageSize.toString(),
        sort: list.settings.defaultSortBy ? `${list.settings.defaultSortBy}:${list.settings.defaultSortOrder}` : ''
    });
    const params = React.useMemo(()=>buildValidParams(query), [
        query
    ]);
    const { data, error, isFetching } = useGetAllDocumentsQuery({
        model,
        params
    });
    /**
   * If the API returns an error, display a notification
   */ React.useEffect(()=>{
        if (error) {
            toggleNotification({
                type: 'danger',
                message: formatAPIError(error)
            });
        }
    }, [
        error,
        formatAPIError,
        toggleNotification
    ]);
    const { results = [], pagination } = data ?? {};
    React.useEffect(()=>{
        if (pagination && pagination.pageCount > 0 && pagination.page > pagination.pageCount) {
            navigate({
                search: stringify({
                    ...query,
                    page: pagination.pageCount
                })
            }, {
                replace: true
            });
        }
    }, [
        pagination,
        formatMessage,
        query,
        navigate
    ]);
    const { canCreate } = useDocumentRBAC('ListViewPage', ({ canCreate })=>({
            canCreate
        }));
    const runHookWaterfall = useStrapiApp('ListViewPage', ({ runHookWaterfall })=>runHookWaterfall);
    /**
   * Run the waterfall and then inject our additional table headers.
   */ const tableHeaders = React.useMemo(()=>{
        const headers = runHookWaterfall(INJECT_COLUMN_IN_TABLE, {
            displayedHeaders,
            layout: list
        });
        const formattedHeaders = headers.displayedHeaders.map((header)=>{
            /**
       * When the header label is a string, it is an attribute on the current content-type:
       * Use the attribute name value to compute the translation.
       * Otherwise, it should be a  translation object coming from a plugin that injects into the table (ie i18n, content-releases, review-workflows):
       * Use the translation object as is.
       */ const translation = typeof header.label === 'string' ? {
                id: `content-manager.content-types.${model}.${header.name}`,
                defaultMessage: header.label
            } : header.label;
            return {
                ...header,
                label: formatMessage(translation),
                name: `${header.name}${header.mainField?.name ? `.${header.mainField.name}` : ''}`
            };
        });
        if (schema?.options?.draftAndPublish) {
            formattedHeaders.push({
                attribute: {
                    type: 'custom'
                },
                name: 'status',
                label: formatMessage({
                    id: getTranslation(`containers.list.table-headers.status`),
                    defaultMessage: 'status'
                }),
                searchable: false,
                sortable: false
            });
        }
        return formattedHeaders;
    }, [
        displayedHeaders,
        formatMessage,
        list,
        runHookWaterfall,
        schema?.options?.draftAndPublish,
        model
    ]);
    if (isFetching) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    if (error) {
        return /*#__PURE__*/ jsx(Page.Error, {});
    }
    const contentTypeTitle = schema?.info.displayName ? formatMessage({
        id: schema.info.displayName,
        defaultMessage: schema.info.displayName
    }) : formatMessage({
        id: 'content-manager.containers.untitled',
        defaultMessage: 'Untitled'
    });
    const handleRowClick = (id)=>()=>{
            trackUsage('willEditEntryFromList');
            navigate({
                pathname: id.toString(),
                search: stringify({
                    plugins: query.plugins
                })
            });
        };
    if (!isFetching && results.length === 0) {
        return /*#__PURE__*/ jsxs(Fragment, {
            children: [
                /*#__PURE__*/ jsx(tours.contentManager.Introduction, {
                    children: /*#__PURE__*/ jsx(Box, {})
                }),
                /*#__PURE__*/ jsxs(Page.Main, {
                    children: [
                        /*#__PURE__*/ jsx(Page.Title, {
                            children: `${contentTypeTitle}`
                        }),
                        /*#__PURE__*/ jsx(LayoutsHeaderCustom, {
                            primaryAction: canCreate ? /*#__PURE__*/ jsx(tours.contentManager.CreateNewEntry, {
                                children: /*#__PURE__*/ jsx(CreateButton, {})
                            }) : null,
                            subtitle: formatMessage({
                                id: getTranslation('pages.ListView.header-subtitle'),
                                defaultMessage: '{number, plural, =0 {# entries} one {# entry} other {# entries}} found'
                            }, {
                                number: pagination?.total
                            }),
                            title: contentTypeTitle,
                            navigationAction: /*#__PURE__*/ jsx(BackButton, {})
                        }),
                        /*#__PURE__*/ jsx(Layouts.Action, {
                            endActions: /*#__PURE__*/ jsxs(Fragment, {
                                children: [
                                    /*#__PURE__*/ jsx(InjectionZone, {
                                        area: "listView.actions"
                                    }),
                                    /*#__PURE__*/ jsx(ViewSettingsMenu, {
                                        setHeaders: handleSetHeaders,
                                        resetHeaders: ()=>setDisplayedHeaders(list.layout),
                                        headers: displayedHeaders.map((header)=>header.name)
                                    })
                                ]
                            }),
                            startActions: /*#__PURE__*/ jsxs(Fragment, {
                                children: [
                                    list.settings.searchable && /*#__PURE__*/ jsx(SearchInput, {
                                        label: formatMessage({
                                            id: 'app.component.search.label',
                                            defaultMessage: 'Search for {target}'
                                        }, {
                                            target: contentTypeTitle
                                        }),
                                        placeholder: formatMessage({
                                            id: 'global.search',
                                            defaultMessage: 'Search'
                                        }),
                                        trackedEvent: "didSearch"
                                    }),
                                    list.settings.filterable && schema ? /*#__PURE__*/ jsx(FiltersImpl, {
                                        schema: schema
                                    }) : null
                                ]
                            })
                        }),
                        /*#__PURE__*/ jsx(Layouts.Content, {
                            children: /*#__PURE__*/ jsx(Box, {
                                background: "neutral0",
                                shadow: "filterShadow",
                                hasRadius: true,
                                children: /*#__PURE__*/ jsx(EmptyStateLayout, {
                                    action: canCreate ? /*#__PURE__*/ jsx(CreateButton, {
                                        variant: "secondary"
                                    }) : null,
                                    content: formatMessage({
                                        id: 'app.components.EmptyStateLayout.content-document',
                                        defaultMessage: 'No content found'
                                    }),
                                    hasRadius: true,
                                    icon: /*#__PURE__*/ jsx(EmptyDocuments, {
                                        width: "16rem"
                                    })
                                })
                            })
                        })
                    ]
                })
            ]
        });
    }
    return /*#__PURE__*/ jsxs(Fragment, {
        children: [
            /*#__PURE__*/ jsx(tours.contentManager.Introduction, {
                children: /*#__PURE__*/ jsx(Box, {})
            }),
            /*#__PURE__*/ jsxs(Page.Main, {
                children: [
                    /*#__PURE__*/ jsx(Page.Title, {
                        children: `${contentTypeTitle}`
                    }),
                    /*#__PURE__*/ jsx(LayoutsHeaderCustom, {
                        primaryAction: canCreate ? /*#__PURE__*/ jsx(tours.contentManager.CreateNewEntry, {
                            children: /*#__PURE__*/ jsx(CreateButton, {})
                        }) : null,
                        subtitle: formatMessage({
                            id: getTranslation('pages.ListView.header-subtitle'),
                            defaultMessage: '{number, plural, =0 {# entries} one {# entry} other {# entries}} found'
                        }, {
                            number: pagination?.total
                        }),
                        title: contentTypeTitle,
                        navigationAction: /*#__PURE__*/ jsx(BackButton, {})
                    }),
                    /*#__PURE__*/ jsx(Layouts.Action, {
                        endActions: /*#__PURE__*/ jsxs(Fragment, {
                            children: [
                                /*#__PURE__*/ jsx(InjectionZone, {
                                    area: "listView.actions"
                                }),
                                /*#__PURE__*/ jsx(ViewSettingsMenu, {
                                    setHeaders: handleSetHeaders,
                                    resetHeaders: ()=>setDisplayedHeaders(list.layout),
                                    headers: displayedHeaders.map((header)=>header.name)
                                })
                            ]
                        }),
                        startActions: /*#__PURE__*/ jsxs(Fragment, {
                            children: [
                                list.settings.searchable && /*#__PURE__*/ jsx(SearchInput, {
                                    disabled: results.length === 0,
                                    label: formatMessage({
                                        id: 'app.component.search.label',
                                        defaultMessage: 'Search for {target}'
                                    }, {
                                        target: contentTypeTitle
                                    }),
                                    placeholder: formatMessage({
                                        id: 'global.search',
                                        defaultMessage: 'Search'
                                    }),
                                    trackedEvent: "didSearch"
                                }),
                                list.settings.filterable && schema ? /*#__PURE__*/ jsx(FiltersImpl, {
                                    disabled: results.length === 0,
                                    schema: schema
                                }) : null
                            ]
                        })
                    }),
                    /*#__PURE__*/ jsx(Layouts.Content, {
                        children: /*#__PURE__*/ jsxs(Flex, {
                            gap: 4,
                            direction: "column",
                            alignItems: "stretch",
                            children: [
                                /*#__PURE__*/ jsxs(Table.Root, {
                                    rows: results,
                                    headers: tableHeaders,
                                    isLoading: isFetching,
                                    children: [
                                        /*#__PURE__*/ jsx(TableActionsBar, {}),
                                        /*#__PURE__*/ jsxs(Table.Content, {
                                            children: [
                                                /*#__PURE__*/ jsxs(Table.Head, {
                                                    children: [
                                                        /*#__PURE__*/ jsx(Table.HeaderCheckboxCell, {}),
                                                        tableHeaders.map((header)=>/*#__PURE__*/ jsx(Table.HeaderCell, {
                                                                ...header
                                                            }, header.name))
                                                    ]
                                                }),
                                                /*#__PURE__*/ jsx(Table.Loading, {}),
                                                /*#__PURE__*/ jsx(Table.Empty, {
                                                    action: canCreate ? /*#__PURE__*/ jsx(CreateButton, {
                                                        variant: "secondary"
                                                    }) : null
                                                }),
                                                /*#__PURE__*/ jsx(Table.Body, {
                                                    children: results.map((row)=>{
                                                        return /*#__PURE__*/ jsxs(Table.Row, {
                                                            cursor: "pointer",
                                                            onClick: handleRowClick(row.documentId),
                                                            children: [
                                                                /*#__PURE__*/ jsx(Table.CheckboxCell, {
                                                                    id: row.id
                                                                }),
                                                                tableHeaders.map(({ cellFormatter, ...header })=>{
                                                                    if (header.name === 'status') {
                                                                        const { status } = row;
                                                                        return /*#__PURE__*/ jsx(Table.Cell, {
                                                                            children: /*#__PURE__*/ jsx(DocumentStatus, {
                                                                                status: status,
                                                                                maxWidth: 'min-content'
                                                                            })
                                                                        }, header.name);
                                                                    }
                                                                    if ([
                                                                        'createdBy',
                                                                        'updatedBy'
                                                                    ].includes(header.name.split('.')[0])) {
                                                                        // Display the users full name
                                                                        // Some entries doesn't have a user assigned as creator/updater (ex: entries created through content API)
                                                                        // In this case, we display a dash
                                                                        return /*#__PURE__*/ jsx(Table.Cell, {
                                                                            children: /*#__PURE__*/ jsx(Typography, {
                                                                                textColor: "neutral800",
                                                                                children: row[header.name.split('.')[0]] ? getDisplayName(row[header.name.split('.')[0]]) : '-'
                                                                            })
                                                                        }, header.name);
                                                                    }
                                                                    if (typeof cellFormatter === 'function') {
                                                                        return /*#__PURE__*/ jsx(Table.Cell, {
                                                                            children: cellFormatter(row, header, {
                                                                                collectionType,
                                                                                model
                                                                            })
                                                                        }, header.name);
                                                                    }
                                                                    return /*#__PURE__*/ jsx(Table.Cell, {
                                                                        children: /*#__PURE__*/ jsx(CellContent, {
                                                                            content: row[header.name.split('.')[0]],
                                                                            rowId: row.documentId,
                                                                            ...header
                                                                        })
                                                                    }, header.name);
                                                                }),
                                                                /*#__PURE__*/ jsx(ActionsCell, {
                                                                    onClick: (e)=>e.stopPropagation(),
                                                                    children: /*#__PURE__*/ jsx(TableActions, {
                                                                        document: row
                                                                    })
                                                                })
                                                            ]
                                                        }, row.id);
                                                    })
                                                })
                                            ]
                                        })
                                    ]
                                }),
                                /*#__PURE__*/ jsxs(Pagination.Root, {
                                    ...pagination,
                                    onPageSizeChange: ()=>trackUsage('willChangeNumberOfEntriesPerPage'),
                                    children: [
                                        /*#__PURE__*/ jsx(Pagination.PageSize, {}),
                                        /*#__PURE__*/ jsx(Pagination.Links, {})
                                    ]
                                })
                            ]
                        })
                    })
                ]
            })
        ]
    });
};
const ActionsCell = styled(Table.Cell)`
  display: flex;
  justify-content: flex-end;
`;
/* -------------------------------------------------------------------------------------------------
 * TableActionsBar
 * -----------------------------------------------------------------------------------------------*/ const TableActionsBar = ()=>{
    const selectRow = useTable('TableActionsBar', (state)=>state.selectRow);
    const [{ query }] = useQueryParams();
    const locale = query?.plugins?.i18n?.locale;
    const prevLocale = usePrev(locale);
    // TODO: find a better way to reset the selected rows when the locale changes across all the app
    React.useEffect(()=>{
        if (prevLocale !== locale) {
            selectRow([]);
        }
    }, [
        selectRow,
        prevLocale,
        locale
    ]);
    return /*#__PURE__*/ jsx(Table.ActionBar, {
        children: /*#__PURE__*/ jsx(BulkActionsRenderer, {})
    });
};
const CreateButton = ({ variant })=>{
    const { formatMessage } = useIntl();
    const { trackUsage } = useTracking();
    const [{ query }] = useQueryParams();
    return /*#__PURE__*/ jsx(Button, {
        variant: variant,
        tag: Link,
        onClick: ()=>{
            trackUsage('willCreateEntry', {
                status: 'draft'
            });
        },
        startIcon: /*#__PURE__*/ jsx(Plus, {}),
        style: {
            textDecoration: 'none'
        },
        to: {
            pathname: 'create',
            search: stringify({
                plugins: query.plugins
            })
        },
        minWidth: "max-content",
        marginLeft: 2,
        children: formatMessage({
            id: getTranslation('HeaderLayout.button.label-add-entry'),
            defaultMessage: 'Create new entry'
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * ProtectedListViewPage
 * -----------------------------------------------------------------------------------------------*/ const ProtectedListViewPage = ()=>{
    const { slug = '' } = useParams();
    const { permissions = [], isLoading, error } = useRBAC(PERMISSIONS.map((action)=>({
            action,
            subject: slug
        })));
    if (isLoading) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    if (error || !slug) {
        return /*#__PURE__*/ jsx(Page.Error, {});
    }
    return /*#__PURE__*/ jsx(Page.Protect, {
        permissions: permissions,
        children: ({ permissions })=>/*#__PURE__*/ jsx(DocumentRBAC, {
                permissions: permissions,
                children: /*#__PURE__*/ jsx(ListViewPage, {})
            })
    });
};

export { ListViewPage, ProtectedListViewPage };
//# sourceMappingURL=ListViewPage.mjs.map
