import { jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { createRulesEngine, useForm, useStrapiApp, useField, InputRenderer } from '@strapi/admin/strapi-admin';
import { useIntl } from 'react-intl';
import { SINGLE_TYPES } from '../../../constants/collections.mjs';
import { useDocumentRBAC } from '../../../features/DocumentRBAC.mjs';
import { useDocumentContext } from '../../../hooks/useDocumentContext.mjs';
import { useDocumentLayout } from '../../../hooks/useDocumentLayout.mjs';
import { useLazyComponents } from '../../../hooks/useLazyComponents.mjs';
import { useHasInputPopoverParent } from '../../../preview/components/InputPopover.mjs';
import { usePreviewInputManager } from '../../../preview/hooks/usePreviewInputManager.mjs';
import { BlocksInput as MemoizedBlocksInput } from './FormInputs/BlocksInput/BlocksInput.mjs';
import { ComponentInput as MemoizedComponentInput } from './FormInputs/Component/Input.mjs';
import { useDynamicZone, DynamicZone } from './FormInputs/DynamicZone/Field.mjs';
import { NotAllowedInput } from './FormInputs/NotAllowed.mjs';
import { RelationsInput as MemoizedRelationsField } from './FormInputs/Relations/Relations.mjs';
import { UIDInput as MemoizedUIDInput } from './FormInputs/UID.mjs';
import { Wysiwyg as MemoizedWysiwyg } from './FormInputs/Wysiwyg/Field.mjs';

/**
 * @internal
 *
 * @description An abstraction around the regular form input renderer designed
 * specifically to be used in the EditView of the content-manager this understands
 * the complete EditFieldLayout and will handle RBAC conditions and rendering CM specific
 * components such as Blocks / Relations.
 */ const BaseInputRenderer = ({ visible, hint: providedHint, document, ...inputProps })=>{
    const localeKey = document?.document?.locale || 'default';
    const { currentDocumentMeta } = useDocumentContext('DynamicComponent');
    const { edit: { components } } = useDocumentLayout(currentDocumentMeta.model);
    const collectionType = document.schema?.kind === 'collectionType' ? 'collection-types' : 'single-types';
    const isInDynamicZone = useDynamicZone('isInDynamicZone', (state)=>state.isInDynamicZone);
    const isInPreviewPopover = useHasInputPopoverParent();
    const shouldIgnorePermissions = isInDynamicZone || isInPreviewPopover;
    const isFormDisabled = useForm('InputRenderer', (state)=>state.disabled);
    const canCreateFields = useDocumentRBAC('InputRenderer', (rbac)=>rbac.canCreateFields);
    const canReadFields = useDocumentRBAC('InputRenderer', (rbac)=>rbac.canReadFields);
    const canUpdateFields = useDocumentRBAC('InputRenderer', (rbac)=>rbac.canUpdateFields);
    const canUserAction = useDocumentRBAC('InputRenderer', (rbac)=>rbac.canUserAction);
    let idToCheck = document.document?.documentId;
    if (collectionType === SINGLE_TYPES) {
        idToCheck = document?.document?.documentId;
    }
    const editableFields = idToCheck ? canUpdateFields : canCreateFields;
    const readableFields = idToCheck ? canReadFields : canCreateFields;
    // Everything preview related
    const previewProps = usePreviewInputManager(inputProps.name, inputProps.attribute);
    const props = {
        ...inputProps,
        ...previewProps
    };
    /**
   * Component fields are always readable and editable,
   * however the fields within them may not be.
   */ const canUserReadField = canUserAction(props.name, readableFields, props.type);
    const canUserEditField = canUserAction(props.name, editableFields, props.type);
    const fields = useStrapiApp('InputRenderer', (app)=>app.fields);
    const { lazyComponentStore } = useLazyComponents(attributeHasCustomFieldProperty(props.attribute) ? [
        props.attribute.customField
    ] : undefined);
    const hint = useFieldHint(providedHint, props.attribute);
    // We pass field in case of Custom Fields to keep backward compatibility
    const field = useField(props.name);
    if (!visible) {
        return null;
    }
    /**
   * If the user can't read the field then we don't want to ever render it.
   */ if (!canUserReadField && !shouldIgnorePermissions) {
        return /*#__PURE__*/ jsx(NotAllowedInput, {
            hint: hint,
            ...props
        });
    }
    const fieldIsDisabled = !canUserEditField && !shouldIgnorePermissions || props.disabled || isFormDisabled;
    /**
   * Because a custom field has a unique prop but the type could be confused with either
   * the useField hook or the type of the field we need to handle it separately and first.
   */ if (attributeHasCustomFieldProperty(props.attribute)) {
        const CustomInput = lazyComponentStore[props.attribute.customField];
        if (CustomInput) {
            return /*#__PURE__*/ jsx(CustomInput, {
                ...props,
                ...field,
                // @ts-expect-error – TODO: fix this type error in the useLazyComponents hook.
                hint: hint,
                disabled: fieldIsDisabled
            });
        }
        return /*#__PURE__*/ jsx(InputRenderer, {
            ...props,
            ...previewProps,
            hint: hint,
            // @ts-expect-error – this workaround lets us display that the custom field is missing.
            type: props.attribute.customField,
            disabled: fieldIsDisabled
        }, `input-${props.name}-${localeKey}`);
    }
    /**
   * This is where we handle ONLY the fields from the `useLibrary` hook.
   */ const addedInputTypes = Object.keys(fields);
    if (!attributeHasCustomFieldProperty(props.attribute) && addedInputTypes.includes(props.type)) {
        const CustomInput = fields[props.type];
        return /*#__PURE__*/ jsx(CustomInput, {
            ...props,
            // @ts-expect-error – TODO: fix this type error in the useLazyComponents hook.
            hint: hint,
            disabled: fieldIsDisabled
        }, `input-${props.name}-${localeKey}`);
    }
    /**
   * These include the content-manager specific fields, failing that we fall back
   * to the more generic form input renderer.
   */ switch(props.type){
        case 'blocks':
            return /*#__PURE__*/ jsx(MemoizedBlocksInput, {
                ...props,
                hint: hint,
                type: props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'component':
            return /*#__PURE__*/ jsx(MemoizedComponentInput, {
                ...props,
                hint: hint,
                layout: components[props.attribute.component].layout,
                disabled: fieldIsDisabled,
                children: (componentInputProps)=>/*#__PURE__*/ jsx(BaseInputRenderer, {
                        ...componentInputProps
                    }, `input-${componentInputProps.name}-${localeKey}`)
            }, `input-${props.name}-${localeKey}`);
        case 'dynamiczone':
            return /*#__PURE__*/ jsx(DynamicZone, {
                ...props,
                hint: hint,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'relation':
            return /*#__PURE__*/ jsx(MemoizedRelationsField, {
                ...props,
                hint: hint,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'richtext':
            return /*#__PURE__*/ jsx(MemoizedWysiwyg, {
                ...props,
                hint: hint,
                type: props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'uid':
            // These props are not needed for the generic form input renderer.
            const { unique: _uniqueUID, ...restUIDProps } = props;
            return /*#__PURE__*/ jsx(MemoizedUIDInput, {
                ...restUIDProps,
                hint: hint,
                type: props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        /**
     * Enumerations are a special case because they require options.
     */ case 'enumeration':
            return /*#__PURE__*/ jsx(InputRenderer, {
                ...props,
                ...previewProps,
                hint: hint,
                options: props.attribute.enum.map((value)=>({
                        value
                    })),
                // @ts-expect-error – Temp workaround so we don't forget custom-fields don't work!
                type: props.customField ? 'custom-field' : props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        default:
            // These props are not needed for the generic form input renderer.
            const { unique: _unique, mainField: _mainField, ...restProps } = props;
            return /*#__PURE__*/ jsx(InputRenderer, {
                ...restProps,
                ...previewProps,
                hint: hint,
                // @ts-expect-error – Temp workaround so we don't forget custom-fields don't work!
                type: props.customField ? 'custom-field' : props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
    }
};
const rulesEngine = createRulesEngine();
/**
 * A wrapper around BaseInputRenderer that conditionally renders it depending on the attribute's condition.
 */ const ConditionAwareInputRenderer = ({ condition, ...props })=>{
    // Note: this selector causes a re-render every time any form value on the page changes
    const fieldValues = useForm('ConditionalInputRenderer', (state)=>state.values);
    const isVisible = rulesEngine.evaluate(condition, fieldValues);
    if (!isVisible) {
        return null;
    }
    return /*#__PURE__*/ jsx(BaseInputRenderer, {
        ...props
    });
};
const attributeHasCustomFieldProperty = (attribute)=>'customField' in attribute && typeof attribute.customField === 'string';
const useFieldHint = (hint = undefined, attribute)=>{
    const { formatMessage } = useIntl();
    const { maximum, minimum } = getMinMax(attribute);
    if (!maximum && !minimum) {
        return hint;
    }
    const units = [
        'string',
        'uid',
        'richtext',
        'email',
        'password',
        'text'
    ].includes(attribute.type) ? formatMessage({
        id: 'content-manager.form.Input.hint.character.unit',
        defaultMessage: '{maxValue, plural, one { character} other { characters}}'
    }, {
        maxValue: Math.max(minimum || 0, maximum || 0)
    }) : null;
    const hasMinAndMax = typeof minimum === 'number' && typeof maximum === 'number';
    return formatMessage({
        id: 'content-manager.form.Input.hint.text',
        defaultMessage: '{min, select, undefined {} other {min. {min}}}{divider}{max, select, undefined {} other {max. {max}}}{unit}{br}{description}'
    }, {
        min: minimum,
        max: maximum,
        description: hint,
        unit: units,
        divider: hasMinAndMax ? formatMessage({
            id: 'content-manager.form.Input.hint.minMaxDivider',
            defaultMessage: ' / '
        }) : null,
        br: /*#__PURE__*/ jsx("br", {})
    });
};
const getMinMax = (attribute)=>{
    if ('min' in attribute || 'max' in attribute) {
        return {
            maximum: !Number.isNaN(Number(attribute.max)) ? Number(attribute.max) : undefined,
            minimum: !Number.isNaN(Number(attribute.min)) ? Number(attribute.min) : undefined
        };
    } else if ('maxLength' in attribute || 'minLength' in attribute) {
        return {
            maximum: attribute.maxLength,
            minimum: attribute.minLength
        };
    } else {
        return {
            maximum: undefined,
            minimum: undefined
        };
    }
};
/**
 * Conditionally routes the exported InputRender component towards ConditionalInputRenderer
 * (when there's a JSON logic condition on the attribute, or BaseInputRenderer otherwise.
 * We do this because rendering a conditional field requires access to the values of
 * other form fields, which causes many re-renders and performance issues on complex content
 * types. By splitting the component into two, we isolate the performance issue to
 * conditional fields only, not all edit view fields.
 */ const MemoizedInputRenderer = /*#__PURE__*/ React.memo((props)=>{
    const condition = props.attribute.conditions?.visible;
    if (condition) {
        return /*#__PURE__*/ jsx(ConditionAwareInputRenderer, {
            ...props,
            condition: condition
        });
    }
    return /*#__PURE__*/ jsx(BaseInputRenderer, {
        ...props
    });
});

export { MemoizedInputRenderer as InputRenderer, useFieldHint };
//# sourceMappingURL=InputRenderer.mjs.map
