'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var reactIntl = require('react-intl');
var collections = require('../../../constants/collections.js');
var DocumentRBAC = require('../../../features/DocumentRBAC.js');
var useDocumentContext = require('../../../hooks/useDocumentContext.js');
var useDocumentLayout = require('../../../hooks/useDocumentLayout.js');
var useLazyComponents = require('../../../hooks/useLazyComponents.js');
var InputPopover = require('../../../preview/components/InputPopover.js');
var usePreviewInputManager = require('../../../preview/hooks/usePreviewInputManager.js');
var BlocksInput = require('./FormInputs/BlocksInput/BlocksInput.js');
var Input = require('./FormInputs/Component/Input.js');
var Field = require('./FormInputs/DynamicZone/Field.js');
var NotAllowed = require('./FormInputs/NotAllowed.js');
var Relations = require('./FormInputs/Relations/Relations.js');
var UID = require('./FormInputs/UID.js');
var Field$1 = require('./FormInputs/Wysiwyg/Field.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

/**
 * @internal
 *
 * @description An abstraction around the regular form input renderer designed
 * specifically to be used in the EditView of the content-manager this understands
 * the complete EditFieldLayout and will handle RBAC conditions and rendering CM specific
 * components such as Blocks / Relations.
 */ const BaseInputRenderer = ({ visible, hint: providedHint, document, ...inputProps })=>{
    const localeKey = document?.document?.locale || 'default';
    const { currentDocumentMeta } = useDocumentContext.useDocumentContext('DynamicComponent');
    const { edit: { components } } = useDocumentLayout.useDocumentLayout(currentDocumentMeta.model);
    const collectionType = document.schema?.kind === 'collectionType' ? 'collection-types' : 'single-types';
    const isInDynamicZone = Field.useDynamicZone('isInDynamicZone', (state)=>state.isInDynamicZone);
    const isInPreviewPopover = InputPopover.useHasInputPopoverParent();
    const shouldIgnorePermissions = isInDynamicZone || isInPreviewPopover;
    const isFormDisabled = strapiAdmin.useForm('InputRenderer', (state)=>state.disabled);
    const canCreateFields = DocumentRBAC.useDocumentRBAC('InputRenderer', (rbac)=>rbac.canCreateFields);
    const canReadFields = DocumentRBAC.useDocumentRBAC('InputRenderer', (rbac)=>rbac.canReadFields);
    const canUpdateFields = DocumentRBAC.useDocumentRBAC('InputRenderer', (rbac)=>rbac.canUpdateFields);
    const canUserAction = DocumentRBAC.useDocumentRBAC('InputRenderer', (rbac)=>rbac.canUserAction);
    let idToCheck = document.document?.documentId;
    if (collectionType === collections.SINGLE_TYPES) {
        idToCheck = document?.document?.documentId;
    }
    const editableFields = idToCheck ? canUpdateFields : canCreateFields;
    const readableFields = idToCheck ? canReadFields : canCreateFields;
    // Everything preview related
    const previewProps = usePreviewInputManager.usePreviewInputManager(inputProps.name, inputProps.attribute);
    const props = {
        ...inputProps,
        ...previewProps
    };
    /**
   * Component fields are always readable and editable,
   * however the fields within them may not be.
   */ const canUserReadField = canUserAction(props.name, readableFields, props.type);
    const canUserEditField = canUserAction(props.name, editableFields, props.type);
    const fields = strapiAdmin.useStrapiApp('InputRenderer', (app)=>app.fields);
    const { lazyComponentStore } = useLazyComponents.useLazyComponents(attributeHasCustomFieldProperty(props.attribute) ? [
        props.attribute.customField
    ] : undefined);
    const hint = useFieldHint(providedHint, props.attribute);
    // We pass field in case of Custom Fields to keep backward compatibility
    const field = strapiAdmin.useField(props.name);
    if (!visible) {
        return null;
    }
    /**
   * If the user can't read the field then we don't want to ever render it.
   */ if (!canUserReadField && !shouldIgnorePermissions) {
        return /*#__PURE__*/ jsxRuntime.jsx(NotAllowed.NotAllowedInput, {
            hint: hint,
            ...props
        });
    }
    const fieldIsDisabled = !canUserEditField && !shouldIgnorePermissions || props.disabled || isFormDisabled;
    /**
   * Because a custom field has a unique prop but the type could be confused with either
   * the useField hook or the type of the field we need to handle it separately and first.
   */ if (attributeHasCustomFieldProperty(props.attribute)) {
        const CustomInput = lazyComponentStore[props.attribute.customField];
        if (CustomInput) {
            return /*#__PURE__*/ jsxRuntime.jsx(CustomInput, {
                ...props,
                ...field,
                // @ts-expect-error – TODO: fix this type error in the useLazyComponents hook.
                hint: hint,
                disabled: fieldIsDisabled
            });
        }
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.InputRenderer, {
            ...props,
            ...previewProps,
            hint: hint,
            // @ts-expect-error – this workaround lets us display that the custom field is missing.
            type: props.attribute.customField,
            disabled: fieldIsDisabled
        }, `input-${props.name}-${localeKey}`);
    }
    /**
   * This is where we handle ONLY the fields from the `useLibrary` hook.
   */ const addedInputTypes = Object.keys(fields);
    if (!attributeHasCustomFieldProperty(props.attribute) && addedInputTypes.includes(props.type)) {
        const CustomInput = fields[props.type];
        return /*#__PURE__*/ jsxRuntime.jsx(CustomInput, {
            ...props,
            // @ts-expect-error – TODO: fix this type error in the useLazyComponents hook.
            hint: hint,
            disabled: fieldIsDisabled
        }, `input-${props.name}-${localeKey}`);
    }
    /**
   * These include the content-manager specific fields, failing that we fall back
   * to the more generic form input renderer.
   */ switch(props.type){
        case 'blocks':
            return /*#__PURE__*/ jsxRuntime.jsx(BlocksInput.BlocksInput, {
                ...props,
                hint: hint,
                type: props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'component':
            return /*#__PURE__*/ jsxRuntime.jsx(Input.ComponentInput, {
                ...props,
                hint: hint,
                layout: components[props.attribute.component].layout,
                disabled: fieldIsDisabled,
                children: (componentInputProps)=>/*#__PURE__*/ jsxRuntime.jsx(BaseInputRenderer, {
                        ...componentInputProps
                    }, `input-${componentInputProps.name}-${localeKey}`)
            }, `input-${props.name}-${localeKey}`);
        case 'dynamiczone':
            return /*#__PURE__*/ jsxRuntime.jsx(Field.DynamicZone, {
                ...props,
                hint: hint,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'relation':
            return /*#__PURE__*/ jsxRuntime.jsx(Relations.RelationsInput, {
                ...props,
                hint: hint,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'richtext':
            return /*#__PURE__*/ jsxRuntime.jsx(Field$1.Wysiwyg, {
                ...props,
                hint: hint,
                type: props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        case 'uid':
            // These props are not needed for the generic form input renderer.
            const { unique: _uniqueUID, ...restUIDProps } = props;
            return /*#__PURE__*/ jsxRuntime.jsx(UID.UIDInput, {
                ...restUIDProps,
                hint: hint,
                type: props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        /**
     * Enumerations are a special case because they require options.
     */ case 'enumeration':
            return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.InputRenderer, {
                ...props,
                ...previewProps,
                hint: hint,
                options: props.attribute.enum.map((value)=>({
                        value
                    })),
                // @ts-expect-error – Temp workaround so we don't forget custom-fields don't work!
                type: props.customField ? 'custom-field' : props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
        default:
            // These props are not needed for the generic form input renderer.
            const { unique: _unique, mainField: _mainField, ...restProps } = props;
            return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.InputRenderer, {
                ...restProps,
                ...previewProps,
                hint: hint,
                // @ts-expect-error – Temp workaround so we don't forget custom-fields don't work!
                type: props.customField ? 'custom-field' : props.type,
                disabled: fieldIsDisabled
            }, `input-${props.name}-${localeKey}`);
    }
};
const rulesEngine = strapiAdmin.createRulesEngine();
/**
 * A wrapper around BaseInputRenderer that conditionally renders it depending on the attribute's condition.
 */ const ConditionAwareInputRenderer = ({ condition, ...props })=>{
    // Note: this selector causes a re-render every time any form value on the page changes
    const fieldValues = strapiAdmin.useForm('ConditionalInputRenderer', (state)=>state.values);
    const isVisible = rulesEngine.evaluate(condition, fieldValues);
    if (!isVisible) {
        return null;
    }
    return /*#__PURE__*/ jsxRuntime.jsx(BaseInputRenderer, {
        ...props
    });
};
const attributeHasCustomFieldProperty = (attribute)=>'customField' in attribute && typeof attribute.customField === 'string';
const useFieldHint = (hint = undefined, attribute)=>{
    const { formatMessage } = reactIntl.useIntl();
    const { maximum, minimum } = getMinMax(attribute);
    if (!maximum && !minimum) {
        return hint;
    }
    const units = [
        'string',
        'uid',
        'richtext',
        'email',
        'password',
        'text'
    ].includes(attribute.type) ? formatMessage({
        id: 'content-manager.form.Input.hint.character.unit',
        defaultMessage: '{maxValue, plural, one { character} other { characters}}'
    }, {
        maxValue: Math.max(minimum || 0, maximum || 0)
    }) : null;
    const hasMinAndMax = typeof minimum === 'number' && typeof maximum === 'number';
    return formatMessage({
        id: 'content-manager.form.Input.hint.text',
        defaultMessage: '{min, select, undefined {} other {min. {min}}}{divider}{max, select, undefined {} other {max. {max}}}{unit}{br}{description}'
    }, {
        min: minimum,
        max: maximum,
        description: hint,
        unit: units,
        divider: hasMinAndMax ? formatMessage({
            id: 'content-manager.form.Input.hint.minMaxDivider',
            defaultMessage: ' / '
        }) : null,
        br: /*#__PURE__*/ jsxRuntime.jsx("br", {})
    });
};
const getMinMax = (attribute)=>{
    if ('min' in attribute || 'max' in attribute) {
        return {
            maximum: !Number.isNaN(Number(attribute.max)) ? Number(attribute.max) : undefined,
            minimum: !Number.isNaN(Number(attribute.min)) ? Number(attribute.min) : undefined
        };
    } else if ('maxLength' in attribute || 'minLength' in attribute) {
        return {
            maximum: attribute.maxLength,
            minimum: attribute.minLength
        };
    } else {
        return {
            maximum: undefined,
            minimum: undefined
        };
    }
};
/**
 * Conditionally routes the exported InputRender component towards ConditionalInputRenderer
 * (when there's a JSON logic condition on the attribute, or BaseInputRenderer otherwise.
 * We do this because rendering a conditional field requires access to the values of
 * other form fields, which causes many re-renders and performance issues on complex content
 * types. By splitting the component into two, we isolate the performance issue to
 * conditional fields only, not all edit view fields.
 */ const MemoizedInputRenderer = /*#__PURE__*/ React__namespace.memo((props)=>{
    const condition = props.attribute.conditions?.visible;
    if (condition) {
        return /*#__PURE__*/ jsxRuntime.jsx(ConditionAwareInputRenderer, {
            ...props,
            condition: condition
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsx(BaseInputRenderer, {
        ...props
    });
});

exports.InputRenderer = MemoizedInputRenderer;
exports.useFieldHint = useFieldHint;
//# sourceMappingURL=InputRenderer.js.map
