import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { useIsDesktop, useNotification, useForm, BackButton, useQueryParams, useStrapiApp, DescriptionComponentRenderer, useDebounce } from '@strapi/admin/strapi-admin';
import { Flex, Typography, Box, SingleSelect, SingleSelectOption, IconButton, Popover, Dialog } from '@strapi/design-system';
import { ListPlus, Pencil, Trash, WarningCircle } from '@strapi/icons';
import { useIntl } from 'react-intl';
import { useNavigate, useMatch, useParams } from 'react-router-dom';
import { RelativeTime } from '../../../components/RelativeTime.mjs';
import { CREATED_BY_ATTRIBUTE_NAME, UPDATED_BY_ATTRIBUTE_NAME, PUBLISHED_AT_ATTRIBUTE_NAME, PUBLISHED_BY_ATTRIBUTE_NAME, UPDATED_AT_ATTRIBUTE_NAME, CREATED_AT_ATTRIBUTE_NAME } from '../../../constants/attributes.mjs';
import { SINGLE_TYPES, COLLECTION_TYPES } from '../../../constants/collections.mjs';
import { useDocumentRBAC } from '../../../features/DocumentRBAC.mjs';
import { useDoc } from '../../../hooks/useDocument.mjs';
import { useDocumentActions } from '../../../hooks/useDocumentActions.mjs';
import { LIST_PATH, CLONE_PATH } from '../../../router.mjs';
import { getDisplayName } from '../../../utils/users.mjs';
import { DocumentActionsMenu } from './DocumentActions.mjs';
import { DocumentStatus } from './DocumentStatus.mjs';

const Header = ({ isCreating, status, title: documentTitle = 'Untitled' })=>{
    const { formatMessage } = useIntl();
    const isCloning = useMatch(CLONE_PATH) !== null;
    const params = useParams();
    const title = isCreating ? formatMessage({
        id: 'content-manager.containers.edit.title.new',
        defaultMessage: 'Create an entry'
    }) : documentTitle;
    return /*#__PURE__*/ jsxs(Flex, {
        direction: "column",
        alignItems: "flex-start",
        paddingTop: {
            initial: 4,
            large: 6
        },
        paddingBottom: {
            initial: 0,
            large: 4
        },
        gap: 2,
        children: [
            /*#__PURE__*/ jsx(BackButton, {
                fallback: params.collectionType === SINGLE_TYPES ? undefined : `../${COLLECTION_TYPES}/${params.slug}`
            }),
            /*#__PURE__*/ jsxs(Flex, {
                width: "100%",
                justifyContent: "space-between",
                gap: {
                    initial: 2,
                    medium: '8rem'
                },
                alignItems: "flex-start",
                direction: {
                    initial: 'column',
                    medium: 'row'
                },
                children: [
                    /*#__PURE__*/ jsx(Typography, {
                        variant: "alpha",
                        tag: "h1",
                        children: title
                    }),
                    /*#__PURE__*/ jsx(HeaderToolbar, {})
                ]
            }),
            status ? /*#__PURE__*/ jsx(Box, {
                marginTop: 1,
                children: /*#__PURE__*/ jsx(DocumentStatus, {
                    status: isCloning ? 'draft' : status
                })
            }) : null
        ]
    });
};
/**
 * @description Contains the document actions that have `position: header`, if there are
 * none we still render the menu because we render the information about the document there.
 */ const HeaderToolbar = ()=>{
    const { formatMessage } = useIntl();
    const isCloning = useMatch(CLONE_PATH) !== null;
    const [{ query: { status = 'draft' } }] = useQueryParams();
    const { model, id, document, meta, collectionType } = useDoc();
    const plugins = useStrapiApp('HeaderToolbar', (state)=>state.plugins);
    return /*#__PURE__*/ jsxs(Flex, {
        gap: 2,
        children: [
            /*#__PURE__*/ jsx(DescriptionComponentRenderer, {
                props: {
                    activeTab: status,
                    model,
                    documentId: id,
                    document: isCloning ? undefined : document,
                    meta: isCloning ? undefined : meta,
                    collectionType
                },
                descriptions: plugins['content-manager'].apis.getHeaderActions(),
                children: (actions)=>{
                    if (actions.length > 0) {
                        return /*#__PURE__*/ jsx(HeaderActions, {
                            actions: actions
                        });
                    } else {
                        return null;
                    }
                }
            }),
            /*#__PURE__*/ jsx(DescriptionComponentRenderer, {
                props: {
                    activeTab: status,
                    model,
                    documentId: id,
                    document: isCloning ? undefined : document,
                    meta: isCloning ? undefined : meta,
                    collectionType
                },
                descriptions: plugins['content-manager'].apis.getDocumentActions('header'),
                children: (actions)=>{
                    const headerActions = actions.filter((action)=>{
                        const positions = Array.isArray(action.position) ? action.position : [
                            action.position
                        ];
                        return positions.includes('header');
                    });
                    return /*#__PURE__*/ jsx(DocumentActionsMenu, {
                        actions: headerActions,
                        label: formatMessage({
                            id: 'content-manager.containers.edit.header.more-actions',
                            defaultMessage: 'More actions'
                        }),
                        children: /*#__PURE__*/ jsx(Information, {
                            activeTab: status
                        })
                    });
                }
            })
        ]
    });
};
const Information = ({ activeTab })=>{
    const { formatMessage } = useIntl();
    const { document, meta } = useDoc();
    if (!document || !document.id) {
        return null;
    }
    /**
   * Because in the backend separate entries are made for draft and published
   * documents, the creator fields are different for each of them. For example,
   * you could make your draft in January and then publish it for the first time
   * in Feb. This would make the createdAt value for the published entry in Feb
   * but really we want to show the document as a whole. The draft entry will also
   * never have the publishedAt values.
   *
   * So, we decipher which document to show the creator for based on the activeTab.
   */ const createAndUpdateDocument = activeTab === 'draft' ? document : meta?.availableStatus.find((status)=>status.publishedAt === null);
    const publishDocument = activeTab === 'published' ? document : meta?.availableStatus.find((status)=>status.publishedAt !== null);
    const creator = createAndUpdateDocument?.[CREATED_BY_ATTRIBUTE_NAME] ? getDisplayName(createAndUpdateDocument[CREATED_BY_ATTRIBUTE_NAME]) : null;
    const updator = createAndUpdateDocument?.[UPDATED_BY_ATTRIBUTE_NAME] ? getDisplayName(createAndUpdateDocument[UPDATED_BY_ATTRIBUTE_NAME]) : null;
    const information = [
        {
            isDisplayed: !!publishDocument?.[PUBLISHED_AT_ATTRIBUTE_NAME],
            label: formatMessage({
                id: 'content-manager.containers.edit.information.last-published.label',
                defaultMessage: 'Published'
            }),
            value: formatMessage({
                id: 'content-manager.containers.edit.information.last-published.value',
                defaultMessage: `{time}{isAnonymous, select, true {} other { by {author}}}`
            }, {
                time: /*#__PURE__*/ jsx(RelativeTime, {
                    timestamp: new Date(publishDocument?.[PUBLISHED_AT_ATTRIBUTE_NAME])
                }),
                isAnonymous: !publishDocument?.[PUBLISHED_BY_ATTRIBUTE_NAME],
                author: publishDocument?.[PUBLISHED_BY_ATTRIBUTE_NAME] ? getDisplayName(publishDocument?.[PUBLISHED_BY_ATTRIBUTE_NAME]) : null
            })
        },
        {
            isDisplayed: !!createAndUpdateDocument?.[UPDATED_AT_ATTRIBUTE_NAME],
            label: formatMessage({
                id: 'content-manager.containers.edit.information.last-draft.label',
                defaultMessage: 'Updated'
            }),
            value: formatMessage({
                id: 'content-manager.containers.edit.information.last-draft.value',
                defaultMessage: `{time}{isAnonymous, select, true {} other { by {author}}}`
            }, {
                time: /*#__PURE__*/ jsx(RelativeTime, {
                    timestamp: new Date(createAndUpdateDocument?.[UPDATED_AT_ATTRIBUTE_NAME])
                }),
                isAnonymous: !updator,
                author: updator
            })
        },
        {
            isDisplayed: !!createAndUpdateDocument?.[CREATED_AT_ATTRIBUTE_NAME],
            label: formatMessage({
                id: 'content-manager.containers.edit.information.document.label',
                defaultMessage: 'Created'
            }),
            value: formatMessage({
                id: 'content-manager.containers.edit.information.document.value',
                defaultMessage: `{time}{isAnonymous, select, true {} other { by {author}}}`
            }, {
                time: /*#__PURE__*/ jsx(RelativeTime, {
                    timestamp: new Date(createAndUpdateDocument?.[CREATED_AT_ATTRIBUTE_NAME])
                }),
                isAnonymous: !creator,
                author: creator
            })
        }
    ].filter((info)=>info.isDisplayed);
    return /*#__PURE__*/ jsx(Flex, {
        borderWidth: "1px 0 0 0",
        borderStyle: "solid",
        borderColor: "neutral150",
        direction: "column",
        marginTop: 2,
        tag: "dl",
        padding: 5,
        gap: 3,
        alignItems: "flex-start",
        /**
       * The menu content has a padding of 4px, but we want our divider (the border top applied) to
       * be flush with the menu content. So we need to adjust the margin & width to account for the padding.
       */ marginLeft: "-0.4rem",
        marginRight: "-0.4rem",
        width: "calc(100% + 8px)",
        children: information.map((info)=>/*#__PURE__*/ jsxs(Flex, {
                gap: 1,
                direction: "column",
                alignItems: "flex-start",
                children: [
                    /*#__PURE__*/ jsx(Typography, {
                        tag: "dt",
                        variant: "pi",
                        fontWeight: "bold",
                        children: info.label
                    }),
                    /*#__PURE__*/ jsx(Typography, {
                        tag: "dd",
                        variant: "pi",
                        textColor: "neutral600",
                        children: info.value
                    })
                ]
            }, info.label))
    });
};
const HeaderActions = ({ actions })=>{
    const [dialogId, setDialogId] = React.useState(null);
    const handleClick = (action)=>async (e)=>{
            if (!('options' in action)) {
                const { onClick = ()=>false, dialog, id } = action;
                const muteDialog = await onClick(e);
                if (dialog && !muteDialog) {
                    e.preventDefault();
                    setDialogId(id);
                }
            }
        };
    const handleClose = ()=>{
        setDialogId(null);
    };
    return /*#__PURE__*/ jsx(Flex, {
        gap: 1,
        children: actions.map((action)=>{
            if (action.options) {
                return /*#__PURE__*/ jsx(SingleSelect, {
                    size: "S",
                    // @ts-expect-error – the DS will handle numbers, but we're not allowing the API.
                    onChange: action.onSelect,
                    "aria-label": action.label,
                    ...action,
                    children: action.options.map(({ label, ...option })=>{
                        if (option._render) {
                            return option._render();
                        }
                        return /*#__PURE__*/ jsx(SingleSelectOption, {
                            ...option,
                            children: label
                        }, option.value);
                    })
                }, action.id);
            } else if (action._status) {
                return /*#__PURE__*/ jsx(HeaderActionStatus, {
                    tooltip: action._status?.tooltip,
                    children: action._status.message
                }, action.id);
            } else {
                return /*#__PURE__*/ jsxs(React.Fragment, {
                    children: [
                        /*#__PURE__*/ jsx(IconButton, {
                            disabled: action.disabled,
                            label: action.label,
                            size: "S",
                            onClick: handleClick(action),
                            children: action.icon
                        }),
                        action.dialog ? /*#__PURE__*/ jsx(HeaderActionDialog, {
                            ...action.dialog,
                            isOpen: dialogId === action.id,
                            onClose: handleClose
                        }) : null
                    ]
                }, action.id);
            }
        })
    });
};
const HeaderActionStatus = ({ tooltip, children })=>{
    const [open, setOpen] = React.useState(false);
    // Debounce the open/close so the user can hover over the popover content before it closes
    const debouncedOpen = useDebounce(open, 100);
    const handleMouseEnter = ()=>{
        if (tooltip) {
            setOpen(true);
        }
    };
    const handleMouseLeave = ()=>{
        if (tooltip) {
            setOpen(false);
        }
    };
    return /*#__PURE__*/ jsxs(Popover.Root, {
        open: debouncedOpen,
        onOpenChange: setOpen,
        children: [
            /*#__PURE__*/ jsx(Popover.Anchor, {
                style: {
                    alignSelf: 'stretch'
                },
                onMouseEnter: handleMouseEnter,
                onMouseLeave: handleMouseLeave,
                "aria-describedby": "document-header-action-status",
                children: /*#__PURE__*/ jsx(Box, {
                    height: "100%",
                    children: children
                })
            }),
            /*#__PURE__*/ jsx(Popover.Content, {
                role: "tooltip",
                id: "document-header-action-status",
                side: "bottom",
                align: "center",
                onMouseEnter: handleMouseEnter,
                onMouseLeave: handleMouseLeave,
                children: tooltip
            })
        ]
    });
};
const HeaderActionDialog = ({ onClose, onCancel, title, content: Content, isOpen })=>{
    const handleClose = async ()=>{
        if (onCancel) {
            await onCancel();
        }
        onClose();
    };
    return /*#__PURE__*/ jsx(Dialog.Root, {
        open: isOpen,
        onOpenChange: handleClose,
        children: /*#__PURE__*/ jsxs(Dialog.Content, {
            children: [
                /*#__PURE__*/ jsx(Dialog.Header, {
                    children: title
                }),
                typeof Content === 'function' ? /*#__PURE__*/ jsx(Content, {
                    onClose: handleClose
                }) : Content
            ]
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * DocumentActionComponents
 * -----------------------------------------------------------------------------------------------*/ const ConfigureTheViewAction = ({ collectionType, model })=>{
    const navigate = useNavigate();
    const { formatMessage } = useIntl();
    const isDesktop = useIsDesktop();
    return isDesktop ? {
        label: formatMessage({
            id: 'app.links.configure-view',
            defaultMessage: 'Configure the view'
        }),
        icon: /*#__PURE__*/ jsx(ListPlus, {}),
        onClick: ()=>{
            navigate(`../${collectionType}/${model}/configurations/edit`);
        },
        position: 'header'
    } : null;
};
ConfigureTheViewAction.type = 'configure-the-view';
ConfigureTheViewAction.position = 'header';
const EditTheModelAction = ({ model })=>{
    const navigate = useNavigate();
    const { formatMessage } = useIntl();
    const isDesktop = useIsDesktop();
    return isDesktop ? {
        label: formatMessage({
            id: 'content-manager.link-to-ctb',
            defaultMessage: 'Edit the model'
        }),
        icon: /*#__PURE__*/ jsx(Pencil, {}),
        onClick: ()=>{
            navigate(`/plugins/content-type-builder/content-types/${model}`);
        },
        position: 'header'
    } : null;
};
EditTheModelAction.type = 'edit-the-model';
EditTheModelAction.position = 'header';
const DeleteAction = ({ documentId, model, collectionType, document })=>{
    const navigate = useNavigate();
    const { formatMessage } = useIntl();
    const listViewPathMatch = useMatch(LIST_PATH);
    const canDelete = useDocumentRBAC('DeleteAction', (state)=>state.canDelete);
    const { delete: deleteAction, isLoading } = useDocumentActions();
    const { toggleNotification } = useNotification();
    const setSubmitting = useForm('DeleteAction', (state)=>state.setSubmitting);
    const isLocalized = document?.locale != null;
    return {
        disabled: !canDelete || !document,
        label: formatMessage({
            id: 'content-manager.actions.delete.label',
            defaultMessage: 'Delete entry{isLocalized, select, true { (all locales)} other {}}'
        }, {
            isLocalized
        }),
        icon: /*#__PURE__*/ jsx(Trash, {}),
        dialog: {
            type: 'dialog',
            title: formatMessage({
                id: 'app.components.ConfirmDialog.title',
                defaultMessage: 'Confirmation'
            }),
            content: /*#__PURE__*/ jsxs(Flex, {
                direction: "column",
                gap: 2,
                children: [
                    /*#__PURE__*/ jsx(WarningCircle, {
                        width: "24px",
                        height: "24px",
                        fill: "danger600"
                    }),
                    /*#__PURE__*/ jsx(Typography, {
                        tag: "p",
                        variant: "omega",
                        textAlign: "center",
                        children: formatMessage({
                            id: 'content-manager.actions.delete.dialog.body',
                            defaultMessage: 'Are you sure?'
                        })
                    })
                ]
            }),
            loading: isLoading,
            onConfirm: async ()=>{
                /**
         * If we have a match, we're in the list view
         * and therefore not in a form and shouldn't be
         * trying to set the submitting value.
         */ if (!listViewPathMatch) {
                    setSubmitting(true);
                }
                try {
                    if (!documentId && collectionType !== SINGLE_TYPES) {
                        console.error("You're trying to delete a document without an id, this is likely a bug with Strapi. Please open an issue.");
                        toggleNotification({
                            message: formatMessage({
                                id: 'content-manager.actions.delete.error',
                                defaultMessage: 'An error occurred while trying to delete the document.'
                            }),
                            type: 'danger'
                        });
                        return;
                    }
                    const res = await deleteAction({
                        documentId,
                        model,
                        collectionType,
                        params: {
                            locale: '*'
                        }
                    });
                    if (!('error' in res)) {
                        navigate({
                            pathname: `../${collectionType}/${model}`
                        }, {
                            replace: true
                        });
                    }
                } finally{
                    if (!listViewPathMatch) {
                        setSubmitting(false);
                    }
                }
            }
        },
        variant: 'danger',
        position: [
            'header',
            'table-row'
        ]
    };
};
DeleteAction.type = 'delete';
DeleteAction.position = [
    'header',
    'table-row'
];
const DEFAULT_HEADER_ACTIONS = [
    EditTheModelAction,
    ConfigureTheViewAction,
    DeleteAction
];

export { DEFAULT_HEADER_ACTIONS, Header };
//# sourceMappingURL=Header.mjs.map
