'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var designSystem = require('@strapi/design-system');
var Icons = require('@strapi/icons');
var reactIntl = require('react-intl');
var reactRouterDom = require('react-router-dom');
var RelativeTime = require('../../../components/RelativeTime.js');
var attributes = require('../../../constants/attributes.js');
var collections = require('../../../constants/collections.js');
var DocumentRBAC = require('../../../features/DocumentRBAC.js');
var useDocument = require('../../../hooks/useDocument.js');
var useDocumentActions = require('../../../hooks/useDocumentActions.js');
var router = require('../../../router.js');
var users = require('../../../utils/users.js');
var DocumentActions = require('./DocumentActions.js');
var DocumentStatus = require('./DocumentStatus.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

const Header = ({ isCreating, status, title: documentTitle = 'Untitled' })=>{
    const { formatMessage } = reactIntl.useIntl();
    const isCloning = reactRouterDom.useMatch(router.CLONE_PATH) !== null;
    const params = reactRouterDom.useParams();
    const title = isCreating ? formatMessage({
        id: 'content-manager.containers.edit.title.new',
        defaultMessage: 'Create an entry'
    }) : documentTitle;
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
        direction: "column",
        alignItems: "flex-start",
        paddingTop: {
            initial: 4,
            large: 6
        },
        paddingBottom: {
            initial: 0,
            large: 4
        },
        gap: 2,
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.BackButton, {
                fallback: params.collectionType === collections.SINGLE_TYPES ? undefined : `../${collections.COLLECTION_TYPES}/${params.slug}`
            }),
            /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                width: "100%",
                justifyContent: "space-between",
                gap: {
                    initial: 2,
                    medium: '8rem'
                },
                alignItems: "flex-start",
                direction: {
                    initial: 'column',
                    medium: 'row'
                },
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        variant: "alpha",
                        tag: "h1",
                        children: title
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(HeaderToolbar, {})
                ]
            }),
            status ? /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                marginTop: 1,
                children: /*#__PURE__*/ jsxRuntime.jsx(DocumentStatus.DocumentStatus, {
                    status: isCloning ? 'draft' : status
                })
            }) : null
        ]
    });
};
/**
 * @description Contains the document actions that have `position: header`, if there are
 * none we still render the menu because we render the information about the document there.
 */ const HeaderToolbar = ()=>{
    const { formatMessage } = reactIntl.useIntl();
    const isCloning = reactRouterDom.useMatch(router.CLONE_PATH) !== null;
    const [{ query: { status = 'draft' } }] = strapiAdmin.useQueryParams();
    const { model, id, document, meta, collectionType } = useDocument.useDoc();
    const plugins = strapiAdmin.useStrapiApp('HeaderToolbar', (state)=>state.plugins);
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
        gap: 2,
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.DescriptionComponentRenderer, {
                props: {
                    activeTab: status,
                    model,
                    documentId: id,
                    document: isCloning ? undefined : document,
                    meta: isCloning ? undefined : meta,
                    collectionType
                },
                descriptions: plugins['content-manager'].apis.getHeaderActions(),
                children: (actions)=>{
                    if (actions.length > 0) {
                        return /*#__PURE__*/ jsxRuntime.jsx(HeaderActions, {
                            actions: actions
                        });
                    } else {
                        return null;
                    }
                }
            }),
            /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.DescriptionComponentRenderer, {
                props: {
                    activeTab: status,
                    model,
                    documentId: id,
                    document: isCloning ? undefined : document,
                    meta: isCloning ? undefined : meta,
                    collectionType
                },
                descriptions: plugins['content-manager'].apis.getDocumentActions('header'),
                children: (actions)=>{
                    const headerActions = actions.filter((action)=>{
                        const positions = Array.isArray(action.position) ? action.position : [
                            action.position
                        ];
                        return positions.includes('header');
                    });
                    return /*#__PURE__*/ jsxRuntime.jsx(DocumentActions.DocumentActionsMenu, {
                        actions: headerActions,
                        label: formatMessage({
                            id: 'content-manager.containers.edit.header.more-actions',
                            defaultMessage: 'More actions'
                        }),
                        children: /*#__PURE__*/ jsxRuntime.jsx(Information, {
                            activeTab: status
                        })
                    });
                }
            })
        ]
    });
};
const Information = ({ activeTab })=>{
    const { formatMessage } = reactIntl.useIntl();
    const { document, meta } = useDocument.useDoc();
    if (!document || !document.id) {
        return null;
    }
    /**
   * Because in the backend separate entries are made for draft and published
   * documents, the creator fields are different for each of them. For example,
   * you could make your draft in January and then publish it for the first time
   * in Feb. This would make the createdAt value for the published entry in Feb
   * but really we want to show the document as a whole. The draft entry will also
   * never have the publishedAt values.
   *
   * So, we decipher which document to show the creator for based on the activeTab.
   */ const createAndUpdateDocument = activeTab === 'draft' ? document : meta?.availableStatus.find((status)=>status.publishedAt === null);
    const publishDocument = activeTab === 'published' ? document : meta?.availableStatus.find((status)=>status.publishedAt !== null);
    const creator = createAndUpdateDocument?.[attributes.CREATED_BY_ATTRIBUTE_NAME] ? users.getDisplayName(createAndUpdateDocument[attributes.CREATED_BY_ATTRIBUTE_NAME]) : null;
    const updator = createAndUpdateDocument?.[attributes.UPDATED_BY_ATTRIBUTE_NAME] ? users.getDisplayName(createAndUpdateDocument[attributes.UPDATED_BY_ATTRIBUTE_NAME]) : null;
    const information = [
        {
            isDisplayed: !!publishDocument?.[attributes.PUBLISHED_AT_ATTRIBUTE_NAME],
            label: formatMessage({
                id: 'content-manager.containers.edit.information.last-published.label',
                defaultMessage: 'Published'
            }),
            value: formatMessage({
                id: 'content-manager.containers.edit.information.last-published.value',
                defaultMessage: `{time}{isAnonymous, select, true {} other { by {author}}}`
            }, {
                time: /*#__PURE__*/ jsxRuntime.jsx(RelativeTime.RelativeTime, {
                    timestamp: new Date(publishDocument?.[attributes.PUBLISHED_AT_ATTRIBUTE_NAME])
                }),
                isAnonymous: !publishDocument?.[attributes.PUBLISHED_BY_ATTRIBUTE_NAME],
                author: publishDocument?.[attributes.PUBLISHED_BY_ATTRIBUTE_NAME] ? users.getDisplayName(publishDocument?.[attributes.PUBLISHED_BY_ATTRIBUTE_NAME]) : null
            })
        },
        {
            isDisplayed: !!createAndUpdateDocument?.[attributes.UPDATED_AT_ATTRIBUTE_NAME],
            label: formatMessage({
                id: 'content-manager.containers.edit.information.last-draft.label',
                defaultMessage: 'Updated'
            }),
            value: formatMessage({
                id: 'content-manager.containers.edit.information.last-draft.value',
                defaultMessage: `{time}{isAnonymous, select, true {} other { by {author}}}`
            }, {
                time: /*#__PURE__*/ jsxRuntime.jsx(RelativeTime.RelativeTime, {
                    timestamp: new Date(createAndUpdateDocument?.[attributes.UPDATED_AT_ATTRIBUTE_NAME])
                }),
                isAnonymous: !updator,
                author: updator
            })
        },
        {
            isDisplayed: !!createAndUpdateDocument?.[attributes.CREATED_AT_ATTRIBUTE_NAME],
            label: formatMessage({
                id: 'content-manager.containers.edit.information.document.label',
                defaultMessage: 'Created'
            }),
            value: formatMessage({
                id: 'content-manager.containers.edit.information.document.value',
                defaultMessage: `{time}{isAnonymous, select, true {} other { by {author}}}`
            }, {
                time: /*#__PURE__*/ jsxRuntime.jsx(RelativeTime.RelativeTime, {
                    timestamp: new Date(createAndUpdateDocument?.[attributes.CREATED_AT_ATTRIBUTE_NAME])
                }),
                isAnonymous: !creator,
                author: creator
            })
        }
    ].filter((info)=>info.isDisplayed);
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Flex, {
        borderWidth: "1px 0 0 0",
        borderStyle: "solid",
        borderColor: "neutral150",
        direction: "column",
        marginTop: 2,
        tag: "dl",
        padding: 5,
        gap: 3,
        alignItems: "flex-start",
        /**
       * The menu content has a padding of 4px, but we want our divider (the border top applied) to
       * be flush with the menu content. So we need to adjust the margin & width to account for the padding.
       */ marginLeft: "-0.4rem",
        marginRight: "-0.4rem",
        width: "calc(100% + 8px)",
        children: information.map((info)=>/*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                gap: 1,
                direction: "column",
                alignItems: "flex-start",
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        tag: "dt",
                        variant: "pi",
                        fontWeight: "bold",
                        children: info.label
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        tag: "dd",
                        variant: "pi",
                        textColor: "neutral600",
                        children: info.value
                    })
                ]
            }, info.label))
    });
};
const HeaderActions = ({ actions })=>{
    const [dialogId, setDialogId] = React__namespace.useState(null);
    const handleClick = (action)=>async (e)=>{
            if (!('options' in action)) {
                const { onClick = ()=>false, dialog, id } = action;
                const muteDialog = await onClick(e);
                if (dialog && !muteDialog) {
                    e.preventDefault();
                    setDialogId(id);
                }
            }
        };
    const handleClose = ()=>{
        setDialogId(null);
    };
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Flex, {
        gap: 1,
        children: actions.map((action)=>{
            if (action.options) {
                return /*#__PURE__*/ jsxRuntime.jsx(designSystem.SingleSelect, {
                    size: "S",
                    // @ts-expect-error – the DS will handle numbers, but we're not allowing the API.
                    onChange: action.onSelect,
                    "aria-label": action.label,
                    ...action,
                    children: action.options.map(({ label, ...option })=>{
                        if (option._render) {
                            return option._render();
                        }
                        return /*#__PURE__*/ jsxRuntime.jsx(designSystem.SingleSelectOption, {
                            ...option,
                            children: label
                        }, option.value);
                    })
                }, action.id);
            } else if (action._status) {
                return /*#__PURE__*/ jsxRuntime.jsx(HeaderActionStatus, {
                    tooltip: action._status?.tooltip,
                    children: action._status.message
                }, action.id);
            } else {
                return /*#__PURE__*/ jsxRuntime.jsxs(React__namespace.Fragment, {
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                            disabled: action.disabled,
                            label: action.label,
                            size: "S",
                            onClick: handleClick(action),
                            children: action.icon
                        }),
                        action.dialog ? /*#__PURE__*/ jsxRuntime.jsx(HeaderActionDialog, {
                            ...action.dialog,
                            isOpen: dialogId === action.id,
                            onClose: handleClose
                        }) : null
                    ]
                }, action.id);
            }
        })
    });
};
const HeaderActionStatus = ({ tooltip, children })=>{
    const [open, setOpen] = React__namespace.useState(false);
    // Debounce the open/close so the user can hover over the popover content before it closes
    const debouncedOpen = strapiAdmin.useDebounce(open, 100);
    const handleMouseEnter = ()=>{
        if (tooltip) {
            setOpen(true);
        }
    };
    const handleMouseLeave = ()=>{
        if (tooltip) {
            setOpen(false);
        }
    };
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Popover.Root, {
        open: debouncedOpen,
        onOpenChange: setOpen,
        children: [
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Popover.Anchor, {
                style: {
                    alignSelf: 'stretch'
                },
                onMouseEnter: handleMouseEnter,
                onMouseLeave: handleMouseLeave,
                "aria-describedby": "document-header-action-status",
                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                    height: "100%",
                    children: children
                })
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Popover.Content, {
                role: "tooltip",
                id: "document-header-action-status",
                side: "bottom",
                align: "center",
                onMouseEnter: handleMouseEnter,
                onMouseLeave: handleMouseLeave,
                children: tooltip
            })
        ]
    });
};
const HeaderActionDialog = ({ onClose, onCancel, title, content: Content, isOpen })=>{
    const handleClose = async ()=>{
        if (onCancel) {
            await onCancel();
        }
        onClose();
    };
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Dialog.Root, {
        open: isOpen,
        onOpenChange: handleClose,
        children: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Dialog.Content, {
            children: [
                /*#__PURE__*/ jsxRuntime.jsx(designSystem.Dialog.Header, {
                    children: title
                }),
                typeof Content === 'function' ? /*#__PURE__*/ jsxRuntime.jsx(Content, {
                    onClose: handleClose
                }) : Content
            ]
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * DocumentActionComponents
 * -----------------------------------------------------------------------------------------------*/ const ConfigureTheViewAction = ({ collectionType, model })=>{
    const navigate = reactRouterDom.useNavigate();
    const { formatMessage } = reactIntl.useIntl();
    const isDesktop = strapiAdmin.useIsDesktop();
    return isDesktop ? {
        label: formatMessage({
            id: 'app.links.configure-view',
            defaultMessage: 'Configure the view'
        }),
        icon: /*#__PURE__*/ jsxRuntime.jsx(Icons.ListPlus, {}),
        onClick: ()=>{
            navigate(`../${collectionType}/${model}/configurations/edit`);
        },
        position: 'header'
    } : null;
};
ConfigureTheViewAction.type = 'configure-the-view';
ConfigureTheViewAction.position = 'header';
const EditTheModelAction = ({ model })=>{
    const navigate = reactRouterDom.useNavigate();
    const { formatMessage } = reactIntl.useIntl();
    const isDesktop = strapiAdmin.useIsDesktop();
    return isDesktop ? {
        label: formatMessage({
            id: 'content-manager.link-to-ctb',
            defaultMessage: 'Edit the model'
        }),
        icon: /*#__PURE__*/ jsxRuntime.jsx(Icons.Pencil, {}),
        onClick: ()=>{
            navigate(`/plugins/content-type-builder/content-types/${model}`);
        },
        position: 'header'
    } : null;
};
EditTheModelAction.type = 'edit-the-model';
EditTheModelAction.position = 'header';
const DeleteAction = ({ documentId, model, collectionType, document })=>{
    const navigate = reactRouterDom.useNavigate();
    const { formatMessage } = reactIntl.useIntl();
    const listViewPathMatch = reactRouterDom.useMatch(router.LIST_PATH);
    const canDelete = DocumentRBAC.useDocumentRBAC('DeleteAction', (state)=>state.canDelete);
    const { delete: deleteAction, isLoading } = useDocumentActions.useDocumentActions();
    const { toggleNotification } = strapiAdmin.useNotification();
    const setSubmitting = strapiAdmin.useForm('DeleteAction', (state)=>state.setSubmitting);
    const isLocalized = document?.locale != null;
    return {
        disabled: !canDelete || !document,
        label: formatMessage({
            id: 'content-manager.actions.delete.label',
            defaultMessage: 'Delete entry{isLocalized, select, true { (all locales)} other {}}'
        }, {
            isLocalized
        }),
        icon: /*#__PURE__*/ jsxRuntime.jsx(Icons.Trash, {}),
        dialog: {
            type: 'dialog',
            title: formatMessage({
                id: 'app.components.ConfirmDialog.title',
                defaultMessage: 'Confirmation'
            }),
            content: /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
                direction: "column",
                gap: 2,
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx(Icons.WarningCircle, {
                        width: "24px",
                        height: "24px",
                        fill: "danger600"
                    }),
                    /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        tag: "p",
                        variant: "omega",
                        textAlign: "center",
                        children: formatMessage({
                            id: 'content-manager.actions.delete.dialog.body',
                            defaultMessage: 'Are you sure?'
                        })
                    })
                ]
            }),
            loading: isLoading,
            onConfirm: async ()=>{
                /**
         * If we have a match, we're in the list view
         * and therefore not in a form and shouldn't be
         * trying to set the submitting value.
         */ if (!listViewPathMatch) {
                    setSubmitting(true);
                }
                try {
                    if (!documentId && collectionType !== collections.SINGLE_TYPES) {
                        console.error("You're trying to delete a document without an id, this is likely a bug with Strapi. Please open an issue.");
                        toggleNotification({
                            message: formatMessage({
                                id: 'content-manager.actions.delete.error',
                                defaultMessage: 'An error occurred while trying to delete the document.'
                            }),
                            type: 'danger'
                        });
                        return;
                    }
                    const res = await deleteAction({
                        documentId,
                        model,
                        collectionType,
                        params: {
                            locale: '*'
                        }
                    });
                    if (!('error' in res)) {
                        navigate({
                            pathname: `../${collectionType}/${model}`
                        }, {
                            replace: true
                        });
                    }
                } finally{
                    if (!listViewPathMatch) {
                        setSubmitting(false);
                    }
                }
            }
        },
        variant: 'danger',
        position: [
            'header',
            'table-row'
        ]
    };
};
DeleteAction.type = 'delete';
DeleteAction.position = [
    'header',
    'table-row'
];
const DEFAULT_HEADER_ACTIONS = [
    EditTheModelAction,
    ConfigureTheViewAction,
    DeleteAction
];

exports.DEFAULT_HEADER_ACTIONS = DEFAULT_HEADER_ACTIONS;
exports.Header = Header;
//# sourceMappingURL=Header.js.map
