import { jsxs, jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { useForm, useField, useNotification, useRBAC, useFocusInputField, useIsDesktop } from '@strapi/admin/strapi-admin';
import { Flex, Box, Link, TextButton, EmptyStateLayout, Field, Combobox, ComboboxOption, Typography, VisuallyHidden, useComposedRefs, IconButton } from '@strapi/design-system';
import { ArrowClockwise, WarningCircle, Plus, Link as Link$1, Drag, Cross } from '@strapi/icons';
import { generateNKeysBetween } from 'fractional-indexing';
import pipe from 'lodash/fp/pipe';
import { getEmptyImage } from 'react-dnd-html5-backend';
import { useIntl } from 'react-intl';
import { FixedSizeList } from 'react-window';
import { styled } from 'styled-components';
import { COLLECTION_TYPES } from '../../../../../constants/collections.mjs';
import { ItemTypes } from '../../../../../constants/dragAndDrop.mjs';
import { PERMISSIONS } from '../../../../../constants/plugin.mjs';
import { DocumentRBAC, useDocumentRBAC } from '../../../../../features/DocumentRBAC.mjs';
import { useDebounce } from '../../../../../hooks/useDebounce.mjs';
import { useDocument } from '../../../../../hooks/useDocument.mjs';
import { useDocumentContext } from '../../../../../hooks/useDocumentContext.mjs';
import { useDragAndDrop, DROP_SENSITIVITY } from '../../../../../hooks/useDragAndDrop.mjs';
import { useGetRelationsQuery, useLazySearchRelationsQuery } from '../../../../../services/relations.mjs';
import { getRelationLabel } from '../../../../../utils/relations.mjs';
import { getTranslation } from '../../../../../utils/translations.mjs';
import { DocumentStatus } from '../../DocumentStatus.mjs';
import { useComponent } from '../ComponentContext.mjs';
import { RelationModalRenderer, getCollectionType } from './RelationModal.mjs';

/**
 * Remove a relation, whether it's been already saved or not.
 * It's used both in RelationsList, where the "remove relation" button is, and in the input,
 * because we sometimes need to remove a previous relation when selecting a new one.
 */ function useHandleDisconnect(fieldName, consumerName) {
    const field = useField(fieldName);
    const removeFieldRow = useForm(consumerName, (state)=>state.removeFieldRow);
    const addFieldRow = useForm(consumerName, (state)=>state.addFieldRow);
    const handleDisconnect = (relation)=>{
        if (field.value && field.value.connect) {
            /**
       * A relation will exist in the `connect` array _if_ it has
       * been added without saving. In this case, we just remove it
       * from the connect array
       */ const indexOfRelationInConnectArray = field.value.connect.findIndex((rel)=>rel.id === relation.id);
            if (indexOfRelationInConnectArray >= 0) {
                removeFieldRow(`${fieldName}.connect`, indexOfRelationInConnectArray);
                return;
            }
        }
        addFieldRow(`${fieldName}.disconnect`, {
            id: relation.id,
            apiData: {
                id: relation.id,
                documentId: relation.documentId,
                locale: relation.locale
            }
        });
    };
    return handleDisconnect;
}
/* -------------------------------------------------------------------------------------------------
 * RelationsField
 * -----------------------------------------------------------------------------------------------*/ const RELATIONS_TO_DISPLAY = 5;
const ONE_WAY_RELATIONS = [
    'oneWay',
    'oneToOne',
    'manyToOne',
    'oneToManyMorph',
    'oneToOneMorph'
];
/**
 * TODO: we get a rather ugly flash when we remove a single relation from the list leaving
 * no other relations when we press save. The initial relation re-renders, probably because
 * of the lag in the Form cleaning it's "disconnect" array, whilst our data has not been invalidated.
 *
 * Could we invalidate relation data on the document actions? Should we?
 */ /**
 * @internal
 * @description The relations field holds a lot of domain logic for handling relations which is rather complicated
 * At present we do not expose this to plugin developers, however, they are able to overwrite it themselves should
 * they wish to do so.
 */ const RelationsField = /*#__PURE__*/ React.forwardRef(({ disabled, label, ...props }, ref)=>{
    const { currentDocument, currentDocumentMeta } = useDocumentContext('RelationsField');
    const [currentPage, setCurrentPage] = React.useState(1);
    // Use the documentId from the actual document, not the params (meta)
    const documentId = currentDocument.document?.documentId;
    const { formatMessage } = useIntl();
    const isMorph = props.attribute.relation.toLowerCase().includes('morph');
    const isDisabled = isMorph || disabled;
    const { componentId, componentUID } = useComponent('RelationsField', ({ uid, id })=>({
            componentId: id,
            componentUID: uid
        }));
    const isSubmitting = useForm('RelationsList', (state)=>state.isSubmitting);
    React.useEffect(()=>{
        setCurrentPage(1);
    }, [
        isSubmitting
    ]);
    const component = componentUID && currentDocument.components[componentUID];
    /**
     * We'll always have a documentId in a created entry, so we look for a componentId first.
     * Same with `uid` and `documentModel`.
     * The componentId is empty when adding a new component in a repeatable. Let it be null to skip isRelatedToCurrentDocument
     */ const model = component ? component.uid : currentDocumentMeta.model;
    const id = component ? componentId?.toString() : documentId;
    /**
     * The `name` prop is a complete path to the field, e.g. `field1.field2.field3`.
     * Where the above example would a nested field within two components, however
     * we only require the field on the component not the complete path since we query
     * individual components. Therefore we split the string and take the last item.
     */ const [targetField] = props.name.split('.').slice(-1);
    const schemaAttributes = component ? component.attributes ?? {} : currentDocument.schema?.attributes ?? {};
    /**
     * Confirm the target field is related to the current document.
     * Since relations can exist in a modal on top of the root document,
     * we need to ensure we are fetching relations for the correct document (root document vs related document),
     */ const isRelatedToCurrentDocument = Object.values(schemaAttributes).filter((attribute)=>attribute.type === 'relation' && 'target' in attribute && 'target' in props.attribute && attribute.target === props.attribute.target).length > 0;
    const { data, isLoading, isFetching } = useGetRelationsQuery({
        model,
        targetField,
        // below we don't run the query if there is no id.
        id,
        params: {
            ...currentDocumentMeta.params,
            pageSize: RELATIONS_TO_DISPLAY,
            page: currentPage
        }
    }, {
        refetchOnMountOrArgChange: true,
        skip: !id || !isRelatedToCurrentDocument,
        selectFromResult: (result)=>{
            return {
                ...result,
                data: {
                    ...result.data,
                    results: result.data?.results ? result.data.results : []
                }
            };
        }
    });
    const handleLoadMore = ()=>{
        setCurrentPage((prev)=>prev + 1);
    };
    const field = useField(props.name);
    const isFetchingMoreRelations = isLoading || isFetching;
    const realServerRelationsCount = 'pagination' in data && data.pagination ? data.pagination.total : 0;
    /**
     * Items that are already connected, but reordered would be in
     * this list, so to get an accurate figure, we remove them.
     */ const relationsConnected = (field.value?.connect ?? []).filter((rel)=>data.results.findIndex((relation)=>relation.id === rel.id) === -1).length ?? 0;
    const relationsDisconnected = field.value?.disconnect?.length ?? 0;
    const relationsCount = realServerRelationsCount + relationsConnected - relationsDisconnected;
    /**
     * This is it, the source of truth for reordering in conjunction with partial loading & updating
     * of relations. Relations on load are given __temp_key__ when fetched, because we don't want to
     * create brand new keys everytime the data updates, just keep adding them onto the newly loaded ones.
     */ const relations = React.useMemo(()=>{
        const ctx = {
            field: field.value,
            // @ts-expect-error – targetModel does exist on the attribute. But it's not typed.
            href: `../${COLLECTION_TYPES}/${props.attribute.targetModel}`,
            mainField: props.mainField
        };
        /**
       * Tidy up our data.
       */ const transformations = pipe(removeConnected(ctx), removeDisconnected(ctx), addLabelAndHref(ctx));
        const transformedRels = transformations([
            ...data.results
        ]);
        /**
       * THIS IS CRUCIAL. If you don't sort by the __temp_key__ which comes from fractional indexing
       * then the list will be in the wrong order.
       */ return [
            ...transformedRels,
            ...field.value?.connect ?? []
        ].sort((a, b)=>{
            if (a.__temp_key__ < b.__temp_key__) return -1;
            if (a.__temp_key__ > b.__temp_key__) return 1;
            return 0;
        });
    }, [
        data.results,
        field.value,
        // @ts-expect-error – targetModel does exist on the attribute. But it's not typed.
        props.attribute.targetModel,
        props.mainField
    ]);
    const handleDisconnect = useHandleDisconnect(props.name, 'RelationsField');
    const handleConnect = (relation)=>{
        const [lastItemInList] = relations.slice(-1);
        const item = {
            id: relation.id,
            apiData: {
                id: relation.id,
                documentId: relation.documentId,
                locale: relation.locale,
                isTemporary: true
            },
            status: relation.status,
            /**
         * If there's a last item, that's the first key we use to generate out next one.
         */ __temp_key__: generateNKeysBetween(lastItemInList?.__temp_key__ ?? null, null, 1)[0],
            // Fallback to `id` if there is no `mainField` value, which will overwrite the above `id` property with the exact same data.
            [props.mainField?.name ?? 'documentId']: relation[props.mainField?.name ?? 'documentId'],
            label: getRelationLabel(relation, props.mainField),
            // @ts-expect-error – targetModel does exist on the attribute, but it's not typed.
            href: `../${COLLECTION_TYPES}/${props.attribute.targetModel}/${relation.documentId}?${relation.locale ? `plugins[i18n][locale]=${relation.locale}` : ''}`
        };
        if (ONE_WAY_RELATIONS.includes(props.attribute.relation)) {
            // Remove any existing relation so they can be replaced with the new one
            field.value?.connect?.forEach(handleDisconnect);
            relations.forEach(handleDisconnect);
            field.onChange(`${props.name}.connect`, [
                item
            ]);
        } else {
            field.onChange(`${props.name}.connect`, [
                ...field.value?.connect ?? [],
                item
            ]);
        }
    };
    return /*#__PURE__*/ jsxs(Flex, {
        ref: ref,
        direction: "column",
        gap: 3,
        justifyContent: "space-between",
        alignItems: "stretch",
        wrap: "wrap",
        children: [
            /*#__PURE__*/ jsxs(StyledFlex, {
                direction: "column",
                alignItems: "start",
                gap: 2,
                width: "100%",
                children: [
                    /*#__PURE__*/ jsx(RelationsInput, {
                        disabled: isDisabled,
                        // NOTE: we should not default to using the documentId if the component is being created (componentUID is undefined)
                        id: componentUID && component ? componentId ? `${componentId}` : '' : documentId,
                        label: `${label} ${relationsCount > 0 ? `(${relationsCount})` : ''}`,
                        model: model,
                        onChange: handleConnect,
                        isRelatedToCurrentDocument: isRelatedToCurrentDocument,
                        ...props
                    }),
                    'pagination' in data && data.pagination && data.pagination.pageCount > data.pagination.page ? /*#__PURE__*/ jsx(TextButton, {
                        disabled: isFetchingMoreRelations,
                        onClick: handleLoadMore,
                        loading: isFetchingMoreRelations,
                        startIcon: /*#__PURE__*/ jsx(ArrowClockwise, {}),
                        // prevent the label from line-wrapping
                        shrink: 0,
                        children: formatMessage({
                            id: getTranslation('relation.loadMore'),
                            defaultMessage: 'Load More'
                        })
                    }) : null
                ]
            }),
            /*#__PURE__*/ jsx(RelationsList, {
                data: relations,
                serverData: data.results,
                disabled: isDisabled,
                name: props.name,
                isLoading: isFetchingMoreRelations,
                relationType: props.attribute.relation,
                // @ts-expect-error – targetModel does exist on the attribute. But it's not typed.
                targetModel: props.attribute.targetModel,
                mainField: props.mainField
            })
        ]
    });
});
/**
 * TODO: this can be removed once we stop shipping Inputs with
 * labels wrapped round in DS@2.
 */ const StyledFlex = styled(Flex)`
  & > div {
    width: 100%;
  }
`;
/**
 * If it's in the connected array, it can get out of our data array,
 * we'll be putting it back in later and sorting it anyway.
 */ const removeConnected = ({ field })=>(relations)=>{
        return relations.filter((relation)=>{
            const connectedRelations = field?.connect ?? [];
            return connectedRelations.findIndex((rel)=>rel.id === relation.id) === -1;
        });
    };
/**
 * @description Removes relations that are in the `disconnect` array of the field
 */ const removeDisconnected = ({ field })=>(relations)=>relations.filter((relation)=>{
            const disconnectedRelations = field?.disconnect ?? [];
            return disconnectedRelations.findIndex((rel)=>rel.id === relation.id) === -1;
        });
/**
 * @description Adds a label and href to the relation object we use this to render
 * a better UI where we can link to the relation and display a human-readable label.
 */ const addLabelAndHref = ({ mainField, href })=>(relations)=>relations.map((relation)=>{
            return {
                ...relation,
                // Fallback to `id` if there is no `mainField` value, which will overwrite the above `documentId` property with the exact same data.
                [mainField?.name ?? 'documentId']: relation[mainField?.name ?? 'documentId'],
                label: getRelationLabel(relation, mainField),
                href: `${href}/${relation.documentId}?${relation.locale ? `plugins[i18n][locale]=${relation.locale}` : ''}`
            };
        });
/**
 * @description Contains all the logic for the combobox that can search
 * for relations and then add them to the field's connect array.
 */ const RelationsInput = ({ hint, id, model, label, labelAction, name, mainField, placeholder, required, unique: _unique, 'aria-label': _ariaLabel, onChange, isRelatedToCurrentDocument, ...props })=>{
    const [searchParams, setSearchParams] = React.useState({
        _q: '',
        page: 1
    });
    const { toggleNotification } = useNotification();
    const { currentDocumentMeta } = useDocumentContext('RelationsInput');
    const { formatMessage } = useIntl();
    const field = useField(name);
    const searchParamsDebounced = useDebounce(searchParams, 300);
    const [searchForTrigger, { data, isLoading }] = useLazySearchRelationsQuery();
    /**
   * Because we're using a lazy query, we need to trigger the search
   * when the component mounts and when the search params change.
   * We also need to trigger the search when the field value changes
   * so that we can filter out the relations that are already connected.
   */ React.useEffect(()=>{
        /**
     * The `name` prop is a complete path to the field, e.g. `field1.field2.field3`.
     * Where the above example would a nested field within two components, however
     * we only require the field on the component not the complete path since we query
     * individual components. Therefore we split the string and take the last item.
     */ const [targetField] = name.split('.').slice(-1);
        // Return early if there is no relation to the document
        if (!isRelatedToCurrentDocument) return;
        searchForTrigger({
            model,
            targetField,
            params: {
                ...currentDocumentMeta.params,
                id: id ?? '',
                pageSize: 10,
                idsToInclude: field.value?.disconnect?.map((rel)=>rel.id.toString()) ?? [],
                idsToOmit: field.value?.connect?.map((rel)=>rel.id.toString()) ?? [],
                ...searchParamsDebounced
            }
        });
    }, [
        field.value?.connect,
        field.value?.disconnect,
        id,
        model,
        name,
        searchForTrigger,
        searchParamsDebounced,
        isRelatedToCurrentDocument,
        currentDocumentMeta.params
    ]);
    const hasNextPage = data?.pagination ? data.pagination.page < data.pagination.pageCount : false;
    const options = data?.results ?? [];
    const handleChange = (relationId)=>{
        if (!relationId) {
            return;
        }
        const relation = options.find((opt)=>opt.id.toString() === relationId);
        if (!relation) {
            // This is very unlikely to happen, but it ensures we don't have any data for.
            console.error("You've tried to add a relation with an id that does not exist in the options you can see, this is likely a bug with Strapi. Please open an issue.");
            toggleNotification({
                message: formatMessage({
                    id: getTranslation('relation.error-adding-relation'),
                    defaultMessage: 'An error occurred while trying to add the relation.'
                }),
                type: 'danger'
            });
            return;
        }
        /**
     * You need to give this relation a correct _temp_key_ but
     * this component doesn't know about those ones, you can't rely
     * on the connect array because that doesn't hold items that haven't
     * moved. So use a callback to fill in the gaps when connecting.
     *
     */ onChange(relation);
    };
    const relation = {
        collectionType: COLLECTION_TYPES,
        // @ts-expect-error – targetModel does exist on the attribute. But it's not typed.
        model: props.attribute.targetModel,
        documentId: '',
        params: currentDocumentMeta.params
    };
    const { permissions = [], isLoading: isLoadingPermissions, error } = useRBAC(PERMISSIONS.map((action)=>({
            action,
            subject: relation.model
        })));
    if (error) {
        return /*#__PURE__*/ jsx(Flex, {
            alignItems: "center",
            height: "100%",
            justifyContent: "center",
            children: /*#__PURE__*/ jsx(EmptyStateLayout, {
                icon: /*#__PURE__*/ jsx(WarningCircle, {
                    width: "16rem"
                }),
                content: formatMessage({
                    id: 'anErrorOccurred',
                    defaultMessage: 'Whoops! Something went wrong. Please, try again.'
                })
            })
        });
    }
    return /*#__PURE__*/ jsxs(Field.Root, {
        error: field.error,
        hint: hint,
        name: name,
        required: required,
        children: [
            /*#__PURE__*/ jsx(Field.Label, {
                action: labelAction,
                children: label
            }),
            /*#__PURE__*/ jsx(DocumentRBAC, {
                permissions: permissions,
                model: relation.model,
                children: /*#__PURE__*/ jsx(RelationModalWithContext, {
                    relation: relation,
                    name: name,
                    placeholder: placeholder,
                    hasNextPage: hasNextPage,
                    isLoadingPermissions: isLoadingPermissions,
                    isLoadingSearchRelations: isLoading,
                    handleChange: handleChange,
                    setSearchParams: setSearchParams,
                    data: data,
                    mainField: mainField,
                    fieldValue: field.value,
                    ...props
                })
            }),
            /*#__PURE__*/ jsx(Field.Error, {}),
            /*#__PURE__*/ jsx(Field.Hint, {})
        ]
    });
};
const RelationModalWithContext = ({ relation, name, placeholder, hasNextPage, isLoadingSearchRelations, isLoadingPermissions, handleChange, mainField, setSearchParams, fieldValue, data, ...props })=>{
    const [textValue, setTextValue] = React.useState('');
    const { formatMessage } = useIntl();
    const canCreate = useDocumentRBAC('RelationModalWrapper', (state)=>state.canCreate);
    const fieldRef = useFocusInputField(name);
    const { componentUID } = useComponent('RelationsField', ({ uid })=>({
            componentUID: uid
        }));
    const handleLoadMore = ()=>{
        if (!data || !data.pagination) {
            return;
        } else if (data.pagination.page < data.pagination.pageCount) {
            setSearchParams((s)=>({
                    ...s,
                    page: s.page + 1
                }));
        }
    };
    const options = data?.results ?? [];
    React.useLayoutEffect(()=>{
        setTextValue('');
    }, [
        fieldValue
    ]);
    const handleSearch = async (search)=>{
        setSearchParams((s)=>({
                ...s,
                _q: search,
                page: 1
            }));
    };
    return /*#__PURE__*/ jsx(RelationModalRenderer, {
        children: ({ dispatch })=>/*#__PURE__*/ jsx(Combobox, {
                ref: fieldRef,
                creatable: "visible",
                creatableDisabled: !canCreate,
                createMessage: ()=>formatMessage({
                        id: getTranslation('relation.create'),
                        defaultMessage: 'Create a relation'
                    }),
                onCreateOption: ()=>{
                    if (canCreate) {
                        dispatch({
                            type: 'GO_TO_RELATION',
                            payload: {
                                document: relation,
                                shouldBypassConfirmation: false,
                                fieldToConnect: name,
                                fieldToConnectUID: componentUID
                            }
                        });
                    }
                },
                creatableStartIcon: /*#__PURE__*/ jsx(Plus, {
                    fill: "neutral500"
                }),
                name: name,
                autocomplete: {
                    type: 'list',
                    filter: 'contains'
                },
                placeholder: placeholder || formatMessage({
                    id: getTranslation('relation.add'),
                    defaultMessage: 'Add relation'
                }),
                hasMoreItems: hasNextPage,
                loading: isLoadingSearchRelations || isLoadingPermissions,
                onOpenChange: ()=>{
                    handleSearch(textValue ?? '');
                },
                noOptionsMessage: ()=>formatMessage({
                        id: getTranslation('relation.notAvailable'),
                        defaultMessage: 'No relations available'
                    }),
                loadingMessage: formatMessage({
                    id: getTranslation('relation.isLoading'),
                    defaultMessage: 'Relations are loading'
                }),
                onLoadMore: handleLoadMore,
                textValue: textValue,
                onChange: handleChange,
                onTextValueChange: (text)=>{
                    setTextValue(text);
                },
                onInputChange: (event)=>{
                    handleSearch(event.currentTarget.value);
                },
                ...props,
                children: options?.map((opt)=>{
                    const textValue = getRelationLabel(opt, mainField);
                    return /*#__PURE__*/ jsx(ComboboxOption, {
                        value: opt.id.toString(),
                        textValue: textValue,
                        children: /*#__PURE__*/ jsxs(Flex, {
                            gap: 2,
                            justifyContent: "space-between",
                            children: [
                                /*#__PURE__*/ jsxs(Flex, {
                                    gap: 2,
                                    children: [
                                        /*#__PURE__*/ jsx(Link$1, {
                                            fill: "neutral500"
                                        }),
                                        /*#__PURE__*/ jsx(Typography, {
                                            ellipsis: true,
                                            children: textValue
                                        })
                                    ]
                                }),
                                opt.status ? /*#__PURE__*/ jsx(DocumentStatus, {
                                    status: opt.status
                                }) : null
                            ]
                        })
                    }, opt.id);
                })
            })
    });
};
/* -------------------------------------------------------------------------------------------------
 * RelationsList
 * -----------------------------------------------------------------------------------------------*/ const RELATION_ITEM_HEIGHT = 50;
const RELATION_GUTTER = 4;
const RelationsList = ({ data, serverData, disabled, name, isLoading, relationType, targetModel, mainField })=>{
    const ariaDescriptionId = React.useId();
    const { formatMessage } = useIntl();
    const listRef = React.useRef(null);
    const outerListRef = React.useRef(null);
    const [overflow, setOverflow] = React.useState();
    const [liveText, setLiveText] = React.useState('');
    const field = useField(name);
    React.useEffect(()=>{
        if (data.length <= RELATIONS_TO_DISPLAY) {
            return setOverflow(undefined);
        }
        const handleNativeScroll = (e)=>{
            const el = e.target;
            const parentScrollContainerHeight = el.parentNode.scrollHeight;
            const maxScrollBottom = el.scrollHeight - el.scrollTop;
            if (el.scrollTop === 0) {
                return setOverflow('bottom');
            }
            if (maxScrollBottom === parentScrollContainerHeight) {
                return setOverflow('top');
            }
            return setOverflow('top-bottom');
        };
        const outerListRefCurrent = outerListRef?.current;
        if (!isLoading && data.length > 0 && outerListRefCurrent) {
            outerListRef.current.addEventListener('scroll', handleNativeScroll);
        }
        return ()=>{
            if (outerListRefCurrent) {
                outerListRefCurrent.removeEventListener('scroll', handleNativeScroll);
            }
        };
    }, [
        isLoading,
        data.length
    ]);
    const getItemPos = (index)=>`${index + 1} of ${data.length}`;
    const handleMoveItem = (newIndex, oldIndex)=>{
        const item = data[oldIndex];
        setLiveText(formatMessage({
            id: getTranslation('dnd.reorder'),
            defaultMessage: '{item}, moved. New position in list: {position}.'
        }, {
            item: item.label ?? item.documentId,
            position: getItemPos(newIndex)
        }));
        /**
     * Splicing mutates the array, so we need to create a new array
     */ const newData = [
            ...data
        ];
        const currentRow = data[oldIndex];
        const startKey = oldIndex > newIndex ? newData[newIndex - 1]?.__temp_key__ : newData[newIndex]?.__temp_key__;
        const endKey = oldIndex > newIndex ? newData[newIndex]?.__temp_key__ : newData[newIndex + 1]?.__temp_key__;
        /**
     * We're moving the relation between two other relations, so
     * we need to generate a new key that keeps the order
     */ const [newKey] = generateNKeysBetween(startKey, endKey, 1);
        newData.splice(oldIndex, 1);
        newData.splice(newIndex, 0, {
            ...currentRow,
            __temp_key__: newKey
        });
        /**
     * Now we diff against the server to understand what's different so we
     * can keep the connect array nice and tidy. It also needs reversing because
     * we reverse the relations from the server in the first place.
     */ const connectedRelations = newData.reduce((acc, relation, currentIndex, array)=>{
            const relationOnServer = serverData.find((oldRelation)=>oldRelation.id === relation.id);
            const relationInFront = array[currentIndex + 1];
            if (!relationOnServer || relationOnServer.__temp_key__ !== relation.__temp_key__) {
                const position = relationInFront ? {
                    before: relationInFront.documentId,
                    locale: relationInFront.locale,
                    status: 'publishedAt' in relationInFront && relationInFront.publishedAt ? 'published' : 'draft'
                } : {
                    end: true
                };
                const relationWithPosition = {
                    ...relation,
                    ...{
                        apiData: {
                            id: relation.id,
                            documentId: relation.documentId ?? relation.apiData?.documentId ?? '',
                            locale: relation.locale || relation.apiData?.locale,
                            isTemporary: relation.apiData?.isTemporary,
                            position
                        }
                    }
                };
                return [
                    ...acc,
                    relationWithPosition
                ];
            }
            return acc;
        }, []).toReversed();
        field.onChange(`${name}.connect`, connectedRelations);
    };
    const handleGrabItem = (index)=>{
        const item = data[index];
        setLiveText(formatMessage({
            id: getTranslation('dnd.grab-item'),
            defaultMessage: `{item}, grabbed. Current position in list: {position}. Press up and down arrow to change position, Spacebar to drop, Escape to cancel.`
        }, {
            item: item.label ?? item.documentId,
            position: getItemPos(index)
        }));
    };
    const handleDropItem = (index)=>{
        const { href: _href, label, ...item } = data[index];
        setLiveText(formatMessage({
            id: getTranslation('dnd.drop-item'),
            defaultMessage: `{item}, dropped. Final position in list: {position}.`
        }, {
            item: label ?? item.documentId,
            position: getItemPos(index)
        }));
    };
    const handleCancel = (index)=>{
        const item = data[index];
        setLiveText(formatMessage({
            id: getTranslation('dnd.cancel-item'),
            defaultMessage: '{item}, dropped. Re-order cancelled.'
        }, {
            item: item.label ?? item.documentId
        }));
    };
    const handleDisconnect = useHandleDisconnect(name, 'RelationsList');
    /**
   * These relation types will only ever have one item
   * in their list, so you can't reorder a single item!
   */ const canReorder = !ONE_WAY_RELATIONS.includes(relationType);
    const dynamicListHeight = data.length > RELATIONS_TO_DISPLAY ? Math.min(data.length, RELATIONS_TO_DISPLAY) * (RELATION_ITEM_HEIGHT + RELATION_GUTTER) + RELATION_ITEM_HEIGHT / 2 : Math.min(data.length, RELATIONS_TO_DISPLAY) * (RELATION_ITEM_HEIGHT + RELATION_GUTTER);
    return /*#__PURE__*/ jsxs(ShadowBox, {
        $overflowDirection: overflow,
        children: [
            /*#__PURE__*/ jsx(VisuallyHidden, {
                id: ariaDescriptionId,
                children: formatMessage({
                    id: getTranslation('dnd.instructions'),
                    defaultMessage: `Press spacebar to grab and re-order`
                })
            }),
            /*#__PURE__*/ jsx(VisuallyHidden, {
                "aria-live": "assertive",
                children: liveText
            }),
            /*#__PURE__*/ jsx(FixedSizeList, {
                height: dynamicListHeight,
                ref: listRef,
                outerRef: outerListRef,
                itemCount: data.length,
                itemSize: RELATION_ITEM_HEIGHT + RELATION_GUTTER,
                itemData: {
                    ariaDescribedBy: ariaDescriptionId,
                    canDrag: canReorder,
                    disabled,
                    handleCancel,
                    handleDropItem,
                    handleGrabItem,
                    handleMoveItem,
                    name,
                    handleDisconnect,
                    relations: data,
                    targetModel,
                    mainField
                },
                itemKey: (index)=>data[index].id,
                innerElementType: "ol",
                children: ListItem
            })
        ]
    });
};
const ShadowBox = styled(Box)`
  position: relative;
  overflow: hidden;
  flex: 1;

  &:before,
  &:after {
    position: absolute;
    width: 100%;
    height: 4px;
    z-index: 1;
  }

  &:before {
    /* TODO: as for DS Table component we would need this to be handled by the DS theme */
    content: '';
    background: linear-gradient(rgba(3, 3, 5, 0.2) 0%, rgba(0, 0, 0, 0) 100%);
    top: 0;
    opacity: ${({ $overflowDirection })=>$overflowDirection === 'top-bottom' || $overflowDirection === 'top' ? 1 : 0};
    transition: opacity 0.2s ease-in-out;
  }

  &:after {
    /* TODO: as for DS Table component we would need this to be handled by the DS theme */
    content: '';
    background: linear-gradient(0deg, rgba(3, 3, 5, 0.2) 0%, rgba(0, 0, 0, 0) 100%);
    bottom: 0;
    opacity: ${({ $overflowDirection })=>$overflowDirection === 'top-bottom' || $overflowDirection === 'bottom' ? 1 : 0};
    transition: opacity 0.2s ease-in-out;
  }
`;
const ListItem = ({ data, index, style })=>{
    const { ariaDescribedBy, canDrag = false, disabled = false, handleCancel, handleDisconnect, handleDropItem, handleGrabItem, handleMoveItem, name, relations, targetModel, mainField } = data;
    const isDesktop = useIsDesktop();
    const { currentDocumentMeta } = useDocumentContext('RelationsField');
    const { formatMessage } = useIntl();
    const { href, id, label: originalLabel, status: originalStatus, documentId, apiData, locale } = relations[index];
    /**
   * The code above attempts to retrieve the updated value of a relation that has not yet been saved.
   * This is necessary when a relation modal is opened, and the mainField or its status is updated.
   * These changes need to be reflected in the initial relation field.
   */ const collectionType = getCollectionType(href);
    const isTemporary = apiData?.isTemporary ?? false;
    const { document } = useDocument({
        collectionType,
        model: targetModel,
        documentId: documentId ?? apiData?.documentId,
        params: currentDocumentMeta.params
    }, {
        skip: !isTemporary
    });
    const label = isTemporary && document ? getRelationLabel(document, mainField) : originalLabel;
    const status = isTemporary && document ? document?.status : originalStatus;
    const [{ handlerId, isDragging, handleKeyDown }, relationRef, dropRef, dragRef, dragPreviewRef] = useDragAndDrop(canDrag && !disabled, {
        type: `${ItemTypes.RELATION}_${name}`,
        index,
        item: {
            displayedValue: label,
            status,
            id: id,
            index
        },
        onMoveItem: handleMoveItem,
        onDropItem: handleDropItem,
        onGrabItem: handleGrabItem,
        onCancel: handleCancel,
        dropSensitivity: DROP_SENSITIVITY.REGULAR
    });
    const composedRefs = useComposedRefs(relationRef, dragRef);
    React.useEffect(()=>{
        dragPreviewRef(getEmptyImage());
    }, [
        dragPreviewRef
    ]);
    const safeDocumentId = documentId ?? apiData?.documentId;
    const safeLocale = locale ?? apiData?.locale ?? null;
    const documentMeta = React.useMemo(()=>({
            documentId: safeDocumentId,
            model: targetModel,
            collectionType: getCollectionType(href),
            params: {
                locale: safeLocale
            }
        }), [
        safeDocumentId,
        href,
        safeLocale,
        targetModel
    ]);
    return /*#__PURE__*/ jsx(Box, {
        style: style,
        tag: "li",
        ref: dropRef,
        "aria-describedby": ariaDescribedBy,
        cursor: canDrag ? 'all-scroll' : 'default',
        children: isDragging ? /*#__PURE__*/ jsx(RelationItemPlaceholder, {}) : /*#__PURE__*/ jsxs(Flex, {
            paddingTop: 2,
            paddingBottom: 2,
            paddingLeft: canDrag ? 2 : 4,
            paddingRight: 4,
            hasRadius: true,
            borderColor: "neutral200",
            background: disabled ? 'neutral150' : 'neutral0',
            justifyContent: "space-between",
            ref: composedRefs,
            "data-handler-id": handlerId,
            children: [
                /*#__PURE__*/ jsxs(FlexWrapper, {
                    gap: 1,
                    children: [
                        canDrag && isDesktop ? /*#__PURE__*/ jsx(IconButton, {
                            tag: "div",
                            role: "button",
                            tabIndex: 0,
                            withTooltip: false,
                            label: formatMessage({
                                id: getTranslation('components.RelationInput.icon-button-aria-label'),
                                defaultMessage: 'Drag'
                            }),
                            variant: "ghost",
                            onKeyDown: handleKeyDown,
                            disabled: disabled,
                            children: /*#__PURE__*/ jsx(Drag, {})
                        }) : null,
                        /*#__PURE__*/ jsxs(Flex, {
                            width: "100%",
                            minWidth: 0,
                            gap: 4,
                            justifyContent: "space-between",
                            children: [
                                /*#__PURE__*/ jsx(Box, {
                                    flex: 1,
                                    minWidth: 0,
                                    paddingTop: 1,
                                    paddingBottom: 1,
                                    children: /*#__PURE__*/ jsx(RelationModalRenderer, {
                                        relation: documentMeta,
                                        children: label
                                    })
                                }),
                                status ? /*#__PURE__*/ jsx(DocumentStatus, {
                                    status: status
                                }) : null
                            ]
                        })
                    ]
                }),
                /*#__PURE__*/ jsx(Box, {
                    paddingLeft: 4,
                    children: /*#__PURE__*/ jsx(IconButton, {
                        onClick: ()=>handleDisconnect(relations[index]),
                        disabled: disabled,
                        label: formatMessage({
                            id: getTranslation('relation.disconnect'),
                            defaultMessage: 'Remove'
                        }),
                        variant: "ghost",
                        size: "S",
                        children: /*#__PURE__*/ jsx(Cross, {})
                    })
                })
            ]
        })
    });
};
const FlexWrapper = styled(Flex)`
  width: 100%;
  /* Used to prevent endAction to be pushed out of container */
  min-width: 0;

  & > div[role='button'] {
    cursor: all-scroll;
  }
`;
const DisconnectButton = styled.button`
  svg path {
    fill: ${({ theme, disabled })=>disabled ? theme.colors.neutral600 : theme.colors.neutral500};
  }

  &:hover svg path,
  &:focus svg path {
    fill: ${({ theme, disabled })=>!disabled && theme.colors.neutral600};
  }
`;
const LinkEllipsis = styled(Link)`
  display: block;

  & > span {
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    display: block;
  }
`;
const RelationItemPlaceholder = ()=>/*#__PURE__*/ jsx(Box, {
        paddingTop: 2,
        paddingBottom: 2,
        paddingLeft: 4,
        paddingRight: 4,
        hasRadius: true,
        borderStyle: "dashed",
        borderColor: "primary600",
        borderWidth: "1px",
        background: "primary100",
        height: `calc(100% - ${RELATION_GUTTER}px)`
    });
const MemoizedRelationsField = /*#__PURE__*/ React.memo(RelationsField);

export { DisconnectButton, FlexWrapper, LinkEllipsis, MemoizedRelationsField as RelationsInput };
//# sourceMappingURL=Relations.mjs.map
