import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { useForm, createRulesEngine, useIsDesktop, useField } from '@strapi/admin/strapi-admin';
import { Box, useComposedRefs, IconButton, Menu, Flex, Accordion, Grid } from '@strapi/design-system';
import { Trash, Drag, More } from '@strapi/icons';
import { getEmptyImage } from 'react-dnd-html5-backend';
import { useIntl } from 'react-intl';
import { styled } from 'styled-components';
import { COMPONENT_ICONS } from '../../../../../components/ComponentIcon.mjs';
import { ItemTypes } from '../../../../../constants/dragAndDrop.mjs';
import { useDocumentContext } from '../../../../../hooks/useDocumentContext.mjs';
import { useDocumentLayout } from '../../../../../hooks/useDocumentLayout.mjs';
import { useDragAndDrop } from '../../../../../hooks/useDragAndDrop.mjs';
import { getIn } from '../../../../../utils/objects.mjs';
import { getTranslation } from '../../../../../utils/translations.mjs';
import { ResponsiveGridRoot, ResponsiveGridItem } from '../../FormLayout.mjs';
import { InputRenderer as MemoizedInputRenderer } from '../../InputRenderer.mjs';

const DynamicComponent = ({ componentUid, disabled, index, name, onRemoveComponentClick, onMoveComponent, onGrabItem, onDropItem, onCancel, dynamicComponentsByCategory = {}, onAddComponent, children })=>{
    const { formatMessage } = useIntl();
    const formValues = useForm('DynamicComponent', (state)=>state.values);
    const { currentDocument, currentDocumentMeta } = useDocumentContext('DynamicComponent');
    const rulesEngine = createRulesEngine();
    const isDesktop = useIsDesktop();
    const { edit: { components } } = useDocumentLayout(currentDocumentMeta.model);
    const title = React.useMemo(()=>{
        const { mainField } = components[componentUid]?.settings ?? {
            mainField: 'id'
        };
        const mainFieldValue = getIn(formValues, `${name}.${index}.${mainField}`);
        const displayedValue = mainField === 'id' || !mainFieldValue ? '' : String(mainFieldValue).trim();
        const mainValue = displayedValue.length > 0 ? `- ${displayedValue}` : displayedValue;
        return mainValue;
    }, [
        componentUid,
        components,
        formValues,
        name,
        index
    ]);
    const { icon, displayName } = React.useMemo(()=>{
        const [category] = componentUid.split('.');
        const { icon, displayName } = (dynamicComponentsByCategory[category] ?? []).find((component)=>component.uid === componentUid) ?? {
            icon: null,
            displayName: null
        };
        return {
            icon,
            displayName
        };
    }, [
        componentUid,
        dynamicComponentsByCategory
    ]);
    const [{ handlerId, isDragging, handleKeyDown }, boxRef, dropRef, dragRef, dragPreviewRef] = useDragAndDrop(!disabled, {
        type: `${ItemTypes.DYNAMIC_ZONE}_${name}`,
        index,
        item: {
            index,
            displayedValue: `${displayName} ${title}`,
            icon
        },
        onMoveItem: onMoveComponent,
        onDropItem,
        onGrabItem,
        onCancel
    });
    React.useEffect(()=>{
        dragPreviewRef(getEmptyImage(), {
            captureDraggingState: false
        });
    }, [
        dragPreviewRef,
        index
    ]);
    /**
   * We don't need the accordion's to communicate with each other,
   * so a unique value for their state is enough.
   */ const accordionValue = React.useId();
    const { value = [], rawError } = useField(`${name}.${index}`);
    const [collapseToOpen, setCollapseToOpen] = React.useState('');
    React.useEffect(()=>{
        if (rawError && value) {
            setCollapseToOpen(accordionValue);
        }
    }, [
        rawError,
        value,
        accordionValue
    ]);
    const composedBoxRefs = useComposedRefs(boxRef, dropRef);
    const accordionActions = disabled ? null : /*#__PURE__*/ jsxs(Fragment, {
        children: [
            /*#__PURE__*/ jsx(IconButton, {
                variant: "ghost",
                label: formatMessage({
                    id: getTranslation('components.DynamicZone.delete-label'),
                    defaultMessage: 'Delete {name}'
                }, {
                    name: title
                }),
                onClick: onRemoveComponentClick,
                children: /*#__PURE__*/ jsx(Trash, {})
            }),
            isDesktop && /*#__PURE__*/ jsx(IconButton, {
                variant: "ghost",
                onClick: (e)=>e.stopPropagation(),
                "data-handler-id": handlerId,
                ref: dragRef,
                label: formatMessage({
                    id: getTranslation('components.DragHandle-label'),
                    defaultMessage: 'Drag'
                }),
                onKeyDown: handleKeyDown,
                children: /*#__PURE__*/ jsx(Drag, {})
            }),
            /*#__PURE__*/ jsxs(Menu.Root, {
                children: [
                    /*#__PURE__*/ jsx(Menu.Trigger, {
                        size: "S",
                        endIcon: null,
                        paddingLeft: 0,
                        paddingRight: 0,
                        children: /*#__PURE__*/ jsx(IconButton, {
                            variant: "ghost",
                            label: formatMessage({
                                id: getTranslation('components.DynamicZone.more-actions'),
                                defaultMessage: 'More actions'
                            }),
                            tag: "span",
                            children: /*#__PURE__*/ jsx(More, {
                                "aria-hidden": true,
                                focusable: false
                            })
                        })
                    }),
                    /*#__PURE__*/ jsxs(Menu.Content, {
                        children: [
                            /*#__PURE__*/ jsxs(Menu.SubRoot, {
                                children: [
                                    /*#__PURE__*/ jsx(Menu.SubTrigger, {
                                        children: formatMessage({
                                            id: getTranslation('components.DynamicZone.add-item-above'),
                                            defaultMessage: 'Add component above'
                                        })
                                    }),
                                    /*#__PURE__*/ jsx(Menu.SubContent, {
                                        children: Object.entries(dynamicComponentsByCategory).map(([category, components])=>/*#__PURE__*/ jsxs(React.Fragment, {
                                                children: [
                                                    /*#__PURE__*/ jsx(Menu.Label, {
                                                        children: category
                                                    }),
                                                    components.map(({ displayName, uid })=>/*#__PURE__*/ jsx(Menu.Item, {
                                                            onSelect: ()=>onAddComponent(uid, index),
                                                            children: displayName
                                                        }, uid))
                                                ]
                                            }, category))
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsxs(Menu.SubRoot, {
                                children: [
                                    /*#__PURE__*/ jsx(Menu.SubTrigger, {
                                        children: formatMessage({
                                            id: getTranslation('components.DynamicZone.add-item-below'),
                                            defaultMessage: 'Add component below'
                                        })
                                    }),
                                    /*#__PURE__*/ jsx(Menu.SubContent, {
                                        children: Object.entries(dynamicComponentsByCategory).map(([category, components])=>/*#__PURE__*/ jsxs(React.Fragment, {
                                                children: [
                                                    /*#__PURE__*/ jsx(Menu.Label, {
                                                        children: category
                                                    }),
                                                    components.map(({ displayName, uid })=>/*#__PURE__*/ jsx(Menu.Item, {
                                                            onSelect: ()=>onAddComponent(uid, index + 1),
                                                            children: displayName
                                                        }, uid))
                                                ]
                                            }, category))
                                    })
                                ]
                            })
                        ]
                    })
                ]
            })
        ]
    });
    const accordionTitle = title ? `${displayName} ${title}` : displayName;
    return /*#__PURE__*/ jsxs(ComponentContainer, {
        tag: "li",
        width: "100%",
        children: [
            /*#__PURE__*/ jsx(Flex, {
                justifyContent: "center",
                children: /*#__PURE__*/ jsx(Rectangle, {
                    background: "neutral200"
                })
            }),
            /*#__PURE__*/ jsx(StyledBox, {
                ref: composedBoxRefs,
                hasRadius: true,
                children: isDragging ? /*#__PURE__*/ jsx(Preview, {}) : /*#__PURE__*/ jsx(Accordion.Root, {
                    value: collapseToOpen,
                    onValueChange: setCollapseToOpen,
                    children: /*#__PURE__*/ jsxs(Accordion.Item, {
                        value: accordionValue,
                        children: [
                            /*#__PURE__*/ jsxs(Accordion.Header, {
                                children: [
                                    /*#__PURE__*/ jsx(Accordion.Trigger, {
                                        icon: icon && COMPONENT_ICONS[icon] ? COMPONENT_ICONS[icon] : COMPONENT_ICONS.dashboard,
                                        children: accordionTitle
                                    }),
                                    /*#__PURE__*/ jsx(Accordion.Actions, {
                                        children: accordionActions
                                    })
                                ]
                            }),
                            /*#__PURE__*/ jsx(Accordion.Content, {
                                children: /*#__PURE__*/ jsx(AccordionContentRadius, {
                                    background: "neutral0",
                                    children: /*#__PURE__*/ jsx(Box, {
                                        paddingLeft: 6,
                                        paddingRight: 6,
                                        paddingTop: 6,
                                        paddingBottom: 6,
                                        children: /*#__PURE__*/ jsx(Grid.Root, {
                                            gap: 4,
                                            children: components[componentUid]?.layout?.map((row, rowInd)=>{
                                                const visibleFields = row.filter(({ ...field })=>{
                                                    const condition = field.attribute.conditions?.visible;
                                                    if (condition) {
                                                        return rulesEngine.evaluate(condition, value);
                                                    }
                                                    return true;
                                                });
                                                if (visibleFields.length === 0) {
                                                    return null; // Skip rendering the entire grid row
                                                }
                                                return /*#__PURE__*/ jsx(Grid.Item, {
                                                    col: 12,
                                                    xs: 12,
                                                    direction: "column",
                                                    alignItems: "stretch",
                                                    children: /*#__PURE__*/ jsx(ResponsiveGridRoot, {
                                                        gap: 4,
                                                        children: visibleFields.map(({ size, ...field })=>{
                                                            const fieldName = `${name}.${index}.${field.name}`;
                                                            const fieldWithTranslatedLabel = {
                                                                ...field,
                                                                label: formatMessage({
                                                                    id: `content-manager.components.${componentUid}.${field.name}`,
                                                                    defaultMessage: field.label
                                                                })
                                                            };
                                                            return /*#__PURE__*/ jsx(ResponsiveGridItem, {
                                                                col: size,
                                                                s: 12,
                                                                xs: 12,
                                                                direction: "column",
                                                                alignItems: "stretch",
                                                                children: children ? children({
                                                                    ...fieldWithTranslatedLabel,
                                                                    document: currentDocument,
                                                                    name: fieldName
                                                                }) : /*#__PURE__*/ jsx(MemoizedInputRenderer, {
                                                                    ...fieldWithTranslatedLabel,
                                                                    document: currentDocument,
                                                                    name: fieldName
                                                                })
                                                            }, fieldName);
                                                        })
                                                    })
                                                }, rowInd);
                                            })
                                        })
                                    })
                                })
                            })
                        ]
                    })
                })
            })
        ]
    });
};
// TODO: Delete once https://github.com/strapi/design-system/pull/858
// is merged and released.
const StyledBox = styled(Box)`
  > div:first-child {
    box-shadow: ${({ theme })=>theme.shadows.tableShadow};
  }
`;
const AccordionContentRadius = styled(Box)`
  border-radius: 0 0 ${({ theme })=>theme.spaces[1]} ${({ theme })=>theme.spaces[1]};
`;
const Rectangle = styled(Box)`
  width: ${({ theme })=>theme.spaces[2]};
  height: ${({ theme })=>theme.spaces[4]};
`;
const Preview = styled.span`
  display: block;
  background-color: ${({ theme })=>theme.colors.primary100};
  outline: 1px dashed ${({ theme })=>theme.colors.primary500};
  outline-offset: -1px;
  padding: ${({ theme })=>theme.spaces[6]};
`;
const ComponentContainer = styled(Box)`
  list-style: none;
  padding: 0;
  margin: 0;
`;

export { DynamicComponent };
//# sourceMappingURL=DynamicComponent.mjs.map
