import { jsx } from 'react/jsx-runtime';
import 'react';
import { Typography } from '@strapi/design-system';
import { Bold, Italic, Underline, StrikeThrough, Code } from '@strapi/icons';
import { Editor, Transforms } from 'slate';
import { css, styled } from 'styled-components';

const stylesToInherit = css`
  font-size: inherit;
  color: inherit;
  line-height: inherit;
`;
const BoldText = styled(Typography).attrs({
    fontWeight: 'bold'
})`
  ${stylesToInherit}
`;
const ItalicText = styled(Typography)`
  font-style: italic;
  ${stylesToInherit}
`;
const UnderlineText = styled(Typography).attrs({
    textDecoration: 'underline'
})`
  ${stylesToInherit}
`;
const StrikeThroughText = styled(Typography).attrs({
    textDecoration: 'line-through'
})`
  ${stylesToInherit}
`;
const InlineCode = styled.code`
  background-color: ${({ theme })=>theme.colors.neutral150};
  border-radius: ${({ theme })=>theme.borderRadius};
  padding: ${({ theme })=>`0 ${theme.spaces[2]}`};
  font-family: 'SF Mono', SFMono-Regular, ui-monospace, 'DejaVu Sans Mono', Menlo, Consolas,
    monospace;
  color: inherit;
`;
/**
 * The default handler for checking if a modifier is active
 */ const baseCheckIsActive = (editor, name)=>{
    const marks = Editor.marks(editor);
    if (!marks) return false;
    return Boolean(marks[name]);
};
/**
 * The default handler for toggling a modifier
 */ const baseHandleToggle = (editor, name)=>{
    const marks = Editor.marks(editor);
    // If there is no selection, set selection to the end of line
    if (!editor.selection) {
        const endOfEditor = Editor.end(editor, []);
        Transforms.select(editor, endOfEditor);
    }
    // Toggle the modifier
    if (marks?.[name]) {
        Editor.removeMark(editor, name);
    } else {
        Editor.addMark(editor, name, true);
    }
};
const modifiers = {
    bold: {
        icon: Bold,
        isValidEventKey: (event)=>event.key === 'b',
        label: {
            id: 'components.Blocks.modifiers.bold',
            defaultMessage: 'Bold'
        },
        checkIsActive: (editor)=>baseCheckIsActive(editor, 'bold'),
        handleToggle: (editor)=>baseHandleToggle(editor, 'bold'),
        renderLeaf: (children)=>/*#__PURE__*/ jsx(BoldText, {
                children: children
            })
    },
    italic: {
        icon: Italic,
        isValidEventKey: (event)=>event.key === 'i',
        label: {
            id: 'components.Blocks.modifiers.italic',
            defaultMessage: 'Italic'
        },
        checkIsActive: (editor)=>baseCheckIsActive(editor, 'italic'),
        handleToggle: (editor)=>baseHandleToggle(editor, 'italic'),
        renderLeaf: (children)=>/*#__PURE__*/ jsx(ItalicText, {
                children: children
            })
    },
    underline: {
        icon: Underline,
        isValidEventKey: (event)=>event.key === 'u',
        label: {
            id: 'components.Blocks.modifiers.underline',
            defaultMessage: 'Underline'
        },
        checkIsActive: (editor)=>baseCheckIsActive(editor, 'underline'),
        handleToggle: (editor)=>baseHandleToggle(editor, 'underline'),
        renderLeaf: (children)=>/*#__PURE__*/ jsx(UnderlineText, {
                children: children
            })
    },
    strikethrough: {
        icon: StrikeThrough,
        isValidEventKey: (event)=>event.key === 'S' && event.shiftKey,
        label: {
            id: 'components.Blocks.modifiers.strikethrough',
            defaultMessage: 'Strikethrough'
        },
        checkIsActive: (editor)=>baseCheckIsActive(editor, 'strikethrough'),
        handleToggle: (editor)=>baseHandleToggle(editor, 'strikethrough'),
        renderLeaf: (children)=>/*#__PURE__*/ jsx(StrikeThroughText, {
                children: children
            })
    },
    code: {
        icon: Code,
        isValidEventKey: (event)=>event.key === 'e',
        label: {
            id: 'components.Blocks.modifiers.code',
            defaultMessage: 'Inline code'
        },
        checkIsActive: (editor)=>baseCheckIsActive(editor, 'code'),
        handleToggle: (editor)=>baseHandleToggle(editor, 'code'),
        renderLeaf: (children)=>/*#__PURE__*/ jsx(InlineCode, {
                children: children
            })
    }
};

export { modifiers };
//# sourceMappingURL=Modifiers.mjs.map
