import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { Box, SingleSelect, SingleSelectOption } from '@strapi/design-system';
import { CodeBlock as CodeBlock$1 } from '@strapi/icons';
import * as Prism from 'prismjs';
import { useIntl } from 'react-intl';
import { Element, Node, Transforms, Editor } from 'slate';
import { useFocused, useSelected, ReactEditor } from 'slate-react';
import { styled } from 'styled-components';
import { useBlocksEditorContext } from '../BlocksEditor.mjs';
import { codeLanguages } from '../utils/constants.mjs';
import { baseHandleConvert } from '../utils/conversions.mjs';
import { pressEnterTwiceToExit } from '../utils/enterKey.mjs';
import 'prismjs/themes/prism-solarizedlight.css';
import 'prismjs/components/prism-asmatmel';
import 'prismjs/components/prism-bash';
import 'prismjs/components/prism-basic';
import 'prismjs/components/prism-c';
import 'prismjs/components/prism-clojure';
import 'prismjs/components/prism-cobol';
import 'prismjs/components/prism-cpp';
import 'prismjs/components/prism-csharp';
import 'prismjs/components/prism-dart';
import 'prismjs/components/prism-docker';
import 'prismjs/components/prism-elixir';
import 'prismjs/components/prism-erlang';
import 'prismjs/components/prism-fortran';
import 'prismjs/components/prism-fsharp';
import 'prismjs/components/prism-go';
import 'prismjs/components/prism-graphql';
import 'prismjs/components/prism-groovy';
import 'prismjs/components/prism-haskell';
import 'prismjs/components/prism-haxe';
import 'prismjs/components/prism-ini';
import 'prismjs/components/prism-java';
import 'prismjs/components/prism-javascript';
import 'prismjs/components/prism-jsx';
import 'prismjs/components/prism-json';
import 'prismjs/components/prism-julia';
import 'prismjs/components/prism-kotlin';
import 'prismjs/components/prism-latex';
import 'prismjs/components/prism-lua';
import 'prismjs/components/prism-markdown';
import 'prismjs/components/prism-matlab';
import 'prismjs/components/prism-makefile';
import 'prismjs/components/prism-objectivec';
import 'prismjs/components/prism-perl';
import 'prismjs/components/prism-php';
import 'prismjs/components/prism-powershell';
import 'prismjs/components/prism-python';
import 'prismjs/components/prism-r';
import 'prismjs/components/prism-ruby';
import 'prismjs/components/prism-rust';
import 'prismjs/components/prism-sas';
import 'prismjs/components/prism-scala';
import 'prismjs/components/prism-scheme';
import 'prismjs/components/prism-sql';
import 'prismjs/components/prism-stata';
import 'prismjs/components/prism-swift';
import 'prismjs/components/prism-typescript';
import 'prismjs/components/prism-tsx';
import 'prismjs/components/prism-vbnet';
import 'prismjs/components/prism-yaml';

const decorateCode = ([node, path])=>{
    const ranges = [];
    // make sure it is an Slate Element
    if (!Element.isElement(node) || node.type !== 'code') return ranges;
    // transform the Element into a string
    const text = Node.string(node);
    const language = codeLanguages.find((lang)=>lang.value === node.language);
    const decorateKey = language?.decorate ?? language?.value;
    const selectedLanguage = Prism.languages[decorateKey || 'plaintext'];
    // create "tokens" with "prismjs" and put them in "ranges"
    const tokens = Prism.tokenize(text, selectedLanguage);
    let start = 0;
    for (const token of tokens){
        const length = token.length;
        const end = start + length;
        if (typeof token !== 'string') {
            ranges.push({
                anchor: {
                    path,
                    offset: start
                },
                focus: {
                    path,
                    offset: end
                },
                className: `token ${token.type}`
            });
        }
        start = end;
    }
    // these will be found in "renderLeaf" in "leaf" and their "className" will be applied
    return ranges;
};
const CodeBlock = styled.pre`
  border-radius: ${({ theme })=>theme.borderRadius};
  background-color: ${({ theme })=>theme.colors.neutral100};
  max-width: 100%;
  overflow: auto;
  padding: ${({ theme })=>`${theme.spaces[3]} ${theme.spaces[4]}`};
  flex-shrink: 1;

  & > code {
    font-family: 'SF Mono', SFMono-Regular, ui-monospace, 'DejaVu Sans Mono', Menlo, Consolas,
      monospace;
    color: ${({ theme })=>theme.colors.neutral800};
    overflow: auto;
    max-width: 100%;
  }
`;
const CodeEditor = (props)=>{
    const { editor } = useBlocksEditorContext('ImageDialog');
    const editorIsFocused = useFocused();
    const imageIsSelected = useSelected();
    const { formatMessage } = useIntl();
    const [isSelectOpen, setIsSelectOpen] = React.useState(false);
    const shouldDisplayLanguageSelect = editorIsFocused && imageIsSelected || isSelectOpen;
    return /*#__PURE__*/ jsxs(Box, {
        position: "relative",
        width: "100%",
        children: [
            /*#__PURE__*/ jsx(CodeBlock, {
                ...props.attributes,
                children: /*#__PURE__*/ jsx("code", {
                    children: props.children
                })
            }),
            shouldDisplayLanguageSelect && /*#__PURE__*/ jsx(Box, {
                position: "absolute",
                background: "neutral0",
                borderColor: "neutral150",
                borderStyle: "solid",
                borderWidth: "0.5px",
                shadow: "tableShadow",
                top: "100%",
                marginTop: 1,
                right: 0,
                padding: 1,
                hasRadius: true,
                children: /*#__PURE__*/ jsx(SingleSelect, {
                    onChange: (open)=>{
                        Transforms.setNodes(editor, {
                            language: open.toString()
                        }, {
                            match: (node)=>!Editor.isEditor(node) && node.type === 'code'
                        });
                    },
                    value: props.element.type === 'code' && props.element.language || 'plaintext',
                    onOpenChange: (open)=>{
                        setIsSelectOpen(open);
                        // Focus the editor again when closing the select so the user can continue typing
                        if (!open) {
                            ReactEditor.focus(editor);
                        }
                    },
                    onCloseAutoFocus: (e)=>e.preventDefault(),
                    "aria-label": formatMessage({
                        id: 'components.Blocks.blocks.code.languageLabel',
                        defaultMessage: 'Select a language'
                    }),
                    children: codeLanguages.map(({ value, label })=>/*#__PURE__*/ jsx(SingleSelectOption, {
                            value: value,
                            children: label
                        }, value))
                })
            })
        ]
    });
};
const codeBlocks = {
    code: {
        renderElement: (props)=>/*#__PURE__*/ jsx(CodeEditor, {
                ...props
            }),
        icon: CodeBlock$1,
        label: {
            id: 'components.Blocks.blocks.code',
            defaultMessage: 'Code block'
        },
        matchNode: (node)=>node.type === 'code',
        isInBlocksSelector: true,
        handleConvert (editor) {
            baseHandleConvert(editor, {
                type: 'code',
                language: 'plaintext'
            });
        },
        handleEnterKey (editor) {
            pressEnterTwiceToExit(editor);
        },
        snippets: [
            '```'
        ]
    }
};

export { codeBlocks, decorateCode };
//# sourceMappingURL=Code.mjs.map
